#include <memory.h>
#include <xboxkrnl/xboxkrnl.h>
#include <hal/audio.h>
#include <hal/io.h>
#include <openxdk/debug.h>

// The foundation for this file came from the Cromwell audio driver by 
// Andy (see his comments below).

// Andy@warmcat.com 2003-03-10:
//
// Xbox PC audio is an AC97 comptible audio controller in the MCPX chip
// http://www.amd.com/us-en/assets/content_type/white_papers_and_tech_docs/24467.pdf
// unlike standard AC97 all the regs appear as MMIO from 0xfec00000
// +0 - +7f = Mixer/Codec regs  <-- Wolfson Micro chip
// +100 - +17f = Busmaster regs

// The Wolfson Micro 9709 Codec fitted to the Xbox is a "dumb" codec that
//  has NO PC-controllable registers.  The only regs it has are the
//  manufacturer and device ID ones.

// S/PDIF comes out of the MCPX device and is controlled by an extra set
//  of busmaster DMA registers at +0x170.  These need their own descriptor
//  separate from the PCM Out, although that descriptor can share audio
//  buffers with PCM out successfully.

// global reference to the ac97 device
AC97_DEVICE ac97Device;

// the IDT consists of 256 of these guys. They
// hold the information that tells the CPU where
// to jump to when an interrupt occurs
typedef struct 
{
	unsigned short lowOffset;
	unsigned short selector;
	unsigned short type;
	unsigned short highOffset;
} idtDescriptor;

// fetches the current address of the IDT table
// so that we can hook the audio interrupt later
unsigned long getIDTAddress()
{
	unsigned char idtr[6];
	unsigned long idt;
	__asm__ volatile ("sidt %0": "=m" (idtr));
	idt = *((unsigned long *)&idtr[2]);
	return idt;
}

// declare our asm interrupt handler (see audioInterrupt.s)
extern void IntHandler6(void);

void XDumpAudioStatus()
{
	volatile AC97_DEVICE *pac97device = &ac97Device;
	if (pac97device)
	{
		volatile unsigned char *pb = (unsigned char *)pac97device->mmio;
		//debugPrint("CIV=%02x LVI=%02x SR=%04x CR=%02x\n", pb[0x114], pb[0x115], pb[0x116], pb[0x11B]);
	}
}

// Initialises the audio subsystem.  This *must* be done before
// audio will work.  You can pass NULL as the callback, but if you
// do that, it is your responsibility to keep feeding the data to
// XAudioProvideSamples() manually.
//
// note that I currently ignore sampleSizeInBits and numChannels.  They
// are provided to cope with future enhancements. Currently supported samples
// are 16 bit, 2 channels (stereo)
void XAudioInit(int sampleSizeInBits, int numChannels, XAudioCallback callback, void *data)
{
	volatile AC97_DEVICE * pac97device = &ac97Device;
	
	pac97device->mmio = (unsigned int *)0xfec00000;
	pac97device->nextDescriptorMod31 = 0;
	pac97device->callback = callback;
	pac97device->callbackData = data;
	pac97device->sampleSizeInBits = sampleSizeInBits;
	pac97device->numChannels = numChannels;

	// initialise descriptors to all 0x00 (no samples)        
	memset((void *)&pac97device->pcmSpdifDescriptor[0], 0, sizeof(pac97device->pcmSpdifDescriptor));
	memset((void *)&pac97device->pcmOutDescriptor[0], 0, sizeof(pac97device->pcmOutDescriptor));

	// perform a cold reset
	pac97device->mmio[0x12C>>2] |= 2;
	
	// wait until the chip is finished resetting...
	while(!pac97device->mmio[0x130>>2]&0x100)
		;

	// clear all interrupts
	((unsigned char *)pac97device->mmio)[0x116] = 0xFF;
	((unsigned char *)pac97device->mmio)[0x176] = 0xFF;

	// tell the audio chip where it should look for the descriptors
	unsigned int pcmAddress = (unsigned int)&pac97device->pcmOutDescriptor[0];
	unsigned int spdifAddress = (unsigned int)&pac97device->pcmSpdifDescriptor[0];
	pac97device->mmio[0x100>>2] = 0;  // no PCM input
	pac97device->mmio[0x110>>2] = MmGetPhysicalAddress((void *)pcmAddress);
	pac97device->mmio[0x170>>2] = MmGetPhysicalAddress((void *)spdifAddress);

	// default to being silent...
	XAudioPause(pac97device);

	// now we need to hook our interrupt handler.  We do this even
	// if the user hasn't registered a callback.
	idtDescriptor *idt = (idtDescriptor *)getIDTAddress();
	unsigned short selector;
	asm volatile("movw %%cs,%0" :"=g"(selector));
	
	// our interrupt handler is a pointer to our asm routine
	unsigned int interruptRoutine = (unsigned int)IntHandler6;
	// found this interrupt number through trial and error.  ick!
	int intNum = 0x36;
	idt[intNum].selector   = selector;
	idt[intNum].type       = 0x8e00; 
	idt[intNum].lowOffset  = (unsigned short)interruptRoutine;
	idt[intNum].highOffset = (unsigned short)(((unsigned int)interruptRoutine)>>16);
	
	// and lastly, lets enable the interrupt on the PIC
	IoOutputByte(0x21,(IoInputByte(0x21) & 0xBF));
}

// tell the chip it is OK to play...
void XAudioPlay()
{
	volatile AC97_DEVICE *pac97device = &ac97Device;
	pac97device->mmio[0x118>>2] = (unsigned int)0x1d000000; // PCM out - run, allow interrupts
	pac97device->mmio[0x178>>2] = (unsigned int)0x1d000000; // PCM out - run, allow interrupts
}

// tell the chip it is paused.
void XAudioPause()
{
	volatile AC97_DEVICE *pac97device = &ac97Device;
	pac97device->mmio[0x118>>2] = (unsigned int)0x1c000000; // PCM out - PAUSE, allow interrupts
	pac97device->mmio[0x178>>2] = (unsigned int)0x1c000000; // PCM out - PAUSE, allow interrupts
}

// This is the function you should call when you want to give the
// audio chip some more data.  If you have registered a callback, it
// should call this method.  If you are providing the samples manually,
// you need to make sure you call this function often enough so the
// chip doesn't run out of data
void XAudioProvideSamples(unsigned char *buffer, unsigned short bufferLength, int isFinal)
{
	volatile AC97_DEVICE *pac97device = &ac97Device;
	volatile unsigned char *pb = (unsigned char *)pac97device->mmio;

	unsigned short bufferControl = 0x8000;
	if (isFinal) 
		bufferControl |= 0x4000;

	unsigned int address = (unsigned int)buffer;

	pac97device->pcmOutDescriptor[pac97device->nextDescriptorMod31].bufferStartAddress    = MmGetPhysicalAddress((PVOID)address);
	pac97device->pcmOutDescriptor[pac97device->nextDescriptorMod31].bufferLengthInSamples = bufferLength / (pac97device->sampleSizeInBits / 8);
	pac97device->pcmOutDescriptor[pac97device->nextDescriptorMod31].bufferControl         = bufferControl;
	pb[0x115] = (unsigned char)pac97device->nextDescriptorMod31; // set last active descriptor

	pac97device->pcmSpdifDescriptor[pac97device->nextDescriptorMod31].bufferStartAddress    = MmGetPhysicalAddress((PVOID)address);
	pac97device->pcmSpdifDescriptor[pac97device->nextDescriptorMod31].bufferLengthInSamples = bufferLength / (pac97device->sampleSizeInBits / 8);
	pac97device->pcmSpdifDescriptor[pac97device->nextDescriptorMod31].bufferControl         = bufferControl;
	pb[0x175] = (unsigned char)pac97device->nextDescriptorMod31; // set last active descriptor

	// increment to the next buffer descriptor (rolling around to 0 once you get to 31)
	pac97device->nextDescriptorMod31 = (pac97device->nextDescriptorMod31 +1 ) & 0x1f;
}

// although we have to explicitly clear the S/PDIF interrupt sources, in fact
// the way we are set up PCM and S/PDIF are in lockstep and we only listen for
// PCM actions, since S/PDIF is always spooling through the same buffer.
void XAudioInterrupt()
{
	volatile AC97_DEVICE *pac97device = &ac97Device;
	if (pac97device)
	{
		volatile unsigned char *pb = (unsigned char *)pac97device->mmio;

		// was the interrupt triggered because we were out of data?
		if (pb[0x116]&8) 
		{
			if (pac97device->callback)
				(pac97device->callback)((void *)pac97device, pac97device->callbackData);
		}
	
		// was the interrupt triggered because of FIFO error
		if (pb[0x116]&0x10) 
		{
			// Fifo underrun - what should I do here?
		}

		pb[0x116] = 0xFF; // clear all int sources
		pb[0x176] = 0XFF; // clear all int sources
	}
}
