/// <summary>
/// C# FTP client library.
///  
/// Copyright (C) 2000-2003  Enterprise Distributed Technologies Ltd
/// 
/// www.enterprisedt.com
/// 
/// This library is free software; you can redistribute it and/or
/// modify it under the terms of the GNU Lesser General Public
/// License as published by the Free Software Foundation; either
/// version 2.1 of the License, or (at your option) any later version.
/// 
/// This library is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
/// Lesser General Public License for more details.
/// 
/// You should have received a copy of the GNU Lesser General Public
/// License along with this library; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// Bug fixes, suggestions and comments should be sent to ftp@enterprisedt.com
/// 
/// Change Log:
/// 
/// $Log: FTPUnixFilePaser.cs,v $
/// Revision 1.1  2006/10/25 14:45:33  kayaweed
/// * src/FTP/FTPClient.cs, src/FTP/FTPFile.cs, src/FTP/FTPFileParser.cs, src/FTP/FTPUnixFilePaser.cs:
/// 	- Add listing Save Manager
///
/// * src/Core/FtpToXbox.cs, src/Core/Savegame.cs:
/// 	- Add listing Save Manager
///
/// Revision 1.1  2005/04/14 19:59:51  kayaweed
/// - Initial Commit
///
/// Revision 1.1  2005/03/20 19:24:53  kayaweed
/// *** empty log message ***
///
/// Revision 1.3  2004/07/12 20:46:48  bruceb
/// bugfix re writeline
///
/// Revision 1.2  2003/05/17 13:54:49  bruceb
/// 1.0 release - support for .NET 1.0, tidied comments
///
/// Revision 1.1  2003/05/17 12:33:13  bruceb
/// first release
///
/// Add by KaYa backport Parser features from edtFTPnet 1.1.6 
///
/// </summary>

using System;
using System.Text;
using System.Globalization;
using System.Collections;
  
namespace EnterpriseDT.Net.Ftp
{
	public class FileParser
	{		
		/// <summary> Symbolic link symbol</summary>
		private const string SYMLINK_ARROW = "->";
		
		/// <summary> Indicates symbolic link</summary>
		private const char SYMLINK_CHAR = 'l';
		
		/// <summary> Indicates ordinary file</summary>
		private const char ORDINARY_FILE_CHAR = '-';
		
		/// <summary> Indicates directory</summary>
		private const char DIRECTORY_CHAR = 'd';
		
		/// <summary>Date format 1</summary>
		private const string format1a = "MMM'-'d'-'yyyy";
        
        /// <summary>Date format 1</summary>
		private const string format1b = "MMM'-'dd'-'yyyy";
        
        /// <summary>array of format 1 formats</summary> 
        private string[] format1 = {format1a,format1b};

		/// <summary>Date format 2</summary>
		private const string format2a = "MMM'-'d'-'yyyy'-'HH':'mm";
		
        /// <summary>Date format 2</summary>
		private const string format2b = "MMM'-'dd'-'yyyy'-'HH':'mm";	
        
        /// <summary>Date format 2</summary>
        private const string format2c = "MMM'-'d'-'yyyy'-'H':'mm";         
        
        /// <summary>Date format 2</summary>
        private const string format2d = "MMM'-'dd'-'yyyy'-'H':'mm"; 

        /// <summary>array of format 2 formats</summary>
        private string[] format2 = {format2a,format2b,format2c,format2d};
        
        /// <summary> Minimum number of expected fields</summary>
		private const int MIN_FIELD_COUNT = 8;
        

		/// <summary> 
		/// Parse server supplied string, e.g.:
		/// 
		/// lrwxrwxrwx   1 wuftpd   wuftpd         14 Jul 22  2002 MIRRORS -> README-MIRRORS
		/// -rw-r--r--   1 b173771  users         431 Mar 31 20:04 .htaccess
		/// 
		/// </summary>
		/// <param name="raw">  raw string to parse
		/// </param>
		public FTPFile Parse(string raw)
		{		
			// test it is a valid line, e.g. "total 342522" is invalid
			char ch = raw[0];
			if (ch != ORDINARY_FILE_CHAR && ch != DIRECTORY_CHAR && ch != SYMLINK_CHAR)
				return null;
			
			string[] fields = Split(raw);
			
			if (fields.Length < MIN_FIELD_COUNT)
			{
				StringBuilder listing = new StringBuilder("Unexpected number of fields in listing '");
				listing.Append(raw).Append("' - expected minimum ").Append(MIN_FIELD_COUNT).
                        Append(" fields but found ").Append(fields.Length).Append(" fields");
				throw new FormatException(listing.ToString());
			}
			
			// field pos
			int index = 0;
			
			// first field is perms
			string permissions = fields[index++];
			ch = permissions[0];
			bool isDir = false;
			bool isLink = false;
			if (ch == DIRECTORY_CHAR)
				isDir = true;
			else if (ch == SYMLINK_CHAR)
				isLink = true;
			
			// some servers don't supply the link count
			int linkCount = 0;
            if (Char.IsDigit(fields[index][0])) // assume it is if a digit
            {
                try
				{
					linkCount = System.Int32.Parse(fields[index++]);
				}
				catch (FormatException)
				{
				}
            }
			
			// owner and group
			string owner = fields[index++];
			string group = fields[index++];
            	
			// size
			long size = 0L;
			string sizeStr = fields[index];
            // some listings don't have owner - make owner -> group & group -> size in
            // this case, and use the sizeStr for the start of the date
            if (!Char.IsDigit(sizeStr[0]) && Char.IsDigit(group[0])) 
            {
                sizeStr = group;  
                group = owner;
                owner = "";
            }
            else 
            {
                index++; 
            }
            
			try
			{
				size = Int64.Parse(sizeStr);
			}
			catch (FormatException)
			{
				throw new FormatException("Failed to parse size: " + sizeStr);
			}                     
            
			// next 3 are the date time
			int dateTimePos = index;
			//DateTime lastModified;
			StringBuilder stamp = new StringBuilder(fields[index++]);
			stamp.Append('-').Append(fields[index++]).Append('-');
			
			string field = fields[index++];
			if (field.IndexOf((System.Char) ':') < 0)
			{
				//stamp.Append(field); // year
				//lastModified = DateTime.ParseExact(stamp.ToString(), format1, 
                //                            ParsingCulture.DateTimeFormat, DateTimeStyles.None);
			}
			else
			{
				// add the year ourselves as not present
               // int year = ParsingCulture.Calendar.GetYear(DateTime.Now);
				//stamp.Append(year).Append('-').Append(field);
				//lastModified = DateTime.ParseExact(stamp.ToString(), format2, 
                                    //        ParsingCulture.DateTimeFormat, DateTimeStyles.None);
				
				// can't be in the future - must be the previous year
				//if (lastModified > DateTime.Now)
				//{
                //    lastModified.AddYears(-1);
				//}
			}
			
			// name of file or dir. Extract symlink if possible
			string name = null;
			string linkedname = null;
			
			// we've got to find the starting point of the name. We
			// do this by finding the pos of all the date/time fields, then
			// the name - to ensure we don't get tricked up by a userid the
			// same as the filename,for example
			int pos = 0;
			bool ok = true;
			for (int i = dateTimePos; i < dateTimePos + 3; i++)
			{
				pos = raw.IndexOf(fields[i], pos);
				if (pos < 0)
				{
					ok = false;
					break;
				}
                else {
                    pos += fields[i].Length;
                }
			}
			if (ok)
			{
                string remainder = raw.Substring(pos).Trim();
				if (!isLink)
					name = remainder;
				else
				{
					// symlink, try to extract it
					pos = remainder.IndexOf(SYMLINK_ARROW);
					if (pos <= 0)
					{
						// couldn't find symlink, give up & just assign as name
						name = remainder;
					}
					else
					{
						int len = SYMLINK_ARROW.Length;
						name = remainder.Substring(0, (pos) - (0)).Trim();
						if (pos + len < remainder.Length)
							linkedname = remainder.Substring(pos + len);
					}
				}
			}
			else
			{
				throw new FormatException("Failed to retrieve name: " + raw);
			}
			
			FTPFile file = new FTPFile(FTPFile.UNIX, raw, name, size, isDir, true);
			file.Group = group;
			file.Owner = owner;
			file.Link = isLink;
			file.LinkCount = linkCount;
			file.LinkedName = linkedname;
			file.Permissions = permissions;
			return file;
		}
		
		        /// <summary>
        /// Splits string consisting of fields separated by
        /// whitespace into an array of strings
        /// </summary>
        /// <param name="str">
        /// string to split
        /// </param>   
        protected string[] Split(string str) {
            ArrayList fields = new ArrayList();
            StringBuilder field = new StringBuilder();
            for (int i = 0; i < str.Length; i++) {
                char ch = str[i];
                if (!Char.IsWhiteSpace(ch))
                    field.Append(ch);
                else {
                    if (field.Length > 0) {
                        fields.Add(field.ToString());
                        field.Length = 0;
                    }
                }
            }
            // pick up last field
            if (field.Length > 0) {
                fields.Add(field.ToString());
            }
            string[] result = (string[])fields.ToArray(typeof(string));
            return result;
        }
	}
}
