/*
 * Copyright (C) 2003 Sanjay Madhavan.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
package com.revasoft.shoutcast.plsdownloader;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Properties;
import java.util.Vector;

/**
 * Utility to navigate the www.shoutcast.com website and download pls files for
 * a specified Genre or for all defined Genres. Since shoutcast allows
 * approximately 1000 requests per user per day you can specify the maximum
 * number of pls files to download per genre to prevent your IP address from
 * being banned. Also has support for using proxy servers. You can also request
 * to download stations with a specific bitrate or bitrate range.
 * 
 * The tool will generate a directory for each genre in the specified output
 * directory and the pls files will be named with a friendly name in the format
 * "Sequence StationName(BitRate).pls".
 * 
 * @author Sanjay Madhavan
 *  
 */

public class PlsDownloader implements PlsConstants {

	// statics
	private static final String PROPERTY_FILENAME = "plsdownloader.properties";
	private static final String DEFAULT_GENRE_LIST_FILENAME =
		"default_genres.properties";
	private static final SimpleDateFormat sDateFormat =
		new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
	//
	private GenreDescriptor[] oGenreArray = null;

	private static final SimpleDateFormat sLogDateFormat =
		new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.S");

	// instance variables

	private String oImageDirectory = "images";
	private PlsOptions oPlsOptions = null;
	private Properties oProperties = null;
	private Vector oProxyCollection = new Vector();
	private String oProxyHost = null;
	private int oProxyIndex = -1;
	private int oProxyPort = -1;
	private int SLEEP_BANNED_WAIT_TIME = 1000 * 60 * 60 * 4; // 4 hours
	private int SLEEP_CONNECTION_ERROR_TIME = 30000;
	private int SLEEP_GENRE_WAIT_TIME = 1000;
	private int SLEEP_PLS_WAIT_TIME = 1000;
	//
	/**
	 * Return the list of valid Genres. If a Genre is not listed here this tool
	 * will not process any request for the genre.
	 * 
	 * @return
	 */
	private static GenreDescriptor[] buildDefaultGenreArray() {
		GenreDescriptor[] array = new GenreDescriptor[19];
		int i = -1;
		array[++i] = new GenreDescriptor("TopTen", "Top 25 Streams");
		array[++i] = new GenreDescriptor("Alternative", "Alternative");
		array[++i] = new GenreDescriptor("Classical", "Classical");
		array[++i] = new GenreDescriptor("Comedy", "Comedy");
		array[++i] = new GenreDescriptor("Country", "Country");
		array[++i] = new GenreDescriptor("Dance", "Dance/House");
		array[++i] = new GenreDescriptor("Funk", "Funk");
		array[++i] = new GenreDescriptor("Jazz", "Jazz");
		array[++i] = new GenreDescriptor("Metal", "Metal");
		array[++i] = new GenreDescriptor("Mixed", "Mixed");
		array[++i] = new GenreDescriptor("Pop", "Pop");
		array[++i] = new GenreDescriptor("Rap", "Rap");
		array[++i] = new GenreDescriptor("RnB", "RnB");
		array[++i] = new GenreDescriptor("Rock", "Rock");
		array[++i] = new GenreDescriptor("Talk", "Talk");
		array[++i] = new GenreDescriptor("Techno", "Techno");
		array[++i] = new GenreDescriptor("80s", "The '80s");
		array[++i] = new GenreDescriptor("70s", "The '70s");
		array[++i] = new GenreDescriptor("World", "World");

		return array;
	}

	/**
	 * Standard date format for display.
	 * 
	 * @return
	 */
	private static SimpleDateFormat getDateFormat() {
		return sDateFormat;
	}

	/**
	 * Return the current Genre list
	 * 
	 * @return
	 */
	public GenreDescriptor[] getGenreArray() {
		if (oGenreArray == null) {
			oGenreArray = buildDefaultGenreArray();
		}
		return oGenreArray;
	}
	/**
	 * Standard date format for log entries.
	 * 
	 * @return
	 */
	private static SimpleDateFormat getLogDateFormat() {
		return sLogDateFormat;
	}
	private static void log(String s) {
		System.out.println(getLogDateFormat().format(new Date()) + ": " + s);
	}
	private static void logError(String s) {
		System.err.println(getLogDateFormat().format(new Date()) + ": " + s);
	}
	public static void main(String[] args) {
		if (args.length == 0)
			printUsage();
		else {
			PlsDownloader plsDownloader =
				new PlsDownloader(new PlsOptions(args));
			plsDownloader.run();
		}
	}
	/**
	 * Prints the valid usage of this tool.
	 *  
	 */
	public static void printUsage() {
		log("\nPlsDownloader Valid usage is: \n\tPlsDownloader \n\t\t-DownloadDirectory d:/Shoutcast \n\t\t[-MaxFilesPerGenre 100] \n\t\t[-tProcessGenre Rock]   \n\t\t[-ProxyList proxylist.properties]  \n\t\t[-IgnoreSavedState] \n\t\t[-MaximumBitRate 64] \n\t\t[-MinimumBitRate 128] \n\n");

	}

	/**
	 * Return an instance initialized with the specified options
	 */
	public PlsDownloader(PlsOptions options) {
		setPlsOptions(options);

	}
	/**
	 * Add the pls entry if the entries match the requested bitrate
	 */
	private void addAllAcceptableEntries(
		Vector plsDescriptorList,
		Vector tempList) {
		for (int i = 0; i < tempList.size(); i++) {
			PlsDescriptor pls = (PlsDescriptor) tempList.get(i);
			if (isRequestedBitRate(pls))
				plsDescriptorList.add(pls);
		}

	}
	/**
	 * Copy the image file from the images directory to the target directory
	 * for the specified genre.
	 *  
	 */

	private void copyImageFile(String targetDirectory, GenreDescriptor genre) {
		String srcName =
			getImageDirectory() + "/" + genre.getKey() + TBN_EXTENSION;
		File src = new File(srcName);

		if (src.exists()) {
			boolean overwrite = true;
			String target =
				targetDirectory + "/" + genre.getKey() + TBN_EXTENSION;
			try {
				Util.writeFile(target, Util.readFile(src), overwrite);
			} catch (IOException e) {
				e.printStackTrace();
			} catch (Exception e) {

			}
		} else
			logError("File does not exist: " + src.getAbsolutePath());

	}

	/**
	 * @param parentDirectory
	 * @param directory
	 */

	private void createDirectory(String parentDirectory, String directory) {
		File parent = new File(parentDirectory);
		if (!parent.exists())
			parent.mkdir();
		File d = new File(parent, directory);
		if (!d.exists()) {
			System.out.print("Creating Directory " + d.getAbsolutePath());
			boolean ok = d.mkdir();
			log(" created=" + ok);
		}
	}

	/**
	 * @return
	 */
	private Properties getDefaultDroperties() {
		Properties properties = new Properties();
		properties.put(CURRENTGENRE, getGenreArray()[0].getKey());
		properties.put(STARTPAGEINDEX, "0");
		properties.put(LASTRUN, getDateFormat().format(new Date()));
		return properties;
	}

	/**
	 * Read the specified genre page and return the contents as a String.
	 * 
	 * @param genre
	 * @param startAt
	 * @return
	 */
	private String getGenrePage(GenreDescriptor genre, int startAt) {
		String url =
			Util.replaceString(GENRE_DOWNLOAD_PAGE, _GENRE_, genre.getKey());
		url = Util.replaceString(url, _PAGESTART_, Integer.toString(startAt));
		return Util.getHttpPostContents(url, getProxyHost(), getProxyPort());
	}

	/**
	 * Return the directory where the images used as icons for the genres are
	 * listed.
	 * 
	 * @return
	 */
	public String getImageDirectory() {
		return oImageDirectory;
	}

	/**
	 * Return the current options object being used for the current run.
	 * 
	 * @return Returns the plsOptions.
	 */
	public PlsOptions getOptions() {
		return oPlsOptions;
	}

	/**
	 * Read the properties from the property fileName. The properties contains
	 * the last run state of this tool. If interrupted this tool will attempt
	 * to continue from the lat page processed by reading the properties to
	 * obtain the last genre and last page index processed.
	 * 
	 * @return
	 */
	public Properties getProperties() {
		if (oProperties == null) {
			try {
				File file = new File(PROPERTY_FILENAME);
				if (!file.exists()) {
					oProperties = getDefaultDroperties();
					saveProperties(oProperties);
				}

				FileInputStream in = new FileInputStream(file);

				try {
					oProperties = new Properties();
					oProperties.load(in);
				} finally {
					in.close();
				}
			} catch (Exception exc) {

				logError(
					" properties file "
						+ PROPERTY_FILENAME
						+ " not found: "
						+ exc);
			}
		}
		return oProperties;
	}

	/**
	 * Optional proxy collection to use. If too many requests are detected by
	 * the shoutcast server the local IP address is banned for a day. If a
	 * proxy collection is specified then each time a Banned condition is
	 * detected this tool will try to contibue by switching to the next
	 * available proxy. The proxy list is specified by passing the -ProxyList
	 * filename parameter. The filename must point to a java properties file in
	 * the format below.
	 * 
	 * e.g.
	 * 
	 * <pre>
	 *  ProxyCount=2
	 * 
	 *  Proxy.1=127.0.0.1:80
	 * 
	 *  Proxy.2=127.0.0.2:8001
	 * 
	 * </pre>
	 * 
	 * @return Returns the proxyCollection.
	 */
	public Vector getProxyCollection() {
		return oProxyCollection;
	}
	/**
	 * Return the current proxyHost or null if not defined.
	 * 
	 * @return
	 */
	private String getProxyHost() {

		return oProxyHost;
	}

	/**
	 * Return the current index into the ProxyCollection
	 * 
	 * @return Returns the proxyIndex.
	 */
	public int getProxyIndex() {
		return oProxyIndex;
	}

	/**
	 * Return the current proxyPort or -1 if not defined.
	 * 
	 * @return
	 */
	private int getProxyPort() {
		return oProxyPort;
	}

	/*
	 * Local test method to read the contents of previously saved pages in the
	 * samples directory.
	 */
	private String getTestPageContents(String page) {
		String pageContents = null;
		try {
			pageContents =
				new String(
					Util.readFile(new File("samples" + File.separator + page)));
		} catch (IOException e) {
			e.printStackTrace();
		}
		return pageContents;
	}

	/**
	 * Initialize the tool with the current properties and options and prepare
	 * for a run.
	 *  
	 */
	private void initialize() {
		if (getOptions().isIgnoreSavedState()) {
			saveProperties(getDefaultDroperties());
		}
		initializeProxies();
		initializeGenres();
		log(getOptions().getLogString());
	}
	/**
	 * Read the ProxyList if specified and switch to the first available proxy
	 *  
	 */
	private void initializeProxies() {
		if (getOptions().getProxyList() != null) {
			setProxyCollection(readProxyList(getOptions().getProxyList()));
			if (getProxyCollection().size() > 0)
				switchToNextProxy();
		}
	}
	/**
	 * Read the GenreList if specified or setup default genres
	 *  
	 */
	private void initializeGenres() {
		String fileName =
			getOptions().getGenreList() == null
				? DEFAULT_GENRE_LIST_FILENAME
				: getOptions().getGenreList();

		setGenreArray(readGenreList(fileName));
		if (getGenreArray().length == 0) {
			logError("Genre List was empty using defaults");
			setGenreArray(buildDefaultGenreArray());
		} 
	}

	/**
	 * Return true if the bitrate of the pls is within the requested range.
	 * 
	 * @param pls
	 * @return
	 */
	private boolean isRequestedBitRate(PlsDescriptor pls) {
		if (getOptions().getMinimumBitRate() == null
			&& getOptions().getMaximumBitRate() == null)
			return true;

		boolean bitRateOk = true;
		if (getOptions().getMinimumBitRate() != null) {
			int bitrate = Integer.parseInt(pls.getBitRate());
			bitRateOk =
				(bitrate >= Integer.parseInt(getOptions().getMinimumBitRate()));

		}
		if (getOptions().getMaximumBitRate() != null) {
			int bitrate = Integer.parseInt(pls.getBitRate());
			bitRateOk =
				bitRateOk
					&& (bitrate
						<= Integer.parseInt(getOptions().getMaximumBitRate()));

		}

		return bitRateOk;

	}
	/**
	 * Return true if the current local IP address has been banned by
	 * shoutcast.com
	 * 
	 * @param plsContents
	 * @return
	 */
	private boolean isBanned(String plsContents) {
		return plsContents != null && (plsContents.indexOf(BANNED_TEXT) >= 0);
	}

	/**
	 * Return true if we are at the end of the proxy list
	 * 
	 * @return
	 */
	private boolean isLastProxy() {
		return (getProxyCollection().size() == 0)
			|| (getProxyIndex() == (getProxyCollection().size() - 1));

	}

	/**
	 * Parse the pageContents as a GenrePage and add all references to .pls
	 * files to the plsDescriptorList.
	 * 
	 * @param pageContents
	 * @param plsDescriptorList
	 */
	private boolean parseGenrePage(
		String pageContents,
		Vector plsDescriptorList) {
		boolean hasMore = false;
		int startCount = plsDescriptorList.size();

		int startPage = pageContents.indexOf(PAGEINFOTOKEN);
		if (startPage > 0) {
			int p1 = pageContents.indexOf(PAGETOKEN, startPage - 50);
			if (p1 > 0) {
				int p2 = pageContents.indexOf(OFTOKEN, p1);
				String startPageString =
					pageContents.substring(p1 + 5, p2 - 1).trim();
				String endPageString =
					pageContents.substring(p2 + 3, startPage).trim();
				hasMore = !startPageString.equals(endPageString);
			}

			String temp =
				pageContents.substring(startPage + PAGEINFOTOKEN.length());

			int endPage = pageContents.lastIndexOf(PAGEINFOTOKEN);
			if (endPage < 0)
				endPage = pageContents.length();
			String pageExtract =
				pageContents.substring(
					startPage + PAGEINFOTOKEN.length(),
					endPage);

			int ndx = 0;
			int pageSize = pageExtract.length();
			boolean done = false;
			while (ndx >= 0 && ndx < pageSize && !done) {
				p1 = pageExtract.indexOf(ENDPAGETOKEN, ndx);
				if (p1 < 0) {
					done = true;
					break;
				}
				temp = pageExtract.substring(p1 - 15);
				int p2 = pageExtract.indexOf(BOLDTOKEN, p1 - 15);
				temp = pageExtract.substring(p2);
				String pageSequence =
					pageExtract.substring(p2 + BOLDTOKEN.length(), p1).trim();
				ndx = p1 + ENDPAGETOKEN.length();
				//

				p1 = pageExtract.indexOf(SBIN_START, ndx);
				p2 = pageExtract.indexOf(OPENDOUBLEQUOTE, p1);
				String pageUrl = pageExtract.substring(p1, p2).trim();
				ndx = p2 + 1;
				//
				p1 = pageExtract.indexOf(DESCRIPTION_START_TOKEN, ndx);
				p2 = pageExtract.indexOf(CLOSINGBRACKET, p1);
				String pageDescription =
					pageExtract
						.substring(p1 + DESCRIPTION_START_TOKEN.length(), p2)
						.trim();
				ndx = p2;
				//
				p1 = pageExtract.indexOf(ENDFONT_ENDTD_TOKEN, ndx);
				p1 = pageExtract.indexOf(ENDFONT_ENDTD_TOKEN, p1 + 1);
				p1 = pageExtract.indexOf(ENDFONT_ENDTD_TOKEN, p1 + 1);
				p2 = pageExtract.indexOf(DOUBLEQUOTE_AND_GT, p1 - 6);
				String bitRate = pageExtract.substring(p2 + 2, p1).trim();
				ndx = p1 + 1;

				PlsDescriptor plsDescriptor =
					new PlsDescriptor(
						pageSequence,
						SHOUTCAST_HOME + pageUrl,
						pageDescription,
						bitRate);
				plsDescriptorList.add(plsDescriptor);

			}
		}
		int endCount = plsDescriptorList.size();

		return hasMore;

	}

	/**
	 * The default process method which will process all defined Genres.
	 * Depending on the parameters passed it will attempt to continue from the
	 * last genre and pageIndex processed or will start afresh if the options
	 * specify that the saved state should be ignored.
	 */
	private void processAllGenres() {
		boolean banned = false;
		int startGenreIndex = 0;
		int startPageIndex = 0;
		//
		initialize();
		//
		String startGenre = (String) getProperties().get(CURRENTGENRE);
		if (startGenre != null) {
			boolean found = false;
			for (int i = 0; i < getGenreArray().length && !found; i++) {
				GenreDescriptor genre = getGenreArray()[i];
				if (genre.getKey().equals(startGenre)) {
					found = true;
					startGenreIndex = i;
				}
			}
			if (found) {
				String pageIndex = (String) getProperties().get(STARTPAGEINDEX);
				if (pageIndex != null)
					startPageIndex = Integer.parseInt(pageIndex);
				getProperties().put(
					LASTRUN,
					getDateFormat().format(new Date()));
			}
		}

		//

		for (int i = startGenreIndex;
			i < getGenreArray().length && !banned;
			i++) {

			GenreDescriptor genre = getGenreArray()[i];
			banned = processOneGenre(genre, startPageIndex);
			startPageIndex = 0;
			sleep(SLEEP_GENRE_WAIT_TIME);
		}
		saveProperties("*DONE*", "0");
		log("Finished processing all Genres..");

	}
	/**
	 * Process the specified genre starting from the specified page index.
	 * 
	 * @param genre
	 * @param startIndex
	 * @return
	 */
	private boolean processGenre(GenreDescriptor genre, int startIndex) {

		Vector plsDescriptorList = restoreGenreVector(genre);
		//String pageContents = getTestPageContents("index.htm");
		boolean hasMore = true;

		boolean banned = false;
		while (hasMore
			&& (plsDescriptorList.size() < getOptions().getMaxFilesPerGenre())) {
			log(
				"Processing Genre: ["
					+ genre.getKey()
					+ ","
					+ genre.getDescription()
					+ "] PageIndex="
					+ startIndex);
			int lastIndex = startIndex;
			String pageContents = getGenrePage(genre, startIndex);
			saveProperties(genre.getKey(), Integer.toString(startIndex));
			if (pageContents == null) {
				sleep(SLEEP_CONNECTION_ERROR_TIME);
				switchToNextProxy();
			} else {
				Vector tempList = new Vector();
				hasMore = parseGenrePage(pageContents, tempList);
				startIndex = startIndex + tempList.size();
				for (int i = 0;
					i < tempList.size()
						&& ((i + plsDescriptorList.size())
							< getOptions().getMaxFilesPerGenre());
					i++) {

					PlsDescriptor pls = (PlsDescriptor) tempList.get(i);
					if (isRequestedBitRate(pls)) {
						log("\t" + pls.toString());
						//
						String plsContents = null;
						banned = true;
						while (banned) {
							plsContents =
								Util.getHttpGetContents(
									pls.getUrl(),
									getProxyHost(),
									getProxyPort());
							//
							banned = isBanned(plsContents);
							if (banned) {
								logError("Banned...");
								if (isLastProxy())
									sleep(SLEEP_BANNED_WAIT_TIME);

								switchToNextProxy();
							}

						}
						if (!banned && plsContents != null)
							storePlsContents(genre, pls, plsContents);

					}
				}
				addAllAcceptableEntries(plsDescriptorList, tempList);

			}
		}
		return banned;
	}
	/**
	 * If the ProcessGenre parameter is specified then a single Genre will be
	 * processed by this method.
	 *  
	 */
	private void processOneGenre() {
		initialize();
		String startGenre = (String) getOptions().getProcessGenre();
		boolean found = false;
		GenreDescriptor genre = null;
		for (int i = 0; i < getGenreArray().length && genre == null; i++) {
			if (getGenreArray()[i].getKey().equals(startGenre)) {
				genre = getGenreArray()[i];
			}
		}

		if (genre != null) {
			processOneGenre(genre, 0);
		} else {
			logError("Requested Genre definition not found: " + startGenre);
		}

	}

	/**
	 * Create the target directories and copy the image icon for the specified
	 * genre and kick off the processing of the genre.
	 * 
	 * @param genre
	 * @param startPageIndex
	 * @return
	 */

	private boolean processOneGenre(
		GenreDescriptor genre,
		int startPageIndex) {
		boolean banned;

		createDirectory(getOptions().getDownloadDirectory(), genre.getKey());
		copyImageFile(getOptions().getDownloadDirectory(), genre);

		banned = processGenre(genre, startPageIndex);

		log(
			"Finished processing Genre ["
				+ genre.getKey()
				+ ","
				+ genre.getDescription()
				+ CLOSINGBRACKET);
		return banned;
	}

	/**
	 * Read the specified properties file to obtain the list of valid proxies.
	 * The filename must point to a java properties file in the format below.
	 * 
	 * e.g.
	 * 
	 * <pre>
	 *  ProxyCount=2
	 * 
	 *  Proxy.1=127.0.0.1:80
	 * 
	 *  Proxy.2=127.0.0.2:8001
	 * 
	 * </pre>
	 * 
	 * @return
	 */

	public Vector readProxyList(String proxyListFileName) {
		Properties p = new Properties();
		try {
			File file = new File(proxyListFileName);

			FileInputStream in = new FileInputStream(file);

			try {

				p.load(in);
			} finally {
				in.close();
			}
		} catch (Exception exc) {

			logError(
				" properties file " + proxyListFileName + " not found: " + exc);
		}

		Vector proxyList = new Vector();
		String count = (String) p.get(PROXYCOUNT);
		if (count != null) {
			int c = Integer.parseInt(count.trim());
			for (int i = 0; i < c; i++) {
				String proxy = (String) p.get(PROXY_ + Integer.toString(i));
				if (proxy != null && proxy.trim().length() > 0) {
					proxyList.add(proxy.trim());
				}

			}
		}

		return proxyList;
	}

	/**
	 * Read the specified properties file to obtain the list of valid genres.
	 * The filename must point to a java properties file in the format below.
	 * 
	 * e.g.
	 * 
	 * <pre>
	 *  GenreCount=2
	 * 
	 *  Genre.1=Key,Description
	 * 
	 *  Genre.2=127.0.0.2:8001
	 * 
	 * </pre>
	 * 
	 * @return
	 */

	public GenreDescriptor[] readGenreList(String genreListFileName) {
		Properties p = new Properties();

		try {
			File file = new File(genreListFileName);

			FileInputStream in = new FileInputStream(file);

			try {

				p.load(in);
			} finally {
				in.close();
			}
		} catch (Exception exc) {

			logError(
				" properties file " + genreListFileName + " not found: " + exc);
		}

		Vector genreList = new Vector();
		String count = (String) p.get(GENRECOUNT);
		if (count != null) {
			int c = Integer.parseInt(count.trim());
			for (int i = 1; i <= c; i++) {
				String genre = (String) p.get(GENRE_ + Integer.toString(i));
				if (genre != null && genre.trim().length() > 0) {
					int ndx = genre.indexOf(',');
					if (ndx > 0) {
						GenreDescriptor gd =
							new GenreDescriptor(
								genre.substring(0, ndx).trim(),
								genre
									.substring(ndx + 1, genre.length())
									.trim());
						genreList.add(gd);

					}
				}

			}
		}

		GenreDescriptor[] gdArray = new GenreDescriptor[genreList.size()];
		for (int i = 0; i < genreList.size(); i++) {
			gdArray[i] = (GenreDescriptor) genreList.get(i);

		}

		return gdArray;
	}

	/**
	 * Read the genre directory and return the list of pls files found
	 * 
	 * @param genre
	 * @return
	 */
	private Vector restoreGenreVector(GenreDescriptor genre) {
		File genreDirectory =
			new File(
				getOptions().getDownloadDirectory()
					+ File.separator
					+ genre.getKey());
		File[] files = genreDirectory.listFiles();
		Vector list = new Vector();
		if (files != null) {
			for (int i = 0; i < files.length; i++) {
				File file = files[i];
				if (file.isFile()
					&& file.getAbsolutePath().toLowerCase().endsWith(
						PLS_EXTENSION)) {

					PlsDescriptor plsDescriptor =
						PlsDescriptor.fromFileName(
							file.getAbsolutePath().toLowerCase());
					list.add(plsDescriptor);
				}

			}
		}
		return list;
	}
	/**
	 * Main start processing method
	 *  
	 */
	public void run() {
		if (getOptions().getProcessGenre() != null)
			processOneGenre();
		else
			processAllGenres();
	}

	/**
	 * Save the specified properties to the default property file name.
	 * 
	 * @param properties
	 */
	private void saveProperties(Properties properties) {
		try {

			FileOutputStream fileOS = new FileOutputStream(PROPERTY_FILENAME);

			properties.store(
				fileOS,
				"PLSDownloader Properties. *** Saved run state ***");
			fileOS.close();

		} catch (java.io.FileNotFoundException e) {
			e.printStackTrace();
		} catch (java.io.IOException e) {
			e.printStackTrace();
		}

	}

	private void saveProperties(String genre, String index) {

		getProperties().put(CURRENTGENRE, genre);
		getProperties().put(STARTPAGEINDEX, index);
		getProperties().put(LASTRUN, getDateFormat().format(new Date()));
		saveProperties(getProperties());

	}

	/**
	 * @param imageDirectory
	 */
	private void setImageDirectory(String imageDirectory) {
		oImageDirectory = imageDirectory;
	}

	/**
	 * @param plsOptions
	 *            The plsOptions to set.
	 */
	public void setPlsOptions(PlsOptions plsOptions) {
		oPlsOptions = plsOptions;
	}

	/**
	 * @param proxyCollection
	 *            The proxyCollection to set.
	 */
	public void setProxyCollection(Vector proxyCollection) {
		oProxyCollection = proxyCollection;
	}

	/**
	 * @param proxyHost
	 *            The proxyHost to set.
	 */
	public void setProxyHost(String proxyHost) {
		oProxyHost = proxyHost;
	}

	/**
	 * @param proxyIndex
	 *            The proxyIndex to set.
	 */
	public void setProxyIndex(int proxyIndex) {
		oProxyIndex = proxyIndex;
	}

	/**
	 * @param proxyPort
	 *            The proxyPort to set.
	 */
	public void setProxyPort(int proxyPort) {
		oProxyPort = proxyPort;
	}

	/**
	 * Convenience method to sleep for the specified milliseconds and log the
	 * sleep period.
	 *  
	 */
	private void sleep(long sleeptime) {
		try {
			Date sleepTill = new Date(new Date().getTime() + sleeptime);
			log("Sleeping till: " + getDateFormat().format(sleepTill));
			Thread.sleep(sleeptime);
		} catch (InterruptedException e) {
			e.printStackTrace();
		}

	}

	/**
	 * Store the contents of the pls file using the pls descriptor to obtain
	 * the filename and bitrate etc.
	 * 
	 * @param genre
	 * @param plsContents
	 */
	private void storePlsContents(
		GenreDescriptor genre,
		PlsDescriptor pls,
		String plsContents) {

		String targetFile =
			getOptions().getDownloadDirectory()
				+ File.separator
				+ genre.getKey()
				+ File.separator
				+ pls.getFileName(genre);
		try {
			Util.writeFile(targetFile, plsContents.getBytes(), true);
		} catch (Exception e) {

			e.printStackTrace();
		}
	}

	/**
	 * Start using the next available proxy
	 */
	private void switchToNextProxy() {
		if (getProxyCollection().size() > 0) {
			setProxyIndex((getProxyIndex() + 1) % getProxyCollection().size());
			if (getProxyCollection().size() > 0
				&& getProxyIndex() >= 0
				&& getProxyIndex() < getProxyCollection().size()) {
				String temp =
					(String) getProxyCollection().get(getProxyIndex());
				setProxyHost(temp.substring(0, temp.indexOf(':')));
				setProxyPort(
					Integer.parseInt(
						temp.substring(temp.indexOf(':') + 1, temp.length())));

				log("Switching to proxy: " + temp);

			}
		}

	}

	/**
	 * @param genreArray
	 *            The genreArray to set.
	 */
	public void setGenreArray(GenreDescriptor[] genreArray) {
		oGenreArray = genreArray;
	}

}
