/*
	Copyright 2003 Chris Cavey
	
	This file is part of XID.
	
	XID is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.
	
	XID is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
	
	You should have received a copy of the GNU General Public License
	along with XID; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "xidreport.h"

XIDREPORT::XIDREPORT()
{
	flag = false;
}

void XIDREPORT::generateInputReport(unsigned char *inputReport)
{
	unsigned int x;

	// zero the entire report for safety's sake
	for(x = 0; x < inputReportSize; x++)
		inputReport[x] = 0;

	// map SHORT axes if they exist
	for(x = 0; x < 4; x++)
		if(offsetMapping[x] != -1)
			*((short *)&inputReport[offsetMapping[x]]) = padInterface.Axis(dynConfig.getAxisMap(x));

	// map BYTE axes
	for(x = 0; x < 4; x++)
		if(offsetMapping[4+x] != -1)
			inputReport[offsetMapping[4+x]] = padInterface.Button((const XIDFEATURE_BUTTON *)dynConfig.getAxisMap(4+x));
	
	// hat
	if(offsetMapping[POVHAT] != -1)
		inputReport[offsetMapping[POVHAT]] = padInterface.Hat(dynConfig.getHat());
	
	//buttons 
	for(x = 0; x < 16; x++)
		if(offsetMapping[8+x] != -1)
			inputReport[offsetMapping[8+x]] = padInterface.Button(dynConfig.getButtonMap(x));

	return;
}

// This rather ugly looking routine will generate an HID report descriptor based on the config values
// It will also calculate the size of the descriptor, the size of the input report, and the control offsets
void XIDREPORT::generateReportDescriptor(unsigned char *buffer)
{
	unsigned int configValue = dynConfig.getGeneralConfig(), 
		          buttonValue = dynConfig.getButtonConfig();

	ByteStream reportDesc(buffer);

	/*
	unsigned char tbuffer[50];
	if(dynConfig.ReadRegBinary(L"test", tbuffer, -50))
	{
		for(int x = 0; x < 8; x++)
			DebugPrint("%x", tbuffer[x]);
	}
	*/

	//setup some initial values
	inputReportSize = largeAxesCount = smallAxesCount = 0;
	flag = true;
	for(int x = 0; x < 25; x++)
		offsetMapping[x] = -1;
	
	// must at least define one axis, a no control HID report is an unhappy HID report.
	if(configValue == 0)
		configValue = 1;
	
	// start with basic header info
	reportDesc << 0x05 << 0x01;		// 	USAGE_PAGE (Generic Desktop)
   reportDesc << 0x09 << 0x04;		// 	USAGE (Joystick)
   reportDesc << 0xA1 << 0x01;		//	COLLECTION (Application)

	// if we don't have at least one large axis, don't even both
	if((configValue & (_X_FLAG + _Y_FLAG + _RX_FLAG + _RY_FLAG)) != 0)
	{
		reportDesc << 0x09 << 0x01;		// 		USAGE (Pointer)
		reportDesc << 0xA1 << 0x00;		//   	COLLECTION (Physical)

		// decided which 16-bit axes we're going to include ( really only X, Y, Rx, Ry )

		if(configValue & _X_FLAG)
		{
			reportDesc << 0x09 << 0x30;	//    		USAGE (X)
			largeAxesCount++;
			offsetMapping[X_AXIS] = inputReportSize;
			inputReportSize+=2;
		}

		if(configValue & _Y_FLAG)
		{
			reportDesc << 0x09 << 0x31;	//    		USAGE (Y)
			largeAxesCount++;
			offsetMapping[Y_AXIS] = inputReportSize;
			inputReportSize+=2;
		}

		if(configValue & _RX_FLAG)
		{
			reportDesc << 0x09 << 0x33;	//    		USAGE (Rx)
			largeAxesCount++;
			offsetMapping[RX_AXIS] = inputReportSize;
			inputReportSize+=2;
		}

		if(configValue & _RY_FLAG)
		{
			reportDesc << 0x09 << 0x34;	//    		USAGE (Ry)
			largeAxesCount++;
			offsetMapping[RY_AXIS] = inputReportSize;
			inputReportSize+=2;
		}

		reportDesc << 0x16 << 0x00 << 0x80;		//	LOGICAL_MINIMUM (-32768)
		reportDesc << 0x26 << 0xFF << 0x7F;    // LOGICAL_MAXIMUM (32767)
		reportDesc << 0x95 << largeAxesCount;  //	REPORT_COUNT (4)
		reportDesc << 0x75 << 0x10;				//	REPORT_SIZE (16)
		reportDesc << 0x81 << 0x02;				//	INPUT (Data,Var,Abs)
		reportDesc << 0xc0;					      //	END_COLLECTION
	}

	// if we don't have at least one large axis, don't even both
	if((configValue & (_Z_FLAG + _RZ_FLAG + _S1_FLAG + _S2_FLAG)) != 0)
	{
		reportDesc << 0x05 << 0x01;			//	USAGE_PAGE (Generic Desktop)
		
		if(configValue & _Z_FLAG)
		{
			reportDesc << 0x09 << 0x32;	//    		USAGE (Z)
			smallAxesCount++;
			offsetMapping[Z_AXIS] = inputReportSize;
			inputReportSize++;
		}

		if(configValue & _RZ_FLAG)
		{
			reportDesc << 0x09 << 0x35;	//    		USAGE (Rz)
			smallAxesCount++;
			offsetMapping[RZ_AXIS] = inputReportSize;
			inputReportSize++;
		}

		if(configValue & _S1_FLAG)
		{
			reportDesc << 0x09 << 0x36;	//    		USAGE (some other kind)
			smallAxesCount++;
			offsetMapping[S1_AXIS] = inputReportSize;
			inputReportSize++;
		}

		if(configValue & _S2_FLAG)
		{
			reportDesc << 0x09 << 0x37;	//    		USAGE (some other kind still)
			smallAxesCount++;
			offsetMapping[S2_AXIS] = inputReportSize;
			inputReportSize++;
		}

		reportDesc << 0x15 << 0x00;			  //	LOGICAL_MINIMUM (0)
		reportDesc << 0x26 << 0xFF << 0x00;	  //	LOGICAL_MAXIMUM (255)
		reportDesc << 0x95 << smallAxesCount; //	REPORT_COUNT (2)
		reportDesc << 0x75 << 0x08;			  //	REPORT_SIZE (8)
		reportDesc << 0x81 << 0x02;			  //	INPUT (Data,Var,Abs)
	}

	if(buttonValue != 0)
	{
		for(int bNum = 0; bNum < 32; bNum+=2) // gotta skip bits
			if((1 << bNum) & buttonValue) // digital button
			{
				reportDesc << 0x05 << 0x09;                    	//   	USAGE_PAGE (Button)
				reportDesc << 0x09 << (bNum >> 1)+1;			//   	USAGE (Button X)
				reportDesc << 0x15 << 0x00;			           	//		LOGICAL_MINIMUM (0)
				reportDesc << 0x25 << 0x01;						//	 	LOGICAL_MAXIMUM (1)
				reportDesc << 0x75 << 0x01;                    	//   	REPORT_SIZE (1)
				reportDesc << 0x95 << 0x01;                    	//   	REPORT_COUNT (1)
				reportDesc << 0x81 << 0x02;                   	//   	INPUT (Data,Var,Abs)

				//pad it out to make my life WAY easier
				reportDesc << 0x95 << 0x07;				   		//   	REPORT_COUNT (7)
				reportDesc << 0x75 << 0x01;                    	//   	REPORT_SIZE (1)
				reportDesc << 0x81 << 0x03;                    	//   	INPUT (Cnst,Var,Abs)

				offsetMapping[((bNum >> 1)+1) + 7] = inputReportSize;
				inputReportSize++;
				//DebugPrint("Button %d is digital", (bNum >> 1)+1);
			}
			else if((2 << bNum) & buttonValue) // analog button
			{
				reportDesc << 0x05 << 0x09;                    	//   	USAGE_PAGE (Button)
				reportDesc << 0x09 << (bNum >> 1)+1;           //   	USAGE (Button X)
				reportDesc << 0x15 << 0x00;			           	//		LOGICAL_MINIMUM (0)
				reportDesc << 0x26 << 0xFF << 0x00;		        //	 	LOGICAL_MAXIMUM (255)
				reportDesc << 0x75 << 0x08;                    	//   	REPORT_SIZE (8)
				reportDesc << 0x95 << 0x01;                    	//   	REPORT_COUNT (1)
				reportDesc << 0x81 << 0x02;                   	//   	INPUT (Data,Var,Abs)

				offsetMapping[((bNum >> 1)+1) + 7] = inputReportSize;
				inputReportSize++;
					
				//DebugPrint("Button %d is analog", (bNum >> 1)+1);
			}
	}

	if(configValue & _HAT_FLAG)
	{
		reportDesc << 0x05 << 0x01;                    	// 		USAGE_PAGE (Generic Desktop)
		reportDesc << 0x09 << 0x39;                    	// 		USAGE (Hat switch)
		reportDesc << 0x15 << 0x00;                    	// 		LOGICAL_MINIMUM (0)
		reportDesc << 0x25 << 0x07;                    	// 		LOGICAL_MAXIMUM (7)
		reportDesc << 0x35 << 0x00;                    	// 		PHYSICAL_MINIMUM (0)
		reportDesc << 0x46 << 0x3B << 0x01;              // 	PHYSICAL_MAXIMUM (315)
		reportDesc << 0x65 << 0x14;                    	// 		UNIT (Eng Rot:Angular Pos)
		reportDesc << 0x55 << 0x00;                   	// 		UNIT_EXPONENT (0)
		reportDesc << 0x75 << 0x04;                   	// 		REPORT_SIZE (4)
		reportDesc << 0x95 << 0x01;                    	// 		REPORT_COUNT (1)
		reportDesc << 0x81 << 0x42;                     // 		INPUT (Data,Var,Abs,Null)
		reportDesc << 0x95 << 0x04;					   	//   	REPORT_COUNT (4)
		reportDesc << 0x75 << 0x01;                    	//   	REPORT_SIZE (1)
		reportDesc << 0x81 << 0x03;                    	//   	INPUT (Cnst,Var,Abs)

		offsetMapping[POVHAT] = inputReportSize;
		inputReportSize++;
	}

	// ULTRA-JUICY SUPER CLEVER FEATURE REPORT

	reportDesc << 0x06 << 0x00 << 0xff;             //		USAGE_PAGE (Vendor Defined Page 1)
    reportDesc << 0x09 << 0x01;                     //		USAGE (Vendor Usage 1)
    reportDesc << 0x15 << 0x00;                     //		LOGICAL_MINIMUM (0)
    reportDesc << 0x26 << 0xff << 0x00;             //		LOGICAL_MAXIMUM (255)
    reportDesc << 0x75 << 0x08;                     //		REPORT_SIZE (8)
    reportDesc << 0x95 << 0x7F;                     //		REPORT_COUNT (127) + 1 gives a pretty 128 report size. I like powers of 2
    reportDesc << 0xb1 << 0x02;                     //		FEATURE (Data,Var,Abs)


	// OUTPUT REPORT FOR SOME RUMBLE ACTION

    reportDesc << 0x05 << 0x01;                    	// 		USAGE_PAGE (Generic Desktop)
    reportDesc << 0xa1 << 0x02;                    	// 		COLLECTION (Logical)
    reportDesc << 0x15 << 0x00;                    	//  		LOGICAL_MINIMUM (0)
    reportDesc << 0x26 << 0xff << 0x00;				//   		LOGICAL_MAXIMUM (255)
    reportDesc << 0x35 << 0x00;                    	//   		PHYSICAL_MINIMUM (0)
    reportDesc << 0x46 << 0xff << 0x00;             //   		PHYSICAL_MAXIMUM (255)
    reportDesc << 0x95 << 0x02;                    	//   		REPORT_COUNT (2)
    reportDesc << 0x75 << 0x08;                    	//   		REPORT_SIZE (8)
    reportDesc << 0x06 << 0x00 << 0xff;             //   		USAGE_PAGE (Generic Desktop)
    reportDesc << 0x09 << 0x01;                    	//   		USAGE (Vendor Usage 1)
    reportDesc << 0x91 << 0x42;                     //   		OUTPUT (Data,Var,Abs,Null)
    reportDesc << 0xc0;								// 		END_COLLECTION

	reportDesc << 0xc0;					//	END_COLLECTION

	reportDescSize = reportDesc.counter();
}

unsigned int XIDREPORT::getReportDescSize()
{
	if(!flag)
		generateReportDescriptor(NULL);
	
	return reportDescSize;
}
unsigned int XIDREPORT::getInputReportSize()
{
	if(!flag)
		generateReportDescriptor(NULL);
	return inputReportSize;
}