/*
	Copyright 2003 Chris Cavey
	
	This file is part of XID.
	
	XID is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.
	
	XID is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
	
	You should have received a copy of the GNU General Public License
	along with XID; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "xidconfig.h"

DWORD XIDCONFIG::readRegDword(WCHAR *entry, DWORD *dest)
{
	WCHAR testWChar[] = L"\\REGISTRY\\MACHINE\\SOFTWARE\0";
	NTSTATUS status;

	RTL_QUERY_REGISTRY_TABLE QueryTable[3];
	RtlZeroMemory(QueryTable, sizeof(QueryTable));

	QueryTable[0].Name  = L"XID";
	QueryTable[0].Flags = RTL_QUERY_REGISTRY_SUBKEY;
	QueryTable[0].EntryContext = NULL;
	QueryTable[1].Name  = entry;
	QueryTable[1].Flags = RTL_QUERY_REGISTRY_DIRECT;
	QueryTable[1].EntryContext = dest;

	status = RtlQueryRegistryValues(RTL_REGISTRY_ABSOLUTE, (PWSTR)testWChar, QueryTable, NULL, NULL);

	return status == STATUS_SUCCESS ? 1 : 0;
}

DWORD XIDCONFIG::readRegBinary(WCHAR *entry, void *buffer, signed long bufferSize)
{
	WCHAR testWChar[] = L"\\REGISTRY\\MACHINE\\SOFTWARE\0";

	RTL_QUERY_REGISTRY_TABLE QueryTable[3];
	RtlZeroMemory( QueryTable, sizeof(QueryTable));

	QueryTable[0].Name  = L"XID";
	QueryTable[0].Flags = RTL_QUERY_REGISTRY_SUBKEY;
	QueryTable[0].EntryContext = NULL;
	((signed long *)buffer)[0] = bufferSize;
	QueryTable[1].Name  = entry;
	QueryTable[1].Flags = RTL_QUERY_REGISTRY_DIRECT;
	QueryTable[1].EntryContext = buffer;

	NTSTATUS status = RtlQueryRegistryValues(RTL_REGISTRY_ABSOLUTE, (PWSTR)testWChar, QueryTable, NULL, NULL);
	
	return status == STATUS_SUCCESS ? 1 : 0;
}

void  XIDCONFIG::initializeInternalConfigDesc()
{
	// attempt to read the registry configuration values, if not present use some "safe" defaults
	if(!readRegDword(L"genConfig", &genConfig))
	{	
		genConfig = 3; // defines X Axis and Y Axis in the report
	}

	if(!readRegDword(L"buttonConfig", &buttonConfig))
	{
		buttonConfig = 0x5555; // defines some simple digital buttons
	}

	for(DWORD x = 0; x < 16; x++)
	{
		buttonMap[x].virtualButton = (BYTE)x;	
		buttonMap[x].physicalControl = (x < 10) ? x + 1 : 0;
		buttonMap[x].threshold = 128;
		buttonMap[x].flags = 0;
		buttonMap[x].type = getButtonType(x); // 1 = digital button, 2 = analog button, 0 = undefined
		DbgPrint("the last type was %d", buttonMap[x].type);
	}

	axisMap[0].virtualAxis = 0;
	axisMap[0].physicalControl = 24;
	axisMap[0].flags = 0;
	axisMap[0].threshold = 0;

	axisMap[1].virtualAxis = 1;
	axisMap[1].physicalControl = 23;
	axisMap[1].flags = XIDFEATUREFLAG_INVERT;
	axisMap[1].threshold = 0;

	axisMap[2].virtualAxis = 2;
	axisMap[2].physicalControl = 26;
	axisMap[2].flags = 0;
	axisMap[2].threshold = 0;

	axisMap[3].virtualAxis = 3;
	axisMap[3].physicalControl = 25;
	axisMap[3].flags = XIDFEATUREFLAG_INVERT;
	axisMap[3].threshold = 0;

	axisMap[4].virtualAxis = 4;
	axisMap[4].physicalControl = 19;
	axisMap[4].flags =0;
	axisMap[4].threshold = 0;

	axisMap[5].virtualAxis = 5;
	axisMap[5].physicalControl = 20;
	axisMap[5].flags = 0;
	axisMap[5].threshold = 0;

	axisMap[6].virtualAxis = 6;
	axisMap[6].physicalControl = 0;
	axisMap[6].flags = 0;
	axisMap[6].threshold = 0;

	axisMap[7].virtualAxis = 7;
	axisMap[7].physicalControl = 0;
	axisMap[7].flags = 0;
	axisMap[7].threshold = 0;

	for(DWORD x = 0; x < 8; x++)
	{
		hat[x].hatIndex = (BYTE)x;
		hat[x].physicalControl = 11+x;
		hat[x].flags = 0;
		hat[x].threshold = 25000;
	}
}

DWORD XIDCONFIG::getGeneralConfig()
{
	return genConfig;
}
DWORD XIDCONFIG::getButtonConfig()
{
	return buttonConfig;
}

// using the current button config value, tell me if it is digital or analog. 1 or 2 respectively, 0 = not defined
BYTE XIDCONFIG::getButtonType(BYTE buttonNumber)
{
	BYTE ret = 0;

	// recall that bits 0 and 1 correspond to button 0, buts 2 and 3 to button 1, etc ..
	// so given button # n, 2n = first bit, 2n+1 = second bit.

	if((1 << (buttonNumber*2)) & buttonConfig) // digital button
		ret = 1;
	else if((1 << ((buttonNumber*2)+1)) & buttonConfig) // analog button
		ret = 2;

	return ret;
}

BYTE XIDCONFIG::processButtonFeature(XIDFEATURE_BUTTON *buttonIn)
{
	// boy, this is the easy way. Thank you byte packing ;)
	// notice that we don't copy the whole structure, the last field, type, is reserved.
	// ensure that it isn't modified.
	memcpy(&buttonMap[buttonIn->virtualButton], buttonIn, sizeof(XIDFEATURE_BUTTON)-1);
	//DbgPrint("recieved button map ... vbut(%d) to pbut(%d)..", buttonIn->virtualButton, buttonIn->physicalControl);
	//DbgPrint("thresh: %d, flags: %d, x1: %d, y1: %d, x2: %d, y2: %d, xh: %d, yh: %d", buttonIn->threshold,  buttonIn->flags,  buttonIn->x1,  buttonIn->y1,  buttonIn->x2,  buttonIn->y2,  buttonIn->xh,  buttonIn->yh);
	return 0; 
}

BYTE XIDCONFIG::processAxisFeature(XIDFEATURE_AXIS *axesIn)
{
	//DbgPrint("recieved axis map ... vbut(%d) to pbut(%d)..", axesIn->virtualAxis, axesIn->physicalControl);
	memcpy(&axisMap[axesIn->virtualAxis], axesIn, sizeof(XIDFEATURE_AXIS)-1);
	return 0;
}

BYTE XIDCONFIG::processHatFeature(XIDFEATURE_HAT *hatIn)
{
	memcpy(&hat[hatIn->hatIndex], hatIn, sizeof(XIDFEATURE_HAT));
	return 0;
}

void XIDCONFIG::updateInternalConfigDesc(BYTE *configReport, DWORD size)
{
	BYTE offset = 0, typeID = *configReport;

	while(typeID)
	{
		offset++;
		switch(typeID)
		{
			case XIDFEATURETYPE_BUTTON:
				processButtonFeature((XIDFEATURE_BUTTON*)&configReport[offset]);
				offset += sizeof(XIDFEATURE_BUTTON);
				break;
			case XIDFEATURETYPE_AXIS:
				processAxisFeature((XIDFEATURE_AXIS*)&configReport[offset]);
				offset += sizeof(XIDFEATURE_AXIS);
				break;
			case XIDFEATURETYPE_HAT:
				processHatFeature((XIDFEATURE_HAT*)&configReport[offset]);
				offset += sizeof(XIDFEATURE_HAT);
				break;
			default:
				typeID = 0;
				break;
		}

		if(typeID)
		{
			typeID = configReport[offset];
		}
	}

	return;
}