/*
	Copyright 2003 Chris Cavey
	
	This file is part of XID.
	
	XID is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.
	
	XID is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
	
	You should have received a copy of the GNU General Public License
	along with XID; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "xid.h"
#include "xidioctl.h"

// PROTOTYPES ==========================================================================
NTSTATUS XidAddDevice(IN PDRIVER_OBJECT, IN PDEVICE_OBJECT);
VOID XidUnload(IN PDRIVER_OBJECT);
NTSTATUS XidCreateClose(IN PDEVICE_OBJECT, IN PIRP);
NTSTATUS XidPnp(IN PDEVICE_OBJECT, IN PIRP);
NTSTATUS XidPower(IN PDEVICE_OBJECT, IN PIRP);
NTSTATUS XidInternalIoctl(IN PDEVICE_OBJECT, IN PIRP);
NTSTATUS Xid_InitDevice(IN PDEVICE_OBJECT, IN PIRP);
VOID Xid_RemoveDevice(DEVICE_EXTENSION *);
NTSTATUS Xid_PnPComplete (IN PDEVICE_OBJECT ,IN PIRP, IN PVOID );
NTSTATUS Xid_IncRequestCount(IN DEVICE_EXTENSION *);
VOID Xid_DecRequestCount(IN DEVICE_EXTENSION *);
VOID WorkerThread(IN PDEVICE_OBJECT , IN PVOID );

// END PROTOTYPES ======================================================================

// I'm sure this has to do with code segmentation, TODO: Look up code segements
#pragma code_seg("INIT") // start INIT section
////////////////////////////////////////////////////////////////////////////////////////
extern "C" NTSTATUS DriverEntry(IN PDRIVER_OBJECT DriverObject, IN PUNICODE_STRING RegistryPath)
{
	NTSTATUS ntStatus = STATUS_SUCCESS;
	HID_MINIDRIVER_REGISTRATION hidMinidriverRegistration;

	// Export other driver entry points...
	DriverObject->DriverExtension->AddDevice = XidAddDevice;
	DriverObject->DriverUnload = XidUnload;
	DriverObject->MajorFunction[IRP_MJ_CREATE] = XidCreateClose;
	DriverObject->MajorFunction[IRP_MJ_CLOSE] = XidCreateClose;
	DriverObject->MajorFunction[IRP_MJ_PNP] = XidPnp;
	DriverObject->MajorFunction[IRP_MJ_POWER] = XidPower;
	DriverObject->MajorFunction[IRP_MJ_INTERNAL_DEVICE_CONTROL] = XidInternalIoctl;

	RtlZeroMemory(&hidMinidriverRegistration, sizeof(hidMinidriverRegistration));
    hidMinidriverRegistration.Revision            = HID_REVISION;
    hidMinidriverRegistration.DriverObject        = DriverObject;
    hidMinidriverRegistration.RegistryPath        = RegistryPath;
    hidMinidriverRegistration.DevicesArePolled    = TRUE;
	hidMinidriverRegistration.DeviceExtensionSize = sizeof(DEVICE_EXTENSION);

	DbgPrint("%ws", RegistryPath);

	ntStatus = HidRegisterMinidriver(&hidMinidriverRegistration);

	return ntStatus;
}
///////////////////////////////////////////////////////////////////////////////
#pragma code_seg() // end INIT section


#pragma code_seg("PAGE") // start PAGE section
//////////////////////////////////////////////////////////////////////////////

NTSTATUS XidAddDevice(IN PDRIVER_OBJECT DriverObject, IN PDEVICE_OBJECT FunctionalDeviceObject)
{

	NTSTATUS ntStatus = STATUS_SUCCESS;
	PDEVICE_OBJECT DeviceObject;
	DEVICE_EXTENSION *DeviceExtension;

    DeviceObject = FunctionalDeviceObject;
	DeviceExtension = GET_MINIDRIVER_DEVICE_EXTENSION(DeviceObject);
    DeviceObject->Flags &= ~DO_DEVICE_INITIALIZING;

    //Initialize the remove lock 
    DeviceExtension->RequestCount = 1;
    KeInitializeEvent(&DeviceExtension->RemoveEvent, SynchronizationEvent, FALSE);

	KeInitializeSpinLock(&DeviceExtension->lockReport);

	return ntStatus;
}

VOID XidUnload(IN PDRIVER_OBJECT DriverObject)
{
	return;
}

NTSTATUS XidCreateClose(IN PDEVICE_OBJECT DeviceObject, IN PIRP Irp)
{
	PIO_STACK_LOCATION   IrpStack;
    NTSTATUS ntStatus = STATUS_SUCCESS;

    IrpStack = IoGetCurrentIrpStackLocation(Irp);

    switch(IrpStack->MajorFunction)
    {
        case IRP_MJ_CREATE:
            Irp->IoStatus.Information = 0;
            break;

        case IRP_MJ_CLOSE:
            Irp->IoStatus.Information = 0;
            break;

        default:
            ntStatus = STATUS_INVALID_PARAMETER;
            break;
    }

    Irp->IoStatus.Status = ntStatus;
    IoCompleteRequest(Irp, IO_NO_INCREMENT);

	return ntStatus;
}

NTSTATUS XidPnp(IN PDEVICE_OBJECT DeviceObject, IN PIRP Irp)
{
	NTSTATUS ntStatus;
    DEVICE_EXTENSION *DeviceExtension;
    KEVENT StartEvent;
	PIO_STACK_LOCATION IrpStack;
	DEVICE_CAPABILITIES *pDevCaps;
	
    // Get a pointer to the device extension
    DeviceExtension = GET_MINIDRIVER_DEVICE_EXTENSION(DeviceObject);
    ntStatus = Xid_IncRequestCount(DeviceExtension);
    
    if (!NT_SUCCESS(ntStatus))
    {
        // Someone sent us another plug and play IRP after removed
        Irp->IoStatus.Information = 0;
        Irp->IoStatus.Status = ntStatus;
        IoCompleteRequest (Irp, IO_NO_INCREMENT);
    } 
	else
    {
        // Get a pointer to the current location in the Irp
        IrpStack = IoGetCurrentIrpStackLocation (Irp);

        switch(IrpStack->MinorFunction)
        {
            case IRP_MN_START_DEVICE:
				
                // We cannot touch the device (send it any non pnp irps) until a
                // start device has been passed down to the lower drivers.
                
                KeInitializeEvent(&StartEvent, NotificationEvent, FALSE);

                IoCopyCurrentIrpStackLocationToNext (Irp);
                IoSetCompletionRoutine (Irp, Xid_PnPComplete, &StartEvent, TRUE, TRUE, TRUE);
                ntStatus = IoCallDriver (GET_NEXT_DEVICE_OBJECT(DeviceObject), Irp);

                if( NT_SUCCESS(ntStatus ) )
                {
                    ntStatus = KeWaitForSingleObject
                               (
                               &StartEvent,
                               Executive,   /* Waiting for reason of a driver */
                               KernelMode,  /* Waiting in kernel mode         */
                               FALSE,       /* No allert                      */
                               NULL         /* No timeout                     */
                               );
                }

                if(NT_SUCCESS(ntStatus))
                {
                    ntStatus = Irp->IoStatus.Status;
                }

                if(NT_SUCCESS (ntStatus))
                {
                    // As we are now back from our start device we can do work.
                    ntStatus = Xid_InitDevice (DeviceObject, Irp);
                } 

                DeviceExtension->fStarted = TRUE;
                
				//IoSetDeviceInterfaceState(&DeviceExtension->ifSymLinkName, TRUE);

                //Return Status
                Irp->IoStatus.Information = 0;
                Irp->IoStatus.Status = ntStatus;
                IoCompleteRequest (Irp, IO_NO_INCREMENT);
                break;
                
            case IRP_MN_STOP_DEVICE:
			

                /*
                 * After the start IRP has been sent to the lower driver object, the bus may
                 * NOT send any more IRPS down ``touch'' until another START has occured.
                 * Whatever access is required must be done before Irp passed on.
                 */

                DeviceExtension->fStarted = FALSE;

                /*
                 * We don't need a completion routine so fire and forget.
                 * Set the current stack location to the next stack location and
                 * call the next device object.
                 */

                IoSkipCurrentIrpStackLocation (Irp);
                ntStatus = IoCallDriver (GET_NEXT_DEVICE_OBJECT(DeviceObject), Irp);
                break;

            case IRP_MN_SURPRISE_REMOVAL:
            
                Xid_RemoveDevice(DeviceExtension);
                Irp->IoStatus.Status = STATUS_SUCCESS;
                IoSkipCurrentIrpStackLocation(Irp);
                ntStatus = IoCallDriver (GET_NEXT_DEVICE_OBJECT(DeviceObject), Irp);
                break;

            case IRP_MN_REMOVE_DEVICE:
				
                /*
                 * The PlugPlay system has dictacted the removal of this device. We
                 * have no choice but to detach and delete the device object.
                 * (If we wanted to express an interest in preventing this removal,
                 * we should have filtered the query remove and query stop routines.)
                 * Note: we might receive a remove WITHOUT first receiving a stop.
                 */

                //Make sure we do not allow more IRPs to start touching the device
                DeviceExtension->fRemoved = TRUE;

                //Stop the device without touching the hardware.
                Xid_RemoveDevice(DeviceExtension);

                //Send on the remove IRP
                IoSkipCurrentIrpStackLocation (Irp);
                ntStatus = IoCallDriver (GET_NEXT_DEVICE_OBJECT(DeviceObject), Irp);


                //Remove this IRPs hold which should leave the initial 1 plus any other IRP holds.
                //I can only assume that these braces force scope onto the LONG RequestCount
                {
                    LONG RequestCount = InterlockedDecrement( &DeviceExtension->RequestCount );
                    //ASSERT( RequestCount > 0 );
                }

                //If someone has already started, wait for them to finish
                if(InterlockedDecrement(&DeviceExtension->RequestCount ) > 0)
                {
                    KeWaitForSingleObject( &DeviceExtension->RemoveEvent, Executive, KernelMode, FALSE, NULL );
                }

				//IoSetDeviceInterfaceState(&DeviceExtension->ifSymLinkName, FALSE);
				//RtlFreeUnicodeString(&DeviceExtension->ifSymLinkName);

                ntStatus = STATUS_SUCCESS;
                return ntStatus;
			
			case IRP_MN_QUERY_CAPABILITIES:
			
				pDevCaps = IrpStack->Parameters.DeviceCapabilities.Capabilities;
				
			    //pDevCaps->Version = 1;
			    //pDevCaps->Size = sizeof(DEVICE_CAPABILITIES);
			
			    // We cannot wake the system. Seems fair enough.
			    pDevCaps->SystemWake = PowerSystemUnspecified;
			    pDevCaps->DeviceWake = PowerDeviceUnspecified;
			
			    // We have no latencies .. okay sure
			    pDevCaps->D1Latency = 0;
			    pDevCaps->D2Latency = 0;
			    pDevCaps->D3Latency = 0;
			
			    // No locking or ejection ... this sounds good.
			    pDevCaps->LockSupported = FALSE;
			    pDevCaps->EjectSupported = FALSE;
			    pDevCaps->Removable = TRUE;
			    pDevCaps->SurpriseRemovalOK = TRUE;
			
			    // not Docking device
			    pDevCaps->DockDevice = FALSE;
			 
			    pDevCaps->UniqueID = FALSE;
			
				ntStatus = STATUS_SUCCESS;
			   	Irp->IoStatus.Information = 0;
                Irp->IoStatus.Status = ntStatus;
                IoCompleteRequest (Irp, IO_NO_INCREMENT);
				break;
				
            default:
            
                IoSkipCurrentIrpStackLocation (Irp);
                ntStatus = IoCallDriver(GET_NEXT_DEVICE_OBJECT(DeviceObject), Irp);
                break;
        }

        Xid_DecRequestCount( DeviceExtension );
    }
    
	return ntStatus;
}

NTSTATUS XidPower(IN PDEVICE_OBJECT DeviceObject, IN PIRP Irp)
{
	NTSTATUS ntStatus = STATUS_SUCCESS;
	DEVICE_EXTENSION *DeviceExtension;

	DeviceExtension = GET_MINIDRIVER_DEVICE_EXTENSION (DeviceObject);

	ntStatus = Xid_IncRequestCount(DeviceExtension);

    if(!NT_SUCCESS (ntStatus))
    {
        Irp->IoStatus.Information = 0;
        Irp->IoStatus.Status = ntStatus;
        IoCompleteRequest (Irp, IO_NO_INCREMENT);
    } 
    else
    {
        IoSkipCurrentIrpStackLocation(Irp);
        PoStartNextPowerIrp(Irp);
        ntStatus = PoCallDriver(GET_NEXT_DEVICE_OBJECT(DeviceObject), Irp);
        Xid_DecRequestCount(DeviceExtension);
    }

	return ntStatus;
}

NTSTATUS XidInternalIoctl(IN PDEVICE_OBJECT DeviceObject, IN PIRP Irp)
{
	NTSTATUS ntStatus = STATUS_SUCCESS;
    DEVICE_EXTENSION *DeviceExtension;
    PIO_STACK_LOCATION IrpStack;

	//Get a pointer to the device extension
    DeviceExtension = GET_MINIDRIVER_DEVICE_EXTENSION(DeviceObject);
    //Get a pointer to the current location in the Irp
    IrpStack = IoGetCurrentIrpStackLocation(Irp);

    ntStatus = Xid_IncRequestCount( DeviceExtension );
    
    if(!NT_SUCCESS(ntStatus))
    {
        Irp->IoStatus.Information = 0;
        Irp->IoStatus.Status = ntStatus;
    } 
	else
    {
        switch(IrpStack->Parameters.DeviceIoControl.IoControlCode)
        {
            case IOCTL_HID_GET_DEVICE_DESCRIPTOR:
                ntStatus = XidIoctlGetDeviceDescriptor(DeviceObject, Irp);
                break;

            case IOCTL_HID_GET_REPORT_DESCRIPTOR:
                ntStatus = XidIoctlGetReportDescriptor(DeviceObject, Irp);
                break;

            case IOCTL_HID_READ_REPORT:
                ntStatus = XidIoctlReadReport(DeviceObject, Irp);
                break;
             
			case IOCTL_HID_SET_OUTPUT_REPORT:
            case IOCTL_HID_WRITE_REPORT:
            	 ntStatus = XidIoctlWriteReport(DeviceObject, Irp);
            	 break;

            case IOCTL_HID_GET_DEVICE_ATTRIBUTES:
                ntStatus = XidIoctlGetAttributes(DeviceObject, Irp);
                break;

			case IOCTL_HID_SET_FEATURE:
				ntStatus = XidIoctlSetFeature(DeviceObject, Irp);
				break;

			case IOCTL_HID_GET_FEATURE:
				DbgPrint("get feature");
				ntStatus = XidIoctlGetFeature(DeviceObject, Irp);
				break;

            default:
				DbgPrint("ioctl code: %d", IrpStack->Parameters.DeviceIoControl.IoControlCode);
                ntStatus = STATUS_NOT_SUPPORTED;
                break;
        }


        //Set real return status in Irp
        Irp->IoStatus.Status = ntStatus;

        Xid_DecRequestCount( DeviceExtension );
    }


    if(ntStatus != STATUS_PENDING)
    {
        IoCompleteRequest(Irp, IO_NO_INCREMENT);
        // NOTE: Real return status set in Irp->IoStatus.Status
        ntStatus = STATUS_SUCCESS;
    } 
	else
    {
        IoMarkIrpPending( Irp );
    }

    return ntStatus;
}

// END OF CORE FUNCTIONS //////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////

NTSTATUS Xid_InitDevice(IN PDEVICE_OBJECT DeviceObject, IN PIRP Irp)
{
    NTSTATUS ntStatus = STATUS_SUCCESS;
    DEVICE_EXTENSION *DeviceExtension;
    ULONG  descriptorSize;
	PUSB_DEVICE_DESCRIPTOR deviceDescriptor;
	PUSB_CONFIGURATION_DESCRIPTOR configDescriptors;

	ntStatus = UsbResetDevice(GET_HID_DEVICE_EXTENSION(DeviceObject));
	ntStatus = UsbGetDeviceDescriptor(GET_HID_DEVICE_EXTENSION(DeviceObject), deviceDescriptor, descriptorSize);
	
	if(NT_SUCCESS(ntStatus))
	{
		DeviceExtension = GET_MINIDRIVER_DEVICE_EXTENSION(DeviceObject);
		DeviceExtension->ProductId = deviceDescriptor->idProduct;
		DeviceExtension->VendorId = deviceDescriptor->idVendor;
		ExFreePool(deviceDescriptor);
	}

	ntStatus = UsbSelectConfiguration(GET_HID_DEVICE_EXTENSION(DeviceObject));

	DeviceExtension->reportGen.dynConfig.initializeInternalConfigDesc();

	PIO_WORKITEM ioWorkItem;
	ioWorkItem = IoAllocateWorkItem(DeviceObject);

	if(ioWorkItem == NULL)
	{
		ntStatus = STATUS_UNSUCCESSFUL;
	}
	else
	{
		IoQueueWorkItem(ioWorkItem, WorkerThread, DelayedWorkQueue, (PVOID)ioWorkItem);
	}
    return ntStatus;
}

VOID WorkerThread(IN PDEVICE_OBJECT DeviceObject, IN PVOID Context)
{
	DEVICE_EXTENSION *dx = GET_MINIDRIVER_DEVICE_EXTENSION(DeviceObject);
	PIO_WORKITEM ioWorkItem = (PIO_WORKITEM)Context;
	KIRQL irql;
	BYTE rawIn[20];
	ULONG mySize = 20, temp;

	UsbDoInterruptTransfer(GET_HID_DEVICE_EXTENSION(DeviceObject), (void*)&rawIn, mySize, TRANSFER_IN);

	KeAcquireSpinLock(&dx->lockReport, &irql);
		dx->reportGen.padInterface.UpdateInternalReport((void*)&rawIn, 20);
	KeReleaseSpinLock(&dx->lockReport, irql);

	IoFreeWorkItem(ioWorkItem);

	if(dx->fRemoved != TRUE)
	{
		ioWorkItem = IoAllocateWorkItem(DeviceObject);

		if(ioWorkItem != NULL)
		{
			IoQueueWorkItem(ioWorkItem, WorkerThread, DelayedWorkQueue, (PVOID)ioWorkItem);
		}
	}
}

VOID Xid_RemoveDevice(DEVICE_EXTENSION *DeviceExtension)
{
    if (DeviceExtension->fSurpriseRemoved) 
	{
        return;
    }

    DeviceExtension->fSurpriseRemoved = TRUE;

	return;
} 

NTSTATUS Xid_PnPComplete (IN PDEVICE_OBJECT DeviceObject,IN PIRP Irp, IN PVOID Context)
{
    NTSTATUS ntStatus = STATUS_MORE_PROCESSING_REQUIRED;
    
    KeSetEvent((PKEVENT)Context, 0, FALSE);

    /*
     *  If the lower driver returned PENDING, mark our stack location as
     *  pending also. This prevents the IRP's thread from being freed if
     *  the client's call returns pending.
     */
    if(Irp->PendingReturned)
    {
        IoMarkIrpPending(Irp);
    }

    return ntStatus;
}


NTSTATUS Xid_IncRequestCount(IN DEVICE_EXTENSION *DeviceExtension)
{
    NTSTATUS ntStatus;

    InterlockedIncrement(&DeviceExtension->RequestCount);
     
    if( DeviceExtension->fRemoved )
    {
        /*
         *  PnP has already told us to remove the device so fail and make 
         *  sure that the event has been set.
         */
        if( 0 == InterlockedDecrement( &DeviceExtension->RequestCount ) ) 
        {
            KeSetEvent( &DeviceExtension->RemoveEvent, IO_NO_INCREMENT, FALSE );
        }
        ntStatus = STATUS_DELETE_PENDING;
    }
    else
    {
        ntStatus = STATUS_SUCCESS;
    }

    return ntStatus;
}

VOID Xid_DecRequestCount(IN DEVICE_EXTENSION *DeviceExtension)
{
    LONG LocalCount;

    LocalCount = InterlockedDecrement( &DeviceExtension->RequestCount );

    if( LocalCount == 0 )
    {
        /*
         *  PnP has already told us to remove the device so the PnP remove 
         *  code should have set device as removed and should be waiting on
         *  the event.
         */
        KeSetEvent( &DeviceExtension->RemoveEvent, IO_NO_INCREMENT, FALSE );
    }
    return;
}
//////////////////////////////////////////////////////////////////////////////
#pragma code_seg() // end PAGE section
