/*
	Copyright 2003 Chris Cavey
	
	This file is part of XID.
	
	XID is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.
	
	XID is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
	
	You should have received a copy of the GNU General Public License
	along with XID; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "XIDInterface.h"

namespace XID
{
	int Controller::openHandle()
	{
		currentHandle = CreateFile(symName.c_str(), GENERIC_READ | GENERIC_WRITE, FILE_SHARE_READ | FILE_SHARE_WRITE, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);

		if(currentHandle == INVALID_HANDLE_VALUE) 
		{
			currentHandle = NULL;
			LASTERROR_AND_RETURN(-19)
		}

		return 0;
	}

	int Controller::closeHandle()
	{
		if(currentHandle != INVALID_HANDLE_VALUE && currentHandle != NULL) 
			if(CloseHandle(currentHandle) == 0)
			{
				LASTERROR_AND_RETURN(-19)
			}
		return 0;
	}

	int Controller::getReportSizes()
	{
		PHIDP_PREPARSED_DATA preParsedData;
		HIDP_CAPS caps;

		if(openHandle() != 0)
			LASTERROR_AND_RETURN(-20)

		if(!HidD_GetPreparsedData(currentHandle, &preParsedData))
		{
			closeHandle();
			LASTERROR_AND_RETURN(-21)
		}

		if(HidP_GetCaps(preParsedData, &caps) != HIDP_STATUS_SUCCESS)
		{
			closeHandle();
			LASTERROR_AND_RETURN(-22)
		}

		inputSize =  caps.InputReportByteLength;
		outputSize = caps.OutputReportByteLength;
		featureSize = caps.FeatureReportByteLength;
		
		HidD_FreePreparsedData(preParsedData);

		if(closeHandle() != 0)
			LASTERROR_AND_RETURN(-20)

		return 0;
	}

	int Controller::setFeature(char *data, unsigned int dataSize)
	{
		char *paddedBuffer = NULL;
		if(dataSize >= featureSize) //must have null term to make me happy
			LASTERROR_AND_RETURN(-23)

		if(openHandle() != 0)
			LASTERROR_AND_RETURN(-20)

		paddedBuffer = new char[featureSize];
		ZeroMemory(paddedBuffer, featureSize);
		memcpy(paddedBuffer, data, dataSize);

		if(HidD_SetFeature(currentHandle, (void *)paddedBuffer, featureSize) != TRUE)
		{
			delete [] paddedBuffer;
			closeHandle();
			LASTERROR_AND_RETURN(-24)
		}

		delete [] paddedBuffer;
		
		if(closeHandle() != 0)
			LASTERROR_AND_RETURN(-20)

		return 0;
	}

	int Controller::getFeature(char *data, unsigned int dataSize)
	{
		if(openHandle() != 0)
			LASTERROR_AND_RETURN(-20)

		if(HidD_GetFeature(currentHandle, (void *)data, dataSize) != TRUE)
		{
			closeHandle();
			LASTERROR_AND_RETURN(-24)
		}

		if(closeHandle() != 0)
			LASTERROR_AND_RETURN(-20)

		return 0;
	}

	Controller::Controller(std::string idIn, std::string devDescIn, std::string pidpre, std::string symIn)
	{
		lastError = 0;
		id = idIn;
		devDesc = devDescIn;
		pIdPrefix = pidpre;
		symName = symIn;

		getReportSizes();
	}

	std::string Controller::deviceId()
	{
		return id;
	}
	std::string Controller::description()
	{
		return devDesc;
	}
	std::string Controller::parentIdPrefix()
	{
		return pIdPrefix;
	}
	std::string Controller::symbolicName()
	{
		return symName;
	}

	int Controller::enable()
	{
		HDEVINFO devs = INVALID_HANDLE_VALUE;
		SP_DEVINFO_DATA devInfo;
		SP_PROPCHANGE_PARAMS pcp;
		SP_DEVINSTALL_PARAMS devParams;
		char buffer[1024] = { 0 };

		devs = SetupDiCreateDeviceInfoList(NULL, NULL); // emtpy list, populate manuall with one id

		if(devs == INVALID_HANDLE_VALUE)
			LASTERROR_AND_RETURN(-6)

		if(!SetupDiOpenDeviceInfo(devs, id.c_str(), NULL, 0, NULL))
			LASTERROR_AND_RETURN(-7)
		
		devInfo.cbSize = sizeof(devInfo);
		if(!SetupDiEnumDeviceInfo(devs, 0, &devInfo))
			LASTERROR_AND_RETURN(-8)

		if(!SetupDiGetDeviceInstanceId(devs, &devInfo, (PSTR)buffer, 1024, NULL))
			LASTERROR_AND_RETURN(-9)
		
		//compare full ids just to make sure
		if(std::string(buffer) == id)
		{
			// now we are 100% sure we can enable the device
			pcp.ClassInstallHeader.cbSize = sizeof(SP_CLASSINSTALL_HEADER);
			pcp.ClassInstallHeader.InstallFunction = DIF_PROPERTYCHANGE;
			pcp.StateChange = DICS_ENABLE;
			pcp.Scope = DICS_FLAG_GLOBAL;
			pcp.HwProfile = 0;

			if(SetupDiSetClassInstallParams(devs, &devInfo, &pcp.ClassInstallHeader, sizeof(pcp))) 
			{
				SetupDiCallClassInstaller(DIF_PROPERTYCHANGE, devs, &devInfo);
			}

			pcp.ClassInstallHeader.cbSize = sizeof(SP_CLASSINSTALL_HEADER);
			pcp.ClassInstallHeader.InstallFunction = DIF_PROPERTYCHANGE;
			pcp.StateChange = DICS_ENABLE;
			pcp.Scope = DICS_FLAG_CONFIGSPECIFIC;
			pcp.HwProfile = 0;

			if(!SetupDiSetClassInstallParams(devs, &devInfo, &pcp.ClassInstallHeader, sizeof(pcp)) || !SetupDiCallClassInstaller(DIF_PROPERTYCHANGE, devs, &devInfo)) 
				LASTERROR_AND_RETURN(-10)
			else
			{
				// see if device needs reboot
				devParams.cbSize = sizeof(devParams);
				if(SetupDiGetDeviceInstallParams(devs, &devInfo, &devParams) && (devParams.Flags & (DI_NEEDRESTART|DI_NEEDREBOOT))) 
					return 2; // uh oh ... needs to reboot.
				else
					return 1; // sucess! The device in question was enabled.
			}
		}

		if(devs != INVALID_HANDLE_VALUE) 
			SetupDiDestroyDeviceInfoList(devs);
		
		return 0;
	}
	int Controller::disable()
	{
		HDEVINFO devs = INVALID_HANDLE_VALUE;
		SP_DEVINFO_DATA devInfo;
		SP_PROPCHANGE_PARAMS pcp;
		SP_DEVINSTALL_PARAMS devParams;
		char buffer[1024] = { 0 };

		devs = SetupDiCreateDeviceInfoList(NULL, NULL); // emtpy list, populate manuall with one id

		if(devs == INVALID_HANDLE_VALUE)
			LASTERROR_AND_RETURN(-6)

		if(!SetupDiOpenDeviceInfo(devs, id.c_str(), NULL, 0, NULL))
			LASTERROR_AND_RETURN(-7)
		
		devInfo.cbSize = sizeof(devInfo);
		if(!SetupDiEnumDeviceInfo(devs, 0, &devInfo))
			LASTERROR_AND_RETURN(-8)

		if(!SetupDiGetDeviceInstanceId(devs, &devInfo, (PSTR)buffer, 1024, NULL))
			LASTERROR_AND_RETURN(-9)
		
		//compare full ids just to make sure
		if(std::string(buffer) == id)
		{
			// now we are 100% sure we can enable the device
			pcp.ClassInstallHeader.cbSize = sizeof(SP_CLASSINSTALL_HEADER);
			pcp.ClassInstallHeader.InstallFunction = DIF_PROPERTYCHANGE;
			pcp.StateChange = DICS_DISABLE;
			pcp.Scope = DICS_FLAG_CONFIGSPECIFIC;
			pcp.HwProfile = 0;

			if(!SetupDiSetClassInstallParams(devs, &devInfo, &pcp.ClassInstallHeader, sizeof(pcp)) || !SetupDiCallClassInstaller(DIF_PROPERTYCHANGE, devs, &devInfo)) 
				LASTERROR_AND_RETURN(-16)
			else
			{
				// see if device needs reboot
				devParams.cbSize = sizeof(devParams);
				if(SetupDiGetDeviceInstallParams(devs, &devInfo, &devParams) && (devParams.Flags & (DI_NEEDRESTART|DI_NEEDREBOOT))) 
					return 2; // uh oh ... needs to reboot.
				else
					return 1; // sucess! The device in question was enabled.
			}
		}

		if(devs != INVALID_HANDLE_VALUE) 
			SetupDiDestroyDeviceInfoList(devs);
		
		return 0;
	}

	int Controller::mapButton(BYTE vButton, BYTE pButton, WORD thresh, XIDHOTRECT *hRect, BYTE flags)
	{
		BYTE temp[20] = { 0 };
		temp[0] = 0; // report id
		temp[1] = 1; // typeid
		temp[2] = vButton; // vbut
		temp[3] = pButton; // pbut
		temp[4] = flags; // flags
		*((WORD *)&temp[5]) = thresh; // 5 & 6
		if(hRect)
		{
			SHORT *tempRect = (SHORT *)&temp[7];
			tempRect[0] = hRect->x1;
			tempRect[1] = hRect->y1;
			tempRect[2] = hRect->x2;
			tempRect[3] = hRect->y2;
			tempRect[4] = hRect->xh;
			tempRect[5] = hRect->yh;
		}

		return setFeature((char*)temp, 19);
	}

	int Controller::mapAxis(BYTE virtualAxis, BYTE physicalControl, WORD threshold, BYTE flags)
	{
		BYTE temp[7];
		temp[0] = 0; // report id
		temp[1] = 2; // typeid
		temp[2] = virtualAxis; // vbut
		temp[3] = physicalControl; // pbut
		temp[4] = flags; // flags
		*((WORD *)&temp[5]) = threshold; // 5 & 6

		return setFeature((char*)temp, 7);
	}

	int Controller::rumble(BYTE lowMotor, BYTE highMotor)
	{
		char outputReport[3] = { 0 };
		outputReport[0] = 0; // redundant, but proves a point
		outputReport[1] = lowMotor;
		outputReport[2] = highMotor;

		if(openHandle() != 0)
			LASTERROR_AND_RETURN(-20)

		if(!HidD_SetOutputReport(currentHandle, outputReport, outputSize))
		{
			closeHandle();
			LASTERROR_AND_RETURN(-25)
		}
		
		if(closeHandle() != 0)
			LASTERROR_AND_RETURN(-20)

		return 0;
	}

	// Must follow the rule of the big three, now that dynamic memory is involved
	// I must also implement the copy contructor and assignment.
	ControllerEnumerator::~ControllerEnumerator()
	{
		Controller *temp;
		unsigned int cCount = ControllerList.size();
		for(unsigned int x = 0; x < cCount; x++)
		{
			temp = ControllerList[x];
			ControllerList.erase(ControllerList.begin());
			delete temp;
		}
	}

	int ControllerEnumerator::enumerateSymbolicNames()
	{
		GUID HidGuid;
		SP_INTERFACE_DEVICE_DATA ifdata;
		HDEVINFO info;
		DWORD ReqLen, devIndex;

		symNameList.clear();

		HidD_GetHidGuid(&HidGuid);

		info = SetupDiGetClassDevs(&HidGuid, NULL, NULL, DIGCF_PRESENT | DIGCF_INTERFACEDEVICE);
		if(info == INVALID_HANDLE_VALUE)
			LASTERROR_AND_RETURN(-2)

		ifdata.cbSize = sizeof(ifdata);
		for(devIndex = 0; SetupDiEnumDeviceInterfaces(info, NULL, &HidGuid, devIndex, &ifdata); devIndex++)
		{

			SetupDiGetDeviceInterfaceDetail(info, &ifdata, NULL, 0, &ReqLen, NULL);
			PSP_INTERFACE_DEVICE_DETAIL_DATA ifDetail = (PSP_INTERFACE_DEVICE_DETAIL_DATA)(new char[ReqLen]);
			if(ifDetail == NULL)
			{
				SetupDiDestroyDeviceInfoList(info);
				LASTERROR_AND_RETURN(-17)
			}

			// Get symbolic link name
			ifDetail->cbSize = sizeof(SP_INTERFACE_DEVICE_DETAIL_DATA);
			if(!SetupDiGetDeviceInterfaceDetail(info, &ifdata, ifDetail, ReqLen, NULL, NULL))
			{
				SetupDiDestroyDeviceInfoList(info);
				delete ifDetail;
				LASTERROR_AND_RETURN(-18)
			}

			symNameList.push_back((std::string)ifDetail->DevicePath);
		}

		return 0;
	}

	std::string ControllerEnumerator::getParentIdPrefix(std::string rPath)
	{
		BYTE buffer[64] = { 0 };
		DWORD bufferSize = 64;
		HKEY systemKey, ccsKey, enumKey, usbKey, deviceKey, idKey;
		int firstSlash, secondSlash;
		firstSlash = (int)rPath.find('\\');
		secondSlash = (int)rPath.rfind('\\');

		if(RegOpenKey(HKEY_LOCAL_MACHINE, "SYSTEM", &systemKey) ==  ERROR_SUCCESS)
		{
			if(RegOpenKey(systemKey, "CurrentControlSet", &ccsKey) ==  ERROR_SUCCESS)
			{
				if(RegOpenKey(ccsKey, "Enum", &enumKey) ==  ERROR_SUCCESS)
				{
					if(RegOpenKey(enumKey, rPath.substr(0, firstSlash).c_str(), &usbKey) ==  ERROR_SUCCESS)
					{
						if(RegOpenKey(usbKey, rPath.substr(firstSlash+1, secondSlash-firstSlash-1).c_str(), &deviceKey) ==  ERROR_SUCCESS)
						{
							if(RegOpenKey(deviceKey, rPath.substr(secondSlash+1).c_str(), &idKey) ==  ERROR_SUCCESS)
							{
								RegQueryValueEx(idKey, "ParentIdPrefix", NULL, NULL, buffer, &bufferSize);
								RegCloseKey(idKey);
							}
							RegCloseKey(deviceKey);
						}
						RegCloseKey(usbKey);
					}
					RegCloseKey(enumKey);
				}
				RegCloseKey(ccsKey);
			}
			RegCloseKey(systemKey);
		}

		return std::string((char*)buffer);
	}

	int ControllerEnumerator::refreshControllerList()
	{
		char buffer[1024] = { 0 }, buffer2[1024] = { 0 };
		DWORD numClass;
		GUID HIDClass;
		SP_DEVINFO_DATA devInfo;
		HDEVINFO devs = INVALID_HANDLE_VALUE;

		enumerateSymbolicNames();
		ControllerList.clear();

		if(!SetupDiClassGuidsFromName("HIDClass", (LPGUID)&HIDClass, 1, &numClass))
			LASTERROR_AND_RETURN(-1)

		devs = SetupDiGetClassDevs(&HIDClass, NULL, NULL, DIGCF_PRESENT);
		if(devs == INVALID_HANDLE_VALUE)
			LASTERROR_AND_RETURN(-2)

		devInfo.cbSize = sizeof(devInfo);
		for(DWORD devIndex = 0; SetupDiEnumDeviceInfo(devs,devIndex,&devInfo); devIndex++) 
		{	
			if(!SetupDiGetDeviceRegistryProperty(devs, &devInfo, SPDRP_MFG, NULL, (PBYTE)buffer, 1024, NULL))
				LASTERROR_AND_RETURN(-3)
			
			// This is an awful way to id a device...
			if(strcmp(buffer, "ZeroX") == 0)
			{
				// device instance
				if(SetupDiGetDeviceInstanceId(devs, &devInfo, (PSTR)buffer, 1024, NULL))
				{
					std::string temp;

					// actual name
					if(!SetupDiGetDeviceRegistryProperty(devs, &devInfo, SPDRP_DEVICEDESC , NULL, (PBYTE)buffer2, 1024, NULL))
						LASTERROR_AND_RETURN(-3)

					// now get the symbolic name
					temp = getParentIdPrefix(std::string(buffer));
					for(unsigned int y = 0; y < symNameList.size(); y++)
						if(symNameList[y].find(temp) != std::string::npos)
							ControllerList.push_back(new Controller(std::string(buffer), std::string(buffer2), temp, symNameList[y]));
				}
				else
					LASTERROR_AND_RETURN(-4)
			}
		}

		return (int)ControllerList.size();
	}
	unsigned int ControllerEnumerator::controllerCount()
	{
		return (unsigned int)ControllerList.size();
	}


	Controller *ControllerEnumerator::getController(unsigned int index)
	{
		Controller *retVal = NULL;

		if(index >= 0 && index < controllerCount())
			retVal = ControllerList[index];

		return retVal;
	}
	
}