/*
	Copyright 2003 Chris Cavey
	
	This file is part of XID.
	
	XID is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.
	
	XID is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
	
	You should have received a copy of the GNU General Public License
	along with XID; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "xidusb.h"

NTSTATUS CallUSBDI( IN PHID_DEVICE_EXTENSION dx, IN PVOID UrbEtc, IN ULONG IoControlCode, IN ULONG Arg2)
{
	IO_STATUS_BLOCK IoStatus;
	KEVENT event;

	// Initialise IRP completion event
	KeInitializeEvent(&event, NotificationEvent, FALSE);

	// Build Internal IOCTL IRP
	PIRP Irp = IoBuildDeviceIoControlRequest(
					IoControlCode, dx->NextDeviceObject,
					NULL, 0,	// Input buffer
					NULL, 0,	// Output buffer
					TRUE, &event, &IoStatus);

	// Get IRP stack location for next driver down (already set up)
	PIO_STACK_LOCATION NextIrpStack = IoGetNextIrpStackLocation(Irp);
	// Store pointer to the URB etc
	NextIrpStack->Parameters.Others.Argument1 = UrbEtc;
	NextIrpStack->Parameters.Others.Argument2 = (PVOID)Arg2;

	// Call the driver and wait for completion if necessary
	NTSTATUS status = IoCallDriver( dx->NextDeviceObject, Irp);
	if (status == STATUS_PENDING)
	{
		status = KeWaitForSingleObject( &event, Suspended, KernelMode, FALSE, NULL);
		status = IoStatus.Status;
	}
	
	return status;
}

NTSTATUS UsbGetPortStatus( IN PHID_DEVICE_EXTENSION dx, OUT ULONG& PortStatus)
{
	PortStatus = 0;
	NTSTATUS status = CallUSBDI( dx, &PortStatus, IOCTL_INTERNAL_USB_GET_PORT_STATUS);
	return status;
}

NTSTATUS UsbResetPort( IN PHID_DEVICE_EXTENSION dx)
{
	NTSTATUS status = CallUSBDI( dx, NULL, IOCTL_INTERNAL_USB_RESET_PORT);
	return status;
}

NTSTATUS UsbResetDevice( IN PHID_DEVICE_EXTENSION dx)
{
	ULONG PortStatus;

	NTSTATUS status = UsbGetPortStatus( dx, PortStatus);

	if( !NT_SUCCESS(status))
		return status;

	// Give up if device not connected
	if( !(PortStatus & USBD_PORT_CONNECTED))
		return STATUS_NO_SUCH_DEVICE;

	// Return OK if port enabled
	if( PortStatus & USBD_PORT_ENABLED)
		return status;

	// Port disabled so attempt reset
	status = UsbResetPort(dx);
	if( !NT_SUCCESS(status))
		return status;

	// See if it is now working
	status = UsbGetPortStatus( dx, PortStatus);
	if( !NT_SUCCESS(status))
		return status;
	if( !(PortStatus & USBD_PORT_CONNECTED) ||
		!(PortStatus & USBD_PORT_ENABLED))
		return STATUS_NO_SUCH_DEVICE;
	return status;
}

NTSTATUS UsbGetUsbInfo( IN PHID_DEVICE_EXTENSION dx)
{
#if _WIN32_WINNT>=0x0500
	USB_BUS_NOTIFICATION BusInfo;
	NTSTATUS status = CallUSBDI( dx, &BusInfo, IOCTL_INTERNAL_USB_GET_BUS_INFO);
	

	int len = BusInfo.ControllerNameLength+50;
	PUSB_HUB_NAME HubName = (PUSB_HUB_NAME)ExAllocatePool( NonPagedPool, len);
	RtlZeroMemory( HubName, len);
	if( HubName==NULL)
		return STATUS_INSUFFICIENT_RESOURCES;

	status = CallUSBDI( dx, HubName, IOCTL_INTERNAL_USB_GET_CONTROLLER_NAME, BusInfo.ControllerNameLength);

	ExFreePool(HubName);
	return status;
#else
	return STATUS_NOT_SUPPORTED;
#endif
}

/////////////////////////////////////////////////////////////////////////////
//	UsbGetDeviceDescriptor:	Get device descriptor (allocate memory for it)
//							Remember to ExFreePool this memory
NTSTATUS UsbGetDeviceDescriptor( IN PHID_DEVICE_EXTENSION dx, OUT PUSB_DEVICE_DESCRIPTOR& deviceDescriptor, OUT ULONG& Size)
{
	// Allocate memory for URB
	USHORT UrbSize = sizeof(struct _URB_CONTROL_DESCRIPTOR_REQUEST);
	PURB urb = (PURB)ExAllocatePool(NonPagedPool, UrbSize);
	if(urb == NULL)
	{
		return STATUS_INSUFFICIENT_RESOURCES;
	}

	// Allocate memory for device descriptor
	ULONG sizeDescriptor = sizeof(USB_DEVICE_DESCRIPTOR);
	deviceDescriptor = (PUSB_DEVICE_DESCRIPTOR)ExAllocatePool(NonPagedPool, sizeDescriptor);
	if( deviceDescriptor==NULL)
	{
		ExFreePool(urb);
		return STATUS_INSUFFICIENT_RESOURCES;
	}

	// Build the Get Descriptor URB
	UsbBuildGetDescriptorRequest(
		urb, UrbSize, 
		USB_DEVICE_DESCRIPTOR_TYPE, 0, 0,	// Types, Index & LanguageId
		deviceDescriptor, NULL, sizeDescriptor,    // Transfer buffer
		NULL);	// Link URB

	// Call the USB driver
	NTSTATUS status = CallUSBDI( dx, urb);
	// Check statuses
	if( !NT_SUCCESS(status) || !USBD_SUCCESS( urb->UrbHeader.Status))
	{
		status = STATUS_UNSUCCESSFUL;
	}
	// Remember count of bytes actually transferred
	Size = urb->UrbControlDescriptorRequest.TransferBufferLength;

	ExFreePool(urb);
	return status;
}

/////////////////////////////////////////////////////////////////////////////
//	UsbGetConfigurationDescriptors:	Get specified Config and associated descriptors
//									Allocate memory for descriptors.
//									Remember to ExFreePool this memory

NTSTATUS UsbGetConfigurationDescriptors( IN PHID_DEVICE_EXTENSION dx, 
										 OUT PUSB_CONFIGURATION_DESCRIPTOR& descriptors,
										 IN UCHAR ConfigIndex,
										 OUT ULONG& DescriptorsSize)
{
	// Allocate memory for URB
	USHORT UrbSize = sizeof(struct _URB_CONTROL_DESCRIPTOR_REQUEST);
	PURB urb = (PURB)ExAllocatePool(NonPagedPool, UrbSize);
	if( urb == NULL)
	{
		return STATUS_INSUFFICIENT_RESOURCES;
	}

	// Allocate memory just for basic config descriptor
	DescriptorsSize = sizeof(USB_CONFIGURATION_DESCRIPTOR);
	descriptors = (PUSB_CONFIGURATION_DESCRIPTOR)ExAllocatePool(NonPagedPool, DescriptorsSize+16);
	if( descriptors==NULL)
	{
		return STATUS_INSUFFICIENT_RESOURCES;
	}

	// Build the Get Descriptor URB
	UsbBuildGetDescriptorRequest(
		urb, UrbSize,
		USB_CONFIGURATION_DESCRIPTOR_TYPE, ConfigIndex, 0,
		descriptors, NULL, DescriptorsSize,
		NULL);

	// Call the USB driver
	NTSTATUS status = CallUSBDI( dx, urb);
	// Check statuses
	if( !NT_SUCCESS(status) || !USBD_SUCCESS( urb->UrbHeader.Status))
	{
		status = STATUS_UNSUCCESSFUL;
		goto fail;
	}

	// Reallocate memory for config descriptor and associated descriptors
	DescriptorsSize = descriptors->wTotalLength;
	ExFreePool(descriptors);
	descriptors = (PUSB_CONFIGURATION_DESCRIPTOR)ExAllocatePool(NonPagedPool, DescriptorsSize+16);
	if( descriptors==NULL)
	{
		return STATUS_INSUFFICIENT_RESOURCES;
	}

	// Build the Get Descriptor URB
	UsbBuildGetDescriptorRequest(
		urb, UrbSize, 
		USB_CONFIGURATION_DESCRIPTOR_TYPE, ConfigIndex, 0,
		descriptors, NULL, DescriptorsSize,
		NULL);

	// Call the USB driver
	status = CallUSBDI( dx, urb);
	// Check statuses
	if( !NT_SUCCESS(status) || !USBD_SUCCESS( urb->UrbHeader.Status))
	{
		
		status = STATUS_UNSUCCESSFUL;
	}
	else
	{
		// Remember count of bytes actually transferred
		DescriptorsSize = urb->UrbControlDescriptorRequest.TransferBufferLength;
	}
fail:
	ExFreePool(urb);
	return STATUS_SUCCESS;
}

/////////////////////////////////////////////////////////////////////////////
//	UsbSelectConfiguration:	Select first config, first interface ... should only have one
//TODO: LEAKS?!?
NTSTATUS UsbSelectConfiguration( IN PHID_DEVICE_EXTENSION dx)
{
	DEVICE_EXTENSION *myDx = MINIDRIVER_FROM_HID_DEVICE(dx);

	myDx->UsbInputPipe = myDx->UsbOutputPipe = NULL;

	// Get all first configuration descriptors
	PUSB_CONFIGURATION_DESCRIPTOR Descriptors = NULL;
	ULONG size;
	NTSTATUS status = UsbGetConfigurationDescriptors(dx, Descriptors, 0, size);
	if( !NT_SUCCESS(status))
	{
		FreeIfAllocated(Descriptors);
		return status;
	}

	// Crappy way to do this, just find the first config and first interface, we're not
	// picky
	PUSB_INTERFACE_DESCRIPTOR id = USBD_ParseConfigurationDescriptorEx(
		Descriptors, Descriptors,
		-1, -1,		// Do not search by InterfaceNumber or AlternateSetting
		-1, -1, -1);	
	if(id == NULL)
	{
		FreeIfAllocated(Descriptors);
		return STATUS_NO_SUCH_DEVICE;
	}

	// Build list of interfaces we are interested in
	USBD_INTERFACE_LIST_ENTRY ilist[2];
	ilist[0].InterfaceDescriptor = id;
	ilist[0].Interface = NULL;	// Will point to urb->UrbUsbSelectConfiguration.Interface
	ilist[1].InterfaceDescriptor = NULL;

	// Create select configuration URB
	PURB urb = USBD_CreateConfigurationRequestEx( Descriptors, ilist);

	// Call the USB driver
	status = CallUSBDI( dx, urb);
	// Check statuses
	if( !NT_SUCCESS(status) || !USBD_SUCCESS( urb->UrbHeader.Status))
	{
		status = STATUS_UNSUCCESSFUL;
	}
	else
	{
		// Select config worked
		myDx->UsbConfigurationHandle = urb->UrbSelectConfiguration.ConfigurationHandle;
		// Find pipe handle of first pipe, ie interrupt pipe that returns input HID reports
		PUSBD_INTERFACE_INFORMATION InterfaceInfo = &urb->UrbSelectConfiguration.Interface;
		if( InterfaceInfo->NumberOfPipes > 1)
		{
			PUSBD_PIPE_INFORMATION pi = &InterfaceInfo->Pipes[0];
			myDx->UsbInputPipe = pi->PipeHandle;
			//DebugPrint("maxpacket: %d, max transfer: %d", pi->MaximumPacketSize, pi->MaximumTransferSize);
			pi = &InterfaceInfo->Pipes[1];
			myDx->UsbOutputPipe = pi->PipeHandle;
			//DebugPrint("maxpacket: %d, max transfer: %d", pi->MaximumPacketSize, pi->MaximumTransferSize);
		}
		if( (myDx->UsbInputPipe == NULL) ||( myDx->UsbOutputPipe == NULL))
			status = STATUS_UNSUCCESSFUL;
	}

	FreeIfAllocated(urb);
	FreeIfAllocated(Descriptors);
	return status;
}

/////////////////////////////////////////////////////////////////////////////
//	UsbDeselectConfiguration:	Turn off device by selecting no configuration

NTSTATUS UsbDeselectConfiguration( IN PHID_DEVICE_EXTENSION dx)
{
	DEVICE_EXTENSION *myDx = MINIDRIVER_FROM_HID_DEVICE(dx);

	myDx->UsbOutputPipe = myDx->UsbInputPipe = NULL;

	// Allocate memory for URB
	USHORT UrbSize = sizeof(struct _URB_CONTROL_DESCRIPTOR_REQUEST);
	PURB urb = (PURB)ExAllocatePool(NonPagedPool, UrbSize);
	if( urb==NULL)
	{
		return STATUS_INSUFFICIENT_RESOURCES;
	}

	// Build select configuration URB with NULL Config descriptor
	UsbBuildSelectConfigurationRequest( urb, UrbSize, NULL);

	// Call the USB driver
	NTSTATUS status = CallUSBDI( dx, urb);
	// Check statuses
	if( !NT_SUCCESS(status) || !USBD_SUCCESS( urb->UrbHeader.Status))
	{
		status = STATUS_UNSUCCESSFUL;
	}

	ExFreePool(urb);
	return status;
}

/////////////////////////////////////////////////////////////////////////////
//	UsbDoInterruptTransfer:	Wait for 20-byte button report, or timeout

NTSTATUS UsbDoInterruptTransfer( IN PHID_DEVICE_EXTENSION dx, IN PVOID UserBuffer, ULONG& UserBufferSize, IN ULONG direction)
{
	DEVICE_EXTENSION *myDx = MINIDRIVER_FROM_HID_DEVICE(dx);
	NTSTATUS status = STATUS_SUCCESS;
	ULONG InputBufferSize = UserBufferSize;

	// Check we're selected
	if(direction == TRANSFER_IN)
	{
		if( myDx->UsbInputPipe == NULL)
			return STATUS_INVALID_HANDLE;
	}
	else if(direction == TRANSFER_OUT)
	{
		if(myDx->UsbOutputPipe == NULL)
			return STATUS_INVALID_HANDLE;
	}
	else
	{
		return STATUS_INVALID_HANDLE;
	}

	// Check input parameters
	if(direction == TRANSFER_IN)
	{
		// make sure there is a valid pointer, TODO: checking for 20 bytes could be a bad things, 32 tends to be max
		if( UserBuffer == NULL || InputBufferSize < 20)
			return STATUS_INVALID_PARAMETER;
	}
	else if(direction == TRANSFER_OUT)
	{
		if( UserBuffer == NULL || InputBufferSize < 6)
			return STATUS_INVALID_PARAMETER;
	}
	else
	{
		return STATUS_INVALID_PARAMETER;
	}

	// Allocate memory for URB
	USHORT UrbSize = sizeof(_URB_BULK_OR_INTERRUPT_TRANSFER);
	PURB urb = (PURB)ExAllocatePool(NonPagedPool, UrbSize);
	if(urb == NULL)
	{
		return STATUS_INSUFFICIENT_RESOURCES;
	}


	UsbBuildInterruptOrBulkTransferRequest(
			urb, UrbSize,
			((direction == TRANSFER_IN) ? myDx->UsbInputPipe : myDx->UsbOutputPipe),
			UserBuffer, NULL, InputBufferSize,
			((direction == TRANSFER_IN) ? (USBD_TRANSFER_DIRECTION_IN | USBD_SHORT_TRANSFER_OK) : 0),
			NULL);

		
	// Call the USB driver
	status = CallUSBDI(dx, urb);
	// Check statuses
	if( !NT_SUCCESS(status) || !USBD_SUCCESS( urb->UrbHeader.Status))
	{
		status = STATUS_UNSUCCESSFUL;
	}
		
	UserBufferSize = urb->UrbBulkOrInterruptTransfer.TransferBufferLength;
	ExFreePool(urb);

	return status;
}
