/*
 * MP3 Tag library. It includes an implementation of the ID3 tags and Lyrics3
 * tags as they are defined at www.id3.org
 *
 * Copyright (C) Eric Farng 2003
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.farng.mp3;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Stack;
import java.util.StringTokenizer;


/**
 * This is a holder class that contains static methods that I use in my
 * library. They may or may not be useful for anyone else extending the
 * library.
 *
 * @author Eric Farng
 * @version $Revision: 1.1 $
 */
public class TagUtilities {
    /** integer difference between ASCII 'A' and ASCII 'a' */
    private static final int uppercase;

    /**
     * Convenience <code>HashMap</code> to help fix capitilization of words. It
     * maps all words in <code>TagConstants.upperLowerCase</code> from all
     * lower case to their desired capitilziation.
     */
    public static HashMap capitalizationMap;

    static {
        uppercase         = 'A' - 'a';
        capitalizationMap = new HashMap();

        String   word;
        Iterator iterator = TagOptionSingleton.getInstance()
                            .getUpperLowerCaseWordListIterator();

        while (iterator.hasNext()) {
            word = (String) iterator.next();
            capitalizationMap.put(word.toLowerCase(),
                                  word);
        }
    }

    /**
     * Given an ID, get the ID3v2 frame description or the Lyrics3 field
     * description. This takes any kind of ID (four or three letter ID3v2 IDs,
     * and three letter Lyrics3 IDs)
     *
     * @param identifier frame identifier
     *
     * @return frame description
     */
    public static String getFrameDescription(String identifier) {
        String returnValue;
        String id = identifier.substring(0, 4);

        returnValue = (String) TagConstants.id3v2_4FrameIdToString.get(id);

        if (returnValue == null) {
            returnValue = (String) TagConstants.id3v2_3FrameIdToString.get(id);
        }

        if (returnValue == null) {
            returnValue = (String) TagConstants.id3v2_2FrameIdToString.get(id.substring(0,
                                                                                        3));
        }

        if (returnValue == null) {
            returnValue = (String) TagConstants.lyrics3v2FieldIdToString.get(id.substring(0,
                                                                                          3));
        }

        return returnValue;
    }

    /**
     * Returns true if the identifier is a valid ID3v2.2 frame identifier
     *
     * @param identifier string to test
     *
     * @return true if the identifier is a valid ID3v2.2 frame identifier
     */
    public static boolean isID3v2_2FrameIdentifier(String identifier) {
        return TagConstants.id3v2_2FrameIdToString.containsKey(identifier);
    }

    /**
     * Returns true if the identifier is a valid ID3v2.3 frame identifier
     *
     * @param identifier string to test
     *
     * @return true if the identifier is a valid ID3v2.3 frame identifier
     */
    public static boolean isID3v2_3FrameIdentifier(String identifier) {
        return TagConstants.id3v2_3FrameIdToString.containsKey(identifier);
    }

    /**
     * Returns true if the identifier is a valid ID3v2.4 frame identifier
     *
     * @param identifier string to test
     *
     * @return true if the identifier is a valid ID3v2.4 frame identifier
     */
    public static boolean isID3v2_4FrameIdentifier(String identifier) {
        return TagConstants.id3v2_4FrameIdToString.containsKey(identifier);
    }

    /**
     * Returns true if the identifier is a valid Lyrics3v2 frame identifier
     *
     * @param identifier string to test
     *
     * @return true if the identifier is a valid Lyrics3v2 frame identifier
     */
    public static boolean isLyrics3v2FieldIdentifier(String identifier) {
        return TagConstants.lyrics3v2FieldIdToString.containsKey(identifier);
    }

    /**
     * Given an object, try to return it as a <code>long</code>. This tries to
     * parse a string, and takes <code>Long, Short, Byte, Integer</code>
     * objects and gets their value. I would explicityly throw an exception
     * here, except this causes too many other methods to define it.
     *
     * @param value object to find long from.
     *
     * @return <code>long</code> value
     */
    static public long getWholeNumber(Object value) {
        long number;

        if (value instanceof String) {
            number = Long.parseLong((String) value);
        } else if (value instanceof Byte) {
            number = ((Byte) value).byteValue();
        } else if (value instanceof Short) {
            number = ((Short) value).shortValue();
        } else if (value instanceof Integer) {
            number = ((Integer) value).intValue();
        } else {
            number = ((Long) value).longValue();
        }

        return number;
    }

    /**
     * Add a timestamp string to a given string. This is used in the GUI and
     * I'm not sure why it is defined here.
     *
     * @param text textarea string to insert to
     * @param origPos current position of the cursor
     *
     * @return new string to use in the text area
     *
     * @todo-javadoc move this to a GUI class
     */
    public static String addTimeStampToTextArea(String text, int origPos) {
        /**
         * @todo-javadoc fix the case of adding time stamp to EOLN, EOLN, EOF
         *       (adding to last line which is just \n)
         */
        if (text.length() == 0) { // special empty case
            text = "[00:00]";
        } else {
            int i = origPos;
            i = Math.min(i, text.length() - 1); // if at end of whole string

            if (text.charAt(i) == '\n') {
                i--; // if at the end of line
            }

            for (; i > 0; i--) {
                if (text.charAt(i) == '\n') {
                    break;
                }
            }

            if (i == 0) { // if at very first character
                text = "[00:00]" + text;
            } else {
                i++;

                String before = text.substring(0, i);
                String after = text.substring(i);
                text = before + "[00:00]" + after;
            }
        }

        return text;
    }

    /**
     * DOCUMENT ME!
     *
     * @param filename DOCUMENT ME!
     * @param addition DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    static public String appendBeforeExtension(String filename, String addition) {
        int index = filename.lastIndexOf('.');

        if (index < 0) {
            return filename + addition;
        }

        return filename.substring(0, index) + addition +
               filename.substring(index);
    }

    /**
     * DOCUMENT ME!
     *
     * @param source DOCUMENT ME!
     * @param destination DOCUMENT ME!
     *
     * @throws FileNotFoundException DOCUMENT ME!
     * @throws IOException DOCUMENT ME!
     */
    static public void copyFile(File source, File destination)
                         throws FileNotFoundException, IOException {
        FileInputStream      fio;
        BufferedInputStream  bio;
        FileOutputStream     fos;
        BufferedOutputStream bos;
        byte[]               buffer;

        destination.delete();
        fio = new FileInputStream(source);
        bio = new BufferedInputStream(fio);

        fos = new FileOutputStream(destination);
        bos = new BufferedOutputStream(fos);

        buffer = new byte[1024];

        int b = bio.read(buffer);

        while (b != -1) {
            bos.write(buffer);
            b = bio.read(buffer);
        }

        bio.close();
        bos.flush();
        bos.close();
        fos.close();
        fio.close();
        fio    = null;
        bio    = null;
        fos    = null;
        bos    = null;
        buffer = null;
    }

    /**
     * return the index of the matching of parenthesis. This will match all
     * four parenthesis and enclosed parenthesis.
     *
     * @param str string to search
     * @param index index of string to start searching. This index should point
     *        to the opening parenthesis.
     *
     * @return index of the matching parenthesis. -1 is returned if none is
     *         found, or if the parenthesis are unbalanced.
     */
    static public int findMatchingParenthesis(String str, int index) {
        TagOptionSingleton option   = TagOptionSingleton.getInstance();
        Stack              stack    = new Stack();
        String             chString;
        String             open;
        int                length   = str.length();
        char               ch;

        for (int i = index; i < length; i++) {
            ch       = str.charAt(i);
            chString = ch + "";

            if (option.isOpenParenthesis(chString)) {
                stack.push(chString);
            }

            if (option.isCloseParenthesis(chString)) {
                if (stack.size() <= 0) {
                    return -1;
                } else {
                    open = (String) stack.pop();

                    if (option.getCloseParenthesis(open)
                            .equals(chString) == false) {
                        return -1;
                    }
                }
            }

            if (stack.size() <= 0) {
                return i;
            }
        }

        return -1;
    }

    /**
     * Find the first whole number that can be parsed from the string
     *
     * @param str string to search
     *
     * @return first whole number that can be parsed from the string
     */
    public static long findNumber(String str) {
        return findNumber(str, 0);
    }

    /**
     * Find the first whole number that can be parsed from the string
     *
     * @param str string to search
     * @param offset start seaching from this index
     *
     * @return first whole number that can be parsed from the string
     */
    public static long findNumber(String str, int offset) {
        int  i;
        int  j;
        long num = Long.MIN_VALUE;

        i = offset;

        while (i < str.length()) {
            if ((str.charAt(i) >= '0') && (str.charAt(i) <= '9')) {
                break;
            }

            i++;
        }

        j = i;

        while (j < str.length()) {
            if ((str.charAt(j) < '0') || (str.charAt(i) > '9')) {
                break;
            }

            j++;
        }

        if ((j <= str.length()) && (j > i)) {
            num = Long.parseLong(str.substring(i, j));
        }

        return num;
    }

    /**
     * Returns true if the string has matching parenthesis. This method matches
     * all four parenthesis and also enclosed parenthesis.
     *
     * @param str string to test
     *
     * @return true if the string has matching parenthesis
     */
    static public boolean matchingParenthesis(String str) {
        int  length = str.length();
        char ch;
        int  i = -1;

        for (i = 0; i < length; i++) {
            ch = str.charAt(i);

            if ((ch == '[') || (ch == '<') || (ch == '{') || (ch == '(')) {
                break;
            }
        }

        if (i < length) {
            return findMatchingParenthesis(str, i) >= 0;
        }

        return true;
    }

    /**
     * String formatting function to pad the given string with the given
     * character
     *
     * @param str string to pad
     * @param length total length of new string
     * @param ch character to pad the string with
     * @param padBefore if true, add the padding at the start of the string. if
     *        false, add the padding at the end of the string.
     *
     * @return new padded string.
     */
    static public String padString(String str, int length, char ch,
                                   boolean padBefore) {
        int strLength = str.length();

        if (strLength >= length) {
            return str;
        }

        char[] buffer = new char[length];

        int    next = 0;

        if (padBefore) {
            for (int i = 0; i < (length - strLength); i++) {
                buffer[next++] = ch;
            }
        }

        for (int i = 0; i < str.length(); i++) {
            buffer[next++] = str.charAt(i);
        }

        if (padBefore == false) {
            for (int i = 0; i < (length - strLength); i++) {
                buffer[next++] = ch;
            }
        }

        return new String(buffer);
    }

    /**
     * Replace the end of line character with the DOS end of line character.
     *
     * @param text string to search and replace
     *
     * @return replaced string
     */
    static public String replaceEOLNwithCRLF(String text) {
        String newText = "";

        int    oldPos = 0;
        int    newPos = text.indexOf('\n');

        while (newPos >= 0) {
            newText += (text.substring(oldPos, newPos) + TagConstants.CRLF);
            oldPos = ++newPos;
            newPos = text.indexOf('\n', oldPos);
        }

        newText += text.substring(oldPos);

        return newText;
    }

    /**
     * Search the <code>source</code> string for any occurance of
     * <code>oldString</code> and replaced them all with
     * <code>newString</code>. This searches for the entire word of old
     * string. A blank space is appended to the front and back of
     * <code>oldString</code>
     *
     * @param source DOCUMENT ME!
     * @param oldString DOCUMENT ME!
     * @param newString DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    static public String replaceWord(String source, String oldString,
                                     String newString) {
        oldString = " " + oldString + " ";
        newString = " " + newString + " ";

        StringBuffer str;
        int          index;
        int          length = oldString.length();
        index = source.indexOf(oldString);

        while (index >= 0) {
            str = new StringBuffer(source);
            str.replace(index, index + length, newString);
            source = str.toString();
            index  = source.indexOf(oldString);
        }

        return source;
    }

    /**
     * Remove all occurances of the given character from the string argument.
     *
     * @param str String to search
     * @param ch character to remove
     *
     * @return new String without the given charcter
     */
    static public String stripChar(String str, char ch) {
        if (str != null) {
            char[] buffer = new char[str.length()];
            int    next = 0;

            for (int i = 0; i < str.length(); i++) {
                if (str.charAt(i) != ch) {
                    buffer[next++] = str.charAt(i);
                }
            }

            return new String(buffer, 0, next);
        } else {
            return null;
        }
    }

    /**
     * Change the given string into sentence case. Sentence case has the first
     * words always capitalized. Any words in
     * <code>TagConstants.upperLowerCase</code> will be capitalized that way.
     * Any other words will be turned lower case.
     *
     * @param str String to modify
     * @param keepUppercase if true, keep a word if it is already all in
     *        uppercase
     *
     * @return new string in sentence case.
     */
    static public String toSentenceCase(String str, boolean keepUppercase) {
        if (str == null) {
            return null;
        }

        StringTokenizer tokenizer     = new StringTokenizer(str);
        String          token;
        char            ch;
        int             numberTokens  = tokenizer.countTokens();
        int             countedTokens = 0;
        StringBuffer    newString     = new StringBuffer();

        // Capitalize first word of all sentences.
        if (tokenizer.hasMoreTokens()) {
            token = tokenizer.nextToken();
            newString.append(capitalizeWord(token, keepUppercase));
            newString.append(' ');
            countedTokens++;
        }

        // go through all remainder tokens
        while (tokenizer.hasMoreTokens() && (countedTokens < numberTokens)) {
            token = tokenizer.nextToken();
            countedTokens++;

            if (capitalizationMap.containsKey(token.toLowerCase())) {
                newString.append(capitalizationMap.get(token.toLowerCase()));
            } else if (token.toUpperCase()
                           .equals(token) && keepUppercase) {
                newString.append(token);
            } else {
                newString.append(token.toLowerCase());
            }

            newString.append(' ');
        }

        // remove trailing space
        if (newString.length() > 0) {
            newString.deleteCharAt(newString.length() - 1);
        }

        return newString.toString();
    }

    /**
     * Change the given string to title case. The first and last words of the
     * string are always capitilized. Any words in
     * <code>TagConstants.upperLowerCase</code> will be capitalized that way.
     * Any other words will be capitalized.
     *
     * @param str String to modify
     * @param keepUppercase if true, keep a word if it is already all in
     *        uppercase
     *
     * @return new capitlized string.
     */
    static public String toTitleCase(String str, boolean keepUppercase) {
        if (str == null) {
            return null;
        }

        StringTokenizer tokenizer     = new StringTokenizer(str);
        String          token;
        char            ch;
        int             numberTokens  = tokenizer.countTokens();
        int             countedTokens = 0;
        StringBuffer    newString     = new StringBuffer();

        // Capitalize first word of all titles.
        if (tokenizer.hasMoreTokens()) {
            token = tokenizer.nextToken();
            newString.append(capitalizeWord(token, keepUppercase));
            newString.append(' ');
            countedTokens++;
        }

        // go through all remainder tokens except last
        while (tokenizer.hasMoreTokens() &&
                   (countedTokens < (numberTokens - 1))) {
            token = tokenizer.nextToken();
            countedTokens++;

            if (capitalizationMap.containsKey(token.toLowerCase())) {
                newString.append(capitalizationMap.get(token.toLowerCase()));
            } else {
                newString.append(capitalizeWord(token, keepUppercase));
            }

            newString.append(' ');
        }

        // Capitalize last word of all titles.
        if (tokenizer.hasMoreTokens()) {
            token = tokenizer.nextToken();
            newString.append(capitalizeWord(token, keepUppercase));
            newString.append(' ');
            countedTokens++;
        }

        // remove trailing space
        if (newString.length() > 0) {
            newString.deleteCharAt(newString.length() - 1);
        }

        return newString.toString();
    }

    /**
     * truncate a string if it longer than the argument
     *
     * @param str String to truncate
     * @param len maximum desired length of new string
     *
     * @return DOCUMENT ME!
     */
    public static String truncate(String str, int len) {
        if (str == null) {
            return null;
        }

        if (str.length() > len) {
            return str.substring(0, len);
        } else {
            return str;
        }
    }

    /**
     * Capitalize the word with the first letter upper case and all others
     * lower case.
     *
     * @param word word to capitalize.
     * @param keepUppercase if true, keep a word if it is already all in
     *        uppercase
     *
     * @return new capitalized word.
     */
    static private StringBuffer capitalizeWord(String word,
                                               boolean keepUppercase) {
        StringBuffer wordBuffer = new StringBuffer();
        int          index = 0;

        if (word.toUpperCase()
                .equals(word) && keepUppercase) {
            wordBuffer.append(word);
        } else {
            word = word.toLowerCase();

            int  len = word.length();
            char ch;
            ch = word.charAt(index);

            while (((ch < 'a') || (ch > 'z')) && (index < (len - 1))) {
                ch = word.charAt(++index);
            }

            if (index < len) {
                wordBuffer.append(word.substring(0, index));
                wordBuffer.append((char) (ch + uppercase));
                wordBuffer.append(word.substring(index + 1));
            } else {
                wordBuffer.append(word);
            }
        }

        return wordBuffer;
    }
}