/*
 * MP3 Tag library. It includes an implementation of the ID3 tags and Lyrics3
 * tags as they are defined at www.id3.org
 *
 * Copyright (C) Eric Farng 2003
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.farng.mp3.id3;

import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.Iterator;
import java.util.LinkedList;

import org.farng.mp3.InvalidTagException;
import org.farng.mp3.TagUtilities;
import org.farng.mp3.object.ObjectID3v2LyricLine;
import org.farng.mp3.object.ObjectLyrics3Line;
import org.farng.mp3.object.ObjectLyrics3TimeStamp;


/**
 * Title: Description: Copyright:    Copyright (c) 2002 Company:
 *
 * @author Eric Farng
 * @version $Revision: 1.1 $
 */
public class FrameBodySYLT
    extends AbstractID3v2FrameBody {
    /** DOCUMENT ME! */
    LinkedList lines = new LinkedList();

    /** DOCUMENT ME! */
    String description = "";

    /** DOCUMENT ME! */
    String language = "";

    /** DOCUMENT ME! */
    byte contentType = 0;

    /**
     * DOCUMENT ME!
     *
     * @todo-javadoc write equals functin for this too
     */
    byte textEncoding = 0;

    /** DOCUMENT ME! */
    byte timeStampFormat = 0;

    /**
     * Creates a new FrameBodySYLT object.
     */
    public FrameBodySYLT() {
        //        this.textEncoding = 0;
        //        this.language = "";
        //        this.timeStampFormat = 0;
        //        this.contentType = 0;
        //        this.description = "";
    }

    /**
     * Creates a new FrameBodySYLT object.
     *
     * @param textEncoding DOCUMENT ME!
     * @param language DOCUMENT ME!
     * @param timeStampFormat DOCUMENT ME!
     * @param contentType DOCUMENT ME!
     * @param description DOCUMENT ME!
     * @param is6ByteHeader DOCUMENT ME!
     */
    public FrameBodySYLT(byte textEncoding, String language,
                         byte timeStampFormat, byte contentType,
                         String description, boolean is6ByteHeader) {
        this.textEncoding    = textEncoding;
        this.language        = language;
        this.timeStampFormat = timeStampFormat;
        this.contentType     = contentType;
        this.description     = description;
        this.is6ByteHeader   = is6ByteHeader;
    }

    /**
     * Creates a new FrameBodySYLT object.
     *
     * @param file DOCUMENT ME!
     * @param is6ByteHeader DOCUMENT ME!
     *
     * @throws IOException DOCUMENT ME!
     * @throws InvalidTagException DOCUMENT ME!
     */
    public FrameBodySYLT(java.io.RandomAccessFile file, boolean is6ByteHeader)
                  throws IOException, InvalidTagException {
        this.is6ByteHeader = is6ByteHeader;
        this.read(file);
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public byte getContentType() {
        return contentType;
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public String getDescription() {
        return description;
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public String getIdentifier() {
        return "SYLT";
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public String getLanguage() {
        return language;
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public int getSize() {
        int size;

        size = 1 + 3 + 1 + 1 + description.length();

        for (int i = 0; i < lines.size(); i++) {
            size += ((ObjectID3v2LyricLine) lines.get(i)).getSize();
        }

        return size;
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public byte getTextEncoding() {
        return textEncoding;
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public byte getTimeStampFormat() {
        return timeStampFormat;
    }

    /**
     * DOCUMENT ME!
     *
     * @param timeStamp DOCUMENT ME!
     * @param text DOCUMENT ME!
     */
    public void addLyric(int timeStamp, String text) {
        lines.add(new ObjectID3v2LyricLine(timeStamp, text));
    }

    /**
     * DOCUMENT ME!
     *
     * @param line DOCUMENT ME!
     */
    public void addLyric(ObjectLyrics3Line line) {
        Iterator               iterator  = line.getTimeStamp();
        ObjectLyrics3TimeStamp timeStamp;
        String                 lyric = line.getLyric();
        int                    time;

        if (iterator.hasNext() == false) {
            // no time stamp, give it 0
            time = 0;
            lines.add(new ObjectID3v2LyricLine(time, lyric));
        } else {
            while (iterator.hasNext()) {
                timeStamp = (ObjectLyrics3TimeStamp) iterator.next();
                time      = (timeStamp.getMinute() * 60) +
                            timeStamp.getSecond(); // seconds
                time *= 1000; // milliseconds
                lines.add(new ObjectID3v2LyricLine(time, lyric));
            }
        }
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public Iterator iterator() {
        return lines.iterator();
    }

    /**
     * DOCUMENT ME!
     *
     * @param file DOCUMENT ME!
     *
     * @throws IOException DOCUMENT ME!
     * @throws InvalidTagException DOCUMENT ME!
     */
    public void read(RandomAccessFile file)
              throws IOException, InvalidTagException {
        int    size;
        int    delim;
        int    offset = 0;
        byte[] buffer;
        String str;

        size   = readHeader(file);
        buffer = new byte[size];
        file.read(buffer);
        str = new String(buffer);

        this.textEncoding = buffer[offset++];

        this.language = str.substring(offset, offset + 3);
        offset += 3;

        this.timeStampFormat = buffer[offset++];
        this.contentType     = buffer[offset++];

        delim            = str.indexOf(0, offset);
        this.description = str.substring(offset, delim);
        offset           = delim + 1;

        byte[] data      = new byte[size - offset];
        System.arraycopy(buffer, offset, data, 0, size - offset);
        readByteArray(data);
    }

    /**
     * DOCUMENT ME!
     *
     * @param arr DOCUMENT ME!
     */
    public void readByteArray(byte[] arr) {
        int    offset = 0;
        int    delim;
        byte[] line;

        for (int i = 0; i < arr.length; i++) {
            if (arr[i] == 0) {
                delim = i;
                line  = new byte[offset - delim + 4];
                System.arraycopy(arr, offset, line, 0, offset - delim + 4);
                lines.add(new ObjectID3v2LyricLine(line));
                i += 4;
                offset += 4;
                delim = offset;
            }
        }
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public String toString() {
        String str;
        str = getIdentifier() + " " + textEncoding + " " + language + " " +
              timeStampFormat + " " + contentType + " " + description;

        for (int i = 0; i < lines.size(); i++) {
            str += ((ObjectID3v2LyricLine) lines.get(i)).toString();
        }

        return str;
    }

    /**
     * DOCUMENT ME!
     *
     * @param file DOCUMENT ME!
     *
     * @throws IOException DOCUMENT ME!
     */
    public void write(RandomAccessFile file)
               throws IOException {
        byte[] buffer;
        int    offset = 0;

        writeHeader(file,
                    this.getSize());

        buffer = new byte[this.getSize()];

        buffer[offset++] = this.textEncoding; // text encoding;

        language = TagUtilities.truncate(language, 3);

        for (int i = 0; i < language.length(); i++) {
            buffer[i + offset] = (byte) language.charAt(i);
        }

        offset += language.length();

        buffer[offset++] = this.timeStampFormat;
        buffer[offset++] = this.contentType;

        for (int i = 0; i < description.length(); i++) {
            buffer[i + offset] = (byte) description.charAt(i);
        }

        offset += description.length();
        buffer[offset++] = 0; // null character

        System.arraycopy(writeByteArray(),
                         0,
                         buffer,
                         offset,
                         buffer.length - offset);

        file.write(buffer);
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public byte[] writeByteArray() {
        byte[]               arr;
        ObjectID3v2LyricLine line   = null;
        int                  offset = 0;
        int                  size   = 0;

        for (int i = 0; i < lines.size(); i++) {
            line = (ObjectID3v2LyricLine) lines.get(i);
            size += line.getSize();
        }

        arr = new byte[size];

        for (int i = 0; i < lines.size(); i++) {
            line = (ObjectID3v2LyricLine) lines.get(i);
        }

        System.arraycopy(line.writeByteArray(),
                         0,
                         arr,
                         offset,
                         line.getSize());
        offset += line.getSize();

        return arr;
    }
}