#!/usr/bin/perl

#                    XNSCast v0.4.1 by Tusse/nonSense (c) 2003
#                        a Shoutcast to XNS stream relayer
#                                  Be inspired!

# Licence
# You may with XNSCast as you wish, but any derivative work *must* recognize
# Tusse/nonSense as the original author. If you reimplement this concept in a
# different programming language than perl, some credit would be nice, but it
# is not required. If you base something on this code that you plan to sell
# for money (you must be mad), I expect to be getting a free copy somehow.

# About
# XNSCast is an experimental Shoutcast/Icecast stream relayer for use with
# XBMP. It works by presenting fake files to XBMP, and when these files are
# requested, a connection to a Shoutcast/Icecast-server is made, and data
# from the Shoutcast/Icecast stream is relayed to XBMP, imitating regular
# Relax behaviour.

# Please note that XNSCast is mostly a nasty perl kludge, and there's a big
# chance that it won't work for you. Because of the rather lacking protocol
# specification, I had to do some packet sniffing in order to find out what
# was really going on at times... The intention of this was mostly to get
# some working concept code "out there" to push those who implement real XNS
# servers to get working on stream relaying :-)

# Also note that XNSCast was developed on Win32. I have no idea wether or not
# this works with Linux (shouldn't be too hard making it work, though).

# Check out readme.txt and xnscast.ini for some more info


# Shortlist of things that could be added or use some fixing...

# TODO: Swap out the current buffering system with a real one that doesn't
#       silly hacks to not consume all CPU
# TODO: Proper thread communication instead of lame signalling variables
# TODO: The fake ID3-tag doesn't seem to be working
# TODO: Do something at all for RCAT command
# TODO: Parse standard .pls files to find shoutcast server parameters
# TODO: Strip out and parse Shoutcast info from the shoutcast stream

#############################################################################
# INIT                                                                      #
#############################################################################

use strict;
use Socket;
use threads;
use threads::shared;

# Autoflush
$| = 1;

# About me
my $APP     = "XNSCast";
my $VERSION = "v0.4 (31-Jan-2003)";
my $AUTHOR  = "Tusse / nonSense";
my $INI     = "xnscast.ini";

# Misc constants
my $TRUE  = 1;
my $FALSE = 0;
my $KEEP = 1;
my $DROP = 0;

# Read values from ini's
my %config = ReadIni( $INI );

# Xstream filesystem variables and other related thingies
my $xs_root = $config{'streams'}{'root'};
my $xs_cwd  = $xs_root;
my $xs_cfile = "";
my $xs_file = "";
my $xs_offset = 0;

# Somewhere to stash downloaded Shoutcast data some signalling
my $sbuffer = "";   share( $sbuffer );
my $ssignal = "";   share( $ssignal );
my $sstatus = "";   share( $sstatus );
my $sposition = ""; share( $sposition );

# Flush debug if applicable
if ( $config{'debug'}{'enable'} eq "yes" &&
     $config{'debug'}{'tofile'} eq "yes" &&
	 $config{'debug'}{'tofile'} eq "yes" &&
	 open DEBUG, ">$config{'debug'}{'file'}" ) {

	print DEBUG "";
	close DEBUG;
}

#############################################################################
# MAIN                                                                      #
#############################################################################

# Hello
print "$APP $VERSION by $AUTHOR\n\n";

# Start serving -- taken straight from the Camel book..
SetupServer( $config{'server'}{'port'} );
my $paddr;
$SIG{CHLD} = \&REAPER;

# Handle incoming connections
for ( ; $paddr = accept( Client, Server ); close Client ) {

	my( $port, $iaddr ) = sockaddr_in( $paddr );
	my $name            = gethostbyaddr( $iaddr, AF_INET );
	my $ip              = inet_ntoa( $iaddr );

	Debug( "Connection from $name [$ip] at port $port", "!!", 2 );

	# Authorize
	if ( Authorize( $ip ) ) {

		Debug( "Accepted: Authorization OK", "!!", 2 );

		my $nmessage = "";

		# Handshake
		send(Client, "HELLO XBOX!", undef);
		Debug( "HELLO XBOX!", ">>", 1);
		
		recv(Client, $nmessage, 1024, undef);
		Debug( $nmessage, "<<", 0);

		# Check if handshake is ok before continuing
		if ( $nmessage =~ s/HELLO XSTREAM 6.0\r\n// ) {

			Debug( "HELLO XSTREAM 6.0", "<<", 1);
			Debug( "Accepted: Handshake OK", "!!", 2 );
			
			# Some commands expect the connection to be dropped after execution.. so we'll have
			# them return their wishes into this variable, then check it at the end of processing
			my $action = "";

			while( $TRUE ) {

				if ( length( $nmessage ) > 0 ) {
					Debug( $nmessage, "<<", 1);
					my %packet = DecodeMessage( $nmessage );

					# Look at the message and decide what to do
					SWITCH: {
						if ( $packet{'head'} eq "*CAT" ) { $action = DoCat(  $packet{'data'} ); last SWITCH; }
						if ( $packet{'head'} eq "RCAT" ) { $action = DoRcat( $packet{'data'} ); last SWITCH; }

						if ( $packet{'head'} eq "OPEN" ) { $action = DoOpen( $packet{'data'} ); last SWITCH; }
						if ( $packet{'head'} eq "CLSE" ) { $action = DoClse( $packet{'data'} ); last SWITCH; }
						if ( $packet{'head'} eq "READ" ) { $action = DoRead( $packet{'data'} ); last SWITCH; }
						if ( $packet{'head'} eq "TELL" ) { $action = DoTell( $packet{'data'} ); last SWITCH; }
					}
				}

				# Read next command from xbox or drop out			
				if ( $action ) {
					recv(Client, $nmessage, 1024, undef);
				}
				else {
					last;
				}
			}
		}
		else {
			Debug( "Rejected: Invalid handshake", "!!", 4 );
		}
	}
	else {
		Debug( "Rejected: IP $ip not in list of allowed IPs", "!!", 4 );
	}
}

#############################################################################
# SUBS                                                                      #
#############################################################################

# Utility routines
sub ReadIni($) {

	my $inifile = shift @_;

	my %values  = {};
	my $section = "";
	my $key     = "";
	my $value   = "";
	my $line    = "";

	open INI, "<$inifile";
	while ($line = <INI>) {
		chomp $line;

		if ($line =~ m/^\[(.*)\]$/) {
			$section = $1;
		}
		else {
			unless (($line =~ m/^ {0,}$/) || ($line =~ m/^\#/) || ($line =~ m/^\;/)) {
				($key, $value) = split / {0,}= {0,}/, $line, 2;
				$values{$section}{$key} = $value;
			}
		}
	}
	close INI;

	return %values;
}

sub SetupServer($) {
	my $port = shift @_;
	my $proto = getprotobyname('tcp');

	socket( Server, PF_INET, SOCK_STREAM, $proto )                 or die "socket: $!";
	setsockopt( Server, SOL_SOCKET, SO_REUSEADDR, 1 )              or die "setsockopt: $!";
	bind( Server, sockaddr_in( $port, INADDR_ANY))                 or die "bind: $1";
	listen( Server, SOMAXCONN )                                    or die "listen: $!";

	Debug( "Server started on port $port", "!!", 5 );
}

sub Authorize($) {

	my $ip = shift @_;

	if ( $config{'ipfilter'}{'enable'} eq "yes" ) {

		if ( $config{'ipfilter'}{'iplist'} =~ m/$ip/ ) {

			return $TRUE;
		}
		else {
			return $FALSE;
		}
	}
	else {
		return $TRUE;
	}
}

sub TCPSend($) {
	my $data = shift @_;

	select( Client );
	$| = 1;
	print Client $data;
	$| = 0;
	select(STDOUT);
}

sub Debug($$$) {

	my $message = shift @_;
	my $type    = shift @_;
	my $level   = shift @_;
	
	if ( $config{'debug'}{'enable'} eq "yes" ) {

		if ( $config{'debug'}{'tofile'} eq "yes" &&
			 $level >= $config{'debug'}{'filedetail'} &&
			 open DEBUG, ">>$config{'debug'}{'file'}" ) {

			print DEBUG BuildDebugMessage( $message, $type, $level ) . "\n";
			close DEBUG;
		}

		if ( $config{'debug'}{'toscreen'} eq "yes" &&
			 $level >= $config{'debug'}{'screendetail'} ) {

			print BuildDebugMessage( $message, $type, $level ) . "\n";
		}
	}
}

sub BuildDebugMessage($$$) {
	my $message = shift @_;
	my $type    = shift @_;
	my $level   = shift @_;

	my $timestamp = BuildDebugTimestamp();
	$message =~ s/\n//g;

	return $timestamp . " " . $level . " " . $type . " " . $message;
}

sub BuildDebugTimestamp() {

	my ($sec, $min, $hour, undef, undef, undef, undef, undef, undef) = localtime(time);

	if (length($sec)  < 2) { $sec  = "0$sec"  }
	if (length($min)  < 2) { $min  = "0$min"  }
	if (length($hour) < 2) { $hour = "0$hour" }

	return "[" . $hour . ":" . $min . ":" . $sec . "]";
}

sub DecodeMessage($) {

	my $packet = shift @_;
	my %message = {};

	$packet =~ s/\r//g;
	$packet =~ s/\n//g;

	if ( $packet =~ m/,/ ) {
		( $message{'head'}, $message{'data'} ) = split( /,/, $packet, 2);

		# Remove trash that might encapsulate our data..
		$message{'data'} =~ s/\\\$\$DISPNAME.*//;
		$message{'data'} =~ s/^\///;
	}
	else {
		$message{'head'} = $packet;
	}

	return %message;
}


# Routines for handling Xstream commands
sub DoCat($) {

	Debug( "Received *CAT", "!!", 0 );

	my $data = shift @_;

	# Data payload can be a "BACK", a directory identifier or empty 
	if ( $data eq "BACK" ) {

		# Move up one catalog
		$xs_cwd =~ s/\/(.*?)$//;
		
		if ( $xs_cwd eq "" ) {
			$xs_cwd = $xs_root;
		}

	}
	elsif ( $data ne "" ) {

		# Move into specified directory if it exists ( or the root dir if $ROOT$ is in the string )
		if ( $data =~ m/\$ROOT\$XNSCast/ ) {

			$xs_cwd = $xs_root;

		}
		elsif ( -d $xs_cwd . "/" . $data ) {

			$xs_cwd .= "/" . $data;
		}
	}

	# Return listing for current directory
	my $sharelist = BuildSharelist();
	send( Client, pack( "A32", length( $sharelist) . " OK" ), undef );
	send( Client, $sharelist, undef );

	# Always drop connection after this command
	return $DROP;
}

sub BuildSharelist() {

	my $sharelist = "";
	
	if ( opendir( DIR, $xs_cwd ) ) {

		my @dir = sort grep(!/^\.{1,2}$/,readdir( DIR ) );
		closedir DIR;

		my @files = ();
		my @dirs  = ();

		foreach my $dentry ( @dir ) {
			if ( -d $xs_cwd . "/" . $dentry ) {
				push @dirs, $dentry;
			}

			# We only want files ending in .ini, but we need to hide .ini from XBMP
			if ( -f $xs_cwd . "/" . $dentry && $dentry =~ m/\.ini$/ ) {
				$dentry =~ s/\.ini//;
				push @files, $dentry;
			}
		}

		$sharelist .= "<SHARES>\n";
		foreach my $dir ( @dirs ) {
			#$sharelist .= "<ITEM><ATTRIB>16</ATTRIB><PATH>" . $dir . "\\\$\$DISPNAME\$\$\\" . $dir . "</PATH></ITEM>\n";
			$sharelist .= "<ITEM><ATTRIB>16</ATTRIB><PATH>" . $dir . "\\\$\$DISPNAME\$\$\\" . $dir . "</PATH></ITEM>\n";
		}
		foreach my $file ( @files ) {
			#$sharelist .= "<ITEM><ATTRIB>128</ATTRIB><PATH>" . $file . "\\\$\$DISPNAME\$\$\\" . $file . "</PATH></ITEM>\n";
			$sharelist .= "<ITEM><ATTRIB>128</ATTRIB><PATH>" . $file . "\\\$\$DISPNAME\$\$\\" . $file . "</PATH></ITEM>\n";
		}
		$sharelist .= "</SHARES>\n";
	}

	return $sharelist;
}

sub DoRcat($) {

	# TODO: Implement this function

	Debug( "Received RCAT", "!!", 0 );

	# No info about this command
	my $data = shift @_;

	# Always drop after this command
	return $DROP;
}

sub DoOpen($) {

	Debug( "Received OPEN", "!!", 0 );

	# Data payload is always a file (shoutcast stream) name
	my $data = shift @_;

	# Ignore requests for files that don't end in .mp3 or .ogg
	if ( $data =~ m/\.mp3$/ or $data =~ m/\.ogg$/ ) {

		# See if we can open the ini
		if ( open STREAMINI, "<${xs_cwd}/${data}.ini" ) {

			# Read values
			my %streamini = ReadIni( "${xs_cwd}/${data}.ini" );
			Debug( "Read stream info from ${xs_cwd}/${data}.ini", "!!", 1 );

			# Signal any current shoutcast thread to die, in case there is one
			$ssignal = "die";
			$xs_file = 0;
			$xs_cfile = "";

			# Also reset some stuff
			$xs_offset = 0;
			$sbuffer = "";

			# Spawn Shoutcast process and set thread id in $xs_file
			$ssignal = "live";
			$sstatus  = "alive";
			my $thread = threads->new(\&ShoutcastThread, ( $streamini{'stream'}{'host'}, $streamini{'stream'}{'port'}, $streamini{'stream'}{'file'} ) );
			$xs_file = $thread->tid;
			$xs_cfile = $data;
			$thread->detach;
	
			# Continue if thread-creation was successful
			if ( $xs_file ) {

				# Put this thread to sleep a little bit to allow for the shoutcast buffer to
				# fill up a little bit before the first read.
				Debug( "Prebuffering " . $config{'streams'}{'prebuffer'} . " bytes", "!!", 3 );
				while ( length( $sbuffer ) < $config{'streams'}{'prebuffer'} && $sstatus ne "dead" ) {
					sleep(1);
					Debug( "Buffer is now " . length( $sbuffer ) . " bytes", "!!", 2 );
				}
				Debug( "Done buffering, got " . length($sbuffer) . " bytes", "!!", 2 );

				if ( $sstatus eq "dead" ) {

					send( Client, pack("A32","-1 ERROR PERMISSION DENIED"), undef ); 
					Debug( "Probably unable to fetch from stream defined in '${data}.ini'", "!!", 5 );
					return $DROP;
				}
				else {
					send( Client, pack("A32", $config{'streams'}{'fakesize'} . " OK"), undef );
					$data =~ s/\.(.*?)$//;
					Debug( "Relaying '$data'", "!!", 4 );
					return $KEEP;
				}
			}
			else {

				send( Client, pack("A32","-1 ERROR PERMISSION DENIED"), undef ); 
				return $DROP;
			}
		}
		else {
			send( Client, pack("A32","-1 ERROR PERMISSION DENIED"), undef ); 
			return $DROP;
		}
	}
	else {
		send( Client, pack("A32","-1 ERROR PERMISSION DENIED"), undef ); 
		return $DROP;
	}
}

sub DoClse($) {

	Debug( "Received CLSE", "!!", 0 );

	# Close Shoutcast thread referenced in $xs_file
	# if it's open, that is
	if ( $xs_file != 0 ) {
		
		# Kill current stream
		$ssignal = "die";
		$xs_offset = 0;
		$sbuffer = "";
		$xs_file = 0;
		$xs_cfile = "";

		send( Client, pack("A32","0 OK"), undef );
		Debug( "Relaying stopped, stream closed", "!!", 4 );
	}
	else {
		send( Client, pack("A32","-1 FILE NOT OPEN"), undef ); 
	}

	# Always drop..
	return $DROP;
}

sub DoRead($) {

	Debug( "Received READ", "!!", 0 );

	# Data payload is always "OFFSET,LENGTH"
	my $data = shift @_;
	my ($offset, $length) = split( /,/, $data );

	# XBMP might ask for the last 1280 bytes, but we don't have those do we... We'll just send the ID3 tag untill it gives up.
	# TODO: Is this ID3 tag correct?
	if ( $config{'streams'}{'fakesize'} - $offset <= 1280 ) {

		send( Client, pack( "A32", "128 OK" ), undef );
		send( Client, pack ( "A3Z30Z30Z30A4Z30A", "TAG", "XNSCast: $xs_cfile", $xs_cwd, "Shoutcast", "2003", $APP . " by " . $AUTHOR, chr(255)), undef );

		Debug( "Sent fake ID3-tag", "!!", 2 );

		return $KEEP;
	}

	# Try to stall XBMP a little bit if it is exhausting the buffer..
	my $buffermax = length( $sbuffer );
	if ( $offset + $config{'streams'}{'bufferslack'} + $length >= $xs_offset+$buffermax ) {

		Debug( sprintf("Stalling: %d -> %d requested, buffer only has -> %d", $offset, $offset+$length, $xs_offset+$buffermax ), "!!", 3 );
		sleep( $config{'streams'}{'buffersleep'} );
	}
			
	# Pull data from the Shoutcast buffer (remember to subtract internal offset)
	my $sdata = substr($sbuffer, $offset-$xs_offset, $length );
	my $datalength = length( $sdata );

	# Send it off to XBMP
	send( Client, pack( "A32", $datalength . " OK" ), undef );
	send( Client, $sdata, undef );
	if ( $datalength < $length ) {

		Debug( sprintf("Low buffer: sent %d bytes, but %d were requested", $datalength, $length), "!!", 4 );		
	}

	# Let the shoutcast thread know how much we've sent out
	$sposition = $offset - $xs_offset; # + $datalength;
	
	# Simple (and stupid (and probably buggy)) cache size control
	# TODO: Fix this
	Debug( sprintf( "Last sent: %d   Buffer max: %d", $sposition, $buffermax), "!!", 2 );

	if ( $sposition >= $config{'streams'}{'cachedrop'} ) {
	
		Debug( "Dropped old data from buffer", "!!", 2 );
		$xs_offset += $config{'streams'}{'cachedrop'};
		substr( $sbuffer, 0, $config{'streams'}{'cachedrop'}, "" );
	}

	# Always keep..
	return $KEEP;
}

sub DoTell($) {

	Debug( "Received TELL", "!!", 0 );

	# Return the last sent byte transposed to XBMP's numbering
	send( Client, pack("A32", $sposition+$xs_offset . " OK"), undef );

	# Always keep
	return $KEEP;
}

# Shoutcast stream thread
sub ShoutcastThread($$$) {

	# REMEMBER THAT THIS THREAD ONLY HAS ITS OWN COPIES OF ALL VARIABLES,
	# EXCEPT THOSE THAT ARE EXPLICITLY SHARED BETWEEN THREADS

	# This thread is now alive
	$sstatus = "alive";

	# Clear the buffer and internal offset
	$xs_offset = 0;
	$sbuffer = "";

	my $shouthost = shift @_;
	my $shoutport = shift @_;
	my $shoutfile = shift @_;

	# Setup client connection
	my ($iaddr, $paddr, $proto, $line );

	if ($shoutport =~ /\D/) { $shoutport = getservbyname( $shoutport, 'tcp' ) } die "No port" unless $shoutport;
	$iaddr  = inet_aton($shouthost)                                          or die "no host: $shouthost";
	$paddr  = sockaddr_in( $shoutport, $iaddr );
	$proto  = getprotobyname('tcp');

	socket( SOCK, PF_INET, SOCK_STREAM, $proto )                             or die "socket: $!";
	connect( SOCK, $paddr ) or die "connect: $!";

	my $request = "GET $shoutfile HTTP/1.0\r\nHost: $shouthost\r\n\r\n";

	# CISC's trick
	select(SOCK); $| = 1;
	print SOCK $request;
	$| = 0;	select(STDOUT);

	# Loop and fetch data
	my $line = "";

	# This is a little backwards, but it saves us doing the test on
	# micorsleep every loop.. Will save some CPU
	if ( $config{'streams'}{'microsleep'} eq "yes" ) {

		my $microsleepammount = $config{'streams'}{'microsleepammount'};
		while( $ssignal ne "die" and defined( $line = <SOCK> ) ) {

			# TODO: Look at the data and extract/strip Shoutcast stream info

			# Read in a block of data
			$sbuffer .= $line;

			# Wait for a sec if the buffer is full
			# TODO: This could be way better..
			if ( length( $sbuffer ) >= $config{'streams'}{'cachemax'} ) {
				sleep( $config{'streams'}{'buffersleep'} );
			}

			# Sleep for a tiny little while to try to reduce CPU load
			select( undef, undef, undef, $microsleepammount );
		}
	}
	else {
		while( $ssignal ne "die" and defined( $line = <SOCK> ) ) {

			# TODO: Look at the data and extract/strip Shoutcast stream info

			# Read in a block of data
			$sbuffer .= $line;

			# Wait for a sec if the buffer is full
			# TODO: This could be way better..
			if ( length( $sbuffer ) >= $config{'streams'}{'cachemax'} ) {
				sleep( $config{'streams'}{'buffersleep'} );
			}

			# Don't hog the CPU all the time...
			threads->self->yield();
		}
	}
	
	close SOCK                                                               or die "close: $!";

	# This thread is now dead
	$sstatus = "dead";
}