package kr.gui;

import java.io.IOException;
import java.util.List;

import javax.management.Notification;
import javax.sound.midi.InvalidMidiDataException;
import javax.sound.midi.MidiSystem;
import javax.sound.midi.MidiUnavailableException;
import javax.sound.midi.Receiver;
import javax.sound.midi.ShortMessage;
import javax.sound.midi.Synthesizer;
import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.AudioInputStream;
import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.Control;
import javax.sound.sampled.DataLine;
import javax.sound.sampled.FloatControl;
import javax.sound.sampled.LineUnavailableException;
import javax.sound.sampled.SourceDataLine;
import javax.sound.sampled.AudioFormat.Encoding;

import kr.util.Util;

import kr.AudioModel;
import kr.TimeKeeper;
import kr.util.Listener;
import kr.util.ListenerManager;
import kr.util.NotificationManager;
import kr.util.audio.AudioClip;
import kr.util.audio.MidiAudioClip;

/**
 * Plays notes in as close to real time as possible. 
 * <p>
 * <b>WARNING:</b> When changing the aspects of the audio clip, the write buffer size or the current position while playing
 * the changes should be synchronized against this object. However, the method getWriteBufferSize() can be overridden, in which case,
 * no synchronization is needed.
 */
public class AudioPlayController implements Runnable
{	
	private SourceDataLine line;
	private AudioClip audioOut;
	private byte[] data = new byte[MAX_WRITE_BUFFER_SIZE];
	private int writeBufferSize = MAX_WRITE_BUFFER_SIZE;
	private int curPos = 0;
	private int notifyPos;
	private double playbackRate = 1;
	private boolean stopped = true;
	
	public AudioPlayController()
	{
	}

	public void run() {
		try {
			synchronized (this)
			{
				stopped = false;
			}
			
			AudioFormat targetFormat = new AudioFormat(Encoding.PCM_SIGNED, 
					(float)(audioOut.getRate() * playbackRate), audioOut.getSampleSizeInBits(), audioOut.getNumChannels(), 
					audioOut.getNumChannels() * audioOut.getSampleSizeInBits() / 8, 
					(float)(audioOut.getRate() * playbackRate), false);
			rawplay(targetFormat, audioOut);
		} catch (IOException e) {
			e.printStackTrace();
			NotificationManager.inst().error("Internal error, save your work and exit asap!");
		} catch (LineUnavailableException e) {
			e.printStackTrace();
			NotificationManager.inst().error("Internal error, save your work and exit asap!");
		}
	}
	
	public void stop()
	{
		if(line == null)
			return;
		
		//simple close the line, and the thread should stop automatically
		line.stop();
		line.drain();
		line.close();
		
		synchronized(this)
		{
			while(!stopped )
				try {
					wait();
				} catch (InterruptedException e) {
					NotificationManager.inst().error("Internal error, save your work and exit asap!");
					e.printStackTrace();
				}
		}
	}
	
	protected void setPlaybackRate(double playbackRate) 
	{
		if(this.playbackRate == playbackRate)
			return;
		
		stop();
		this.playbackRate = playbackRate;
		new Thread(this).start();
	}
	
	protected long getMicrosecondPosition() {
		return (line == null ? 0 : line.getMicrosecondPosition());
	}

	//this is the optimal amount of data to have in the outgoing audio buf before we write another chunk
	//to it. Too little and the sound will cut out. Too much, and the audio track can't stop quickly
	private static final int OPTIMAL_DATA_IN_AUDIO_BUF = 8192;
	public static final int MAX_WRITE_BUFFER_SIZE = 8192;	

	private void rawplay(AudioFormat targetFormat, AudioClip audioClip)
			throws IOException, LineUnavailableException {
		//byte lastByte=0;
		
		
		line = getLine(targetFormat);
		if (line != null) {
			// Start
			line.start();
			
			int length = audioClip.length();
			
			int count = 0;
			
			//sleep so that the buffer should be about used up by the time we start again
			//(bps * sl - v0 + 2 * v1 - v2)/bps
			
			double bytesPerMicro = audioClip.getBytesPerSecond()  / 1000000.;
			
			long sleepAmount = 1000;//Math.round(1000000 * WRITE_BUFFER_SIZE / audioClip.getRate() /audioClip.getNumChannels() * .1);
			int lastDataInBuf = -1;
			
			int lastCurPos = -1;
			
			while(curPos < length)
			{
				int bytesToWrite;
				boolean notifyFlag = false;
				
				//if we are playing forever and the current pos is past the point where the
				//int value will wrap around, set it to zero
				if(length == Integer.MAX_VALUE && length - curPos < MAX_WRITE_BUFFER_SIZE)
					curPos = 0;
				
				synchronized (this)
				{
					bytesToWrite = Math.min(MAX_WRITE_BUFFER_SIZE, length - curPos);
					
					
					int dataInBuf = line.getBufferSize() - line.available();
										
					if(lastDataInBuf != -1)
					{
						sleepAmount = Math.round((bytesPerMicro * sleepAmount  * playbackRate - lastDataInBuf + 2 * dataInBuf - OPTIMAL_DATA_IN_AUDIO_BUF)/
								bytesPerMicro);
						if(sleepAmount < 0)
						{
							sleepAmount = 0;
						}
					}
					lastDataInBuf = dataInBuf;
					
//					if(count++ % 10 == 0)
//					{
//						System.out.println("avail is "+line.available()+", buffer size is "+line.getBufferSize()+"sleep amount is "+sleepAmount+
//								"data in buf is "+dataInBuf
////								+" freq is "+((MidiAudioClip)audioClip).getFreqRatio()*((MidiAudioClip)audioClip).baseFreq+
//								//" note is "+((RealTimeNotePlayController)this).getNote()
//									);
//					}
					
					audioClip.copyTo(curPos,bytesToWrite, data, 0);

					if(curPos < notifyPos && curPos + bytesToWrite >= notifyPos)
					{
						notifyFlag = true;
					}

					curPos += bytesToWrite;
				} //synchronized
				
				if(notifyFlag) 
					notifyCrossedPos(); 
				/*for(int i = 0; i < bytesToWrite-1; i++)
				{
					data[i] = audioClip.byteAt(curPos+i);
					if(Math.abs(data[i] - lastByte) > 50)
						System.out.println(String.format("ai caramba curPos is %d, i is %d, data[i] is %d lastByte is %d",curPos, i,data[i],lastByte));
					lastByte = data[i];
					//System.out.println(String.format("%5d: %3d", curPos, data[i]));
				}*/
				
				int bytesWritten = line.write(data, 0, bytesToWrite);
				//System.out.println("zzz"+Util.toHex(data, 0, Math.min(bytesToWrite,1024)));
				
				notifyBufferWrote(bytesWritten); 
				
				
				//if we did not write what we asked for, this means the line has been stopped
				if(bytesWritten != bytesToWrite)
				{
					synchronized (this)
					{
						stopped = true;
						notifyAll();
					}

					return; //so exit
				}
				
				try {
					if(sleepAmount > 0)
						Thread.sleep(sleepAmount / 1000,(int)( (sleepAmount % 1000) * 1000));
				} catch (Exception e) {
					NotificationManager.inst().error("Internal error, save your work and exit asap!");
					e.printStackTrace();
				}
					
			}
			
			// we wrote everything, so notify the model that we're done playing
			line.drain();
			line.stop();
			line.close();
			
			synchronized (this)
			{
				stopped = true;
				notifyAll();
			}

			notifyStopped();
		}
	}

	/**
	 * Will call notifyCrossedPos when we cross it while playing.
	 */
	protected void setNotifyPos(int notifyPos) {
		this.notifyPos = notifyPos; 
	}
	
	/**
	 * Is called when a buffer is written. May be overridden
	 */
	protected void notifyCrossedPos()
	{
	}

	/**
	 * Is called when a buffer is written. May be overridden
	 */
	protected void notifyBufferWrote(int size)
	{
	}

	
	/**
	 * Is called when the playing thread stops. May be overridden.
	 */
	protected void notifyStopped()
	{
	}

	private static SourceDataLine getLine(AudioFormat audioFormat)
			throws LineUnavailableException {
		SourceDataLine res = null;
		DataLine.Info info = new DataLine.Info(SourceDataLine.class,
				audioFormat);
		res = (SourceDataLine) AudioSystem.getLine(info);
		res.open(audioFormat);
		return res;
	}

	public AudioClip getAudioOut() {
		return audioOut;
	}

	public void setAudioOut(AudioClip audioOut) {
		this.audioOut = audioOut;
	}

	public int getCurPos() {
		return curPos;
	}

	public void setCurPos(int curPos) {
		this.curPos = curPos;
	}

	/**
	 * This may be overridden to re-calculate the write buffer size
	 * on every write. Do not return more than MAX_WRITE_BUFFER_SIZE
	 */
	public int getWriteBufferSize() {
		return writeBufferSize;
	}

	public void setWriteBufferSize(int writeBufferSize) {
		this.writeBufferSize = writeBufferSize;
	}

	public double getPlaybackRate() {
		return playbackRate;
	}

}
