package com.superhac.JXBStreamer.Core;

import java.io.File;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.nio.ByteBuffer;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.channels.ServerSocketChannel;
import java.nio.channels.SocketChannel;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.logging.Logger;

/**
 * This is the main class that is used for running a XBMSP server. The server
 * uses non-blocking socket IO and runs as a thread.
 * <p>
 * <p>
 * More information can be found at <a
 * href="http://www.superhac.com">Superhac.com</a>
 * 
 * @author Steven Scott (superhac007@gmail.com)
 */

public class XBMSPServer implements Runnable {

	/**
	 * XBMSP server port to listen to for incoming connections. Default setting
	 * is 1400.
	 */
	private int port = 1400;

	/**
	 * The file directories that make up the virtual root directory. The last
	 * part of the path becomes a directory in the virtual root.
	 */
	private ArrayList<File> virtualRoot;

	/** Activates the XBMSP autoDiscovery server if set. */
	private boolean autoDiscoveryEnabled = true;

	/** Activates the server if set. (Binds the port for listening). */
	private boolean startServer = false;

	/** Stops the server while running. */
	private boolean stopServer = false;

	/** is the server runnning? */
	private boolean serverRunning = false;

	/** the Selector for managing the connections. */
	private Selector selector;

	/** The XBMSP discovery portocol server. */
	private XBMSPDiscoverServer discoveryServer;

	/** Contains all the message handlers for each connected client. */
	private ArrayList<XBMSPServerMessageHandler> smhList = new ArrayList<XBMSPServerMessageHandler>();

	/** debuging logger instance */
	private static Logger logger;

	/** The timer used is polling for inactive clients */
	private Timer timer;

	/**
	 * The amount of nonactivity before the server disconnects a client (in
	 * seconds)
	 */
	private int maximumClientIdleTime = 1800; // in seconds (30 minutes)

	/**
	 * Creates an instance of the XBMSP Server that runs as a thread. The
	 * instance threads itself and does not stop running as a thread until the
	 * virtual machine is terminated.
	 */
	public XBMSPServer() {

		// gets the logger instance
		logger = com.superhac.JXBStreamer.Core.Debug.getLogger();

		// create a new discovery server...
		discoveryServer = new XBMSPDiscoverServer(port);

		// Run the server as a thread. It threads itself!
		Runnable runnable = this;
		Thread thread = new Thread(runnable);
		thread.start();

		// start the dead client monitor
		// check for dead clients every 5 seconds
		timer = new Timer();
		timer
				.scheduleAtFixedRate(new CheckForClientTimeoutsTask(this), 0,
						5000);

	}

	/**
	 * Main loop of the Server running as a Thread. Comes from impementing
	 * Runnable
	 */
	public void run() {
		while (true) {
			if (startServer) {
				if (autoDiscoveryEnabled)
					discoveryServer.startServer();
				serverListeningLoop();
			}
			// take nap for while the server is NOT listening for connection
			try {
				Thread.sleep(500);
			} catch (InterruptedException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		}

	}

	/**
	 * The main socket listening loop. The sockets are bound and messages are
	 * routed.
	 * 
	 */
	private void serverListeningLoop() {
		try {

			// create a selector for geting IO
			selector = Selector.open();
			// create a SSC object
			ServerSocketChannel ssc = ServerSocketChannel.open();
			// configure it for non-blocking IO
			ssc.configureBlocking(false);
			// get the ServerSocket from the SSC
			ServerSocket ss = ssc.socket();
			// Bind the Server Socket to a port
			ss.bind(new InetSocketAddress(port)); // bind to port

			// register for accepting incomming connections
			SelectionKey key = ssc.register(selector, SelectionKey.OP_ACCEPT);

			// buffer for reading
			ByteBuffer buffer = ByteBuffer.allocate(2048);

			// if we make it this far the server is running sucessfully
			serverRunning = true;

			while (!stopServer) {

				if (com.superhac.JXBStreamer.Core.Debug.debug)
					logger.info("Waiting for packet...");

				// main loop wait for events
				selector.select();

				// we got an event
				Set skeys = selector.selectedKeys();
				Iterator it = skeys.iterator();

				while (it.hasNext()) {

					// get a key
					SelectionKey rsk = (SelectionKey) it.next();

					// A new connection
					if ((rsk.readyOps() & SelectionKey.OP_ACCEPT) == SelectionKey.OP_ACCEPT) {
						// accept Connection
						Socket socket = ss.accept();

						SocketChannel sc = socket.getChannel();
						sc.configureBlocking(false);
						sc.register(selector, SelectionKey.OP_READ);

						// socket.setTcpNoDelay(true);
						// create a new handler object for connection
						XBMSPServerMessageHandler smh = new XBMSPServerMessageHandler(
								sc, virtualRoot);
						// add it to the connections list
						// synchronize be cause of ServerStatus
						synchronized (this) {
							smhList.add(smh);
						}

						// send version information
						String version = XBMSPEncoderDecoder.SERVER_VERSION;
						ByteBuffer buf = ByteBuffer.wrap(version
								.getBytes("US-ASCII"));
						sc.write(buf);

						if (com.superhac.JXBStreamer.Core.Debug.debug)
							logger.info("New connection created..");

						// remove the key.. were done with it
						// selector.selectedKeys().remove(rsk);
						it.remove();
					}
					// its a read // route to right handler
					else if ((rsk.readyOps() & SelectionKey.OP_READ) == SelectionKey.OP_READ) {
						SocketChannel sc = (SocketChannel) rsk.channel();

						// remove the key.. were done with it
						// selector.selectedKeys().remove(rsk);
						it.remove();

						for (int i = 0; i < smhList.size(); i++) {
							if (sc.equals(smhList.get(i).getSocketChannel())) {// its
								// a
								// match

								buffer.clear();

								int numBytesRead = sc.read(buffer);

								if (numBytesRead == -1) {
									// No more bytes can be read from the
									// channel
									if (com.superhac.JXBStreamer.Core.Debug.debug)
										logger.info("Client Disconnected");

									synchronized (this) {
										smhList.remove(i);
									}
									sc.close();

								} else {

									buffer.flip();
									smhList.get(i).IncommingMessage(buffer);

								}

							}

						}

					}

				}

			}

			key.selector().close();
			ssc.close();

			synchronized (this) {
				smhList.removeAll(smhList);
			}
			serverRunning = false;
			stopServer = false;
			startServer = false;

		} catch (IOException e) {
			System.out.println("ERROR: " + e);
		}

	}

	/**
	 * Provides detailed information about each connected client.
	 * 
	 * @return The server status object which contains info about each connected
	 *         client.
	 */
	public ServerStatus getServerStatus() {
		ArrayList<ConnectedClientStatus> clients = new ArrayList<ConnectedClientStatus>();
		for (XBMSPServerMessageHandler smh : smhList)
			clients.add(smh.getStatus());

		return new ServerStatus(serverRunning, clients);

	}

	/**
	 * Is the server running. E.g. Listening for incoming connections.
	 * 
	 * @return True if the server is awaiting comming connections and false if
	 *         not.
	 */
	public boolean isServerRunning() {
		return serverRunning;
	}

	/**
	 * Stops the server from accepting connections and also disconnects any
	 * connected clients.
	 * 
	 */
	public void stopServer() {

		// stop discovery server
		if (autoDiscoveryEnabled)
			discoveryServer.stopServer();

		// stop regular server
		stopServer = true;
		selector.wakeup();

	}

	/**
	 * Starts the server so it can recieve incoming connections.
	 * 
	 */
	public void startServer() {
		startServer = true;
	}

	/**
	 * Gets the binding port for the XBMSP server.
	 * 
	 * @return The port number the server is listening on.
	 */
	public int getPort() {
		return port;
	}

	/**
	 * Set the port number the server will listen on.
	 * 
	 * @param port
	 *            The port number the server will bind to.
	 */
	public void setPort(int port) {

		if (serverRunning) {
			stopServer();
			this.port = port;

			while (serverRunning) {
				// waits until the server stops. Need this to impose a delay
				// between stopping and starting
			}
			startServer();
		} else
			this.port = port;
	}

	/**
	 * Is the the XBMSP server configured for the AutoDiscovery Service(UDP)?
	 * Clients can broadcast on port 1400 and the server will respond with
	 * identifying information that can be used to connect to the server.
	 * 
	 * @return True if the Discovery Server is running and False if not.
	 */
	public boolean isAutoDiscoveryEnabled() {
		return autoDiscoveryEnabled;
	}

	/**
	 * Clients can broadcast on port 1400 and the server will respond with
	 * identifying information that can be used to connect to the server.
	 * 
	 * @param autoDiscoveryEnabled
	 *            True if it should be enabled and false if not.
	 */
	public void setAutoDiscoveryEnabled(boolean autoDiscoveryEnabled) {
		this.autoDiscoveryEnabled = autoDiscoveryEnabled;
	}

	/**
	 * Set the virtual root directory. This allows multiple directories to be
	 * shared that are spread over different paths. The client is presented with
	 * a virtual root directory that consists of the supplied paths as a single
	 * dirctory.
	 * 
	 * @param virtualRoot
	 *            An arraylist of files that make up the virtual root directory
	 */
	public void setVirtualRoot(ArrayList<File> virtualRoot) {
		this.virtualRoot = virtualRoot;

	}

	/**
	 * Returns an ArrayList of the message handlers. The message handlers handle
	 * the communicate between the client and the server. Each connected client
	 * has its own message handler. As a packet comes into the server it is
	 * routed to the appropraite message handler for that client.
	 * 
	 * @return The current message handlers for the clients
	 */
	protected ArrayList<XBMSPServerMessageHandler> getServerMessageHandlers() {
		return smhList;
	}

	/**
	 * Gets the maximum client idle time before being disconnected by the
	 * server.
	 * 
	 * @return The currently set Idle time in seconds.
	 */
	protected int getMaximumClientIdleTime() {
		return maximumClientIdleTime;
	}

	/**
	 * Sets the maximum client idle time before being disconnected by the
	 * server.
	 * 
	 * @param maximumClientIdleTime
	 *            The maximum amount of idle time before the server disconnects
	 *            the client.
	 */
	protected void setMaximumClientIdleTime(int maximumClientIdleTime) {
		this.maximumClientIdleTime = maximumClientIdleTime;
	}

}

/**
 * This is a inner class of XBMSPServer that is used to check the idle time of
 * connected clients. If the time a client has been idle for exceeds the
 * maximumClientIdleTime then the client is disconnected. Operates by the task
 * being schduled by a Timer (timer.scheduleAtFixedRate).
 * 
 * @author sjscott
 * 
 */
class CheckForClientTimeoutsTask extends TimerTask {

	int maximumDelay;

	XBMSPServer caller;

	CheckForClientTimeoutsTask(XBMSPServer caller) {
		this.caller = caller;
		maximumDelay = caller.getMaximumClientIdleTime();
	}

	public void run() {

		ArrayList<XBMSPServerMessageHandler> clients = caller
				.getServerMessageHandlers();
		ArrayList<XBMSPServerMessageHandler> remove = new ArrayList<XBMSPServerMessageHandler>();

		for (XBMSPServerMessageHandler client : clients)
			if (client.getLastTransmissionDelay() > maximumDelay)
				remove.add(client);

		// if any are over the set idel time remove
		for (int i = 0; i < remove.size(); i++) {
			synchronized (this) {
				clients.remove(remove.get(i));
			}

			// close the clients socket!
			try {
				remove.get(i).getSocketChannel().close();
			} catch (IOException e) {

			}

		}
	}

}
