package com.superhac.JXBStreamer.Core;

import java.io.IOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetSocketAddress;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.nio.ByteBuffer;

/**
 * This class handles the server socket connections for the XBMSP Discovery
 * protocol. By default it listens on port 1400 for UDP requests. The server
 * responds with connection information for connecting to the XBMSPServer
 * instance. This runs as a thread started by itself.
 * <p>
 * <p>
 * More information can be found at <a
 * href="http://www.superhac.com">Superhac.com</a>
 * 
 * @author Steven Scott (superhac007@gmail.com)
 */
public class XBMSPDiscoverServer extends Thread {

	private int port = 1400;

	private String serverIP;

	private DatagramSocket socket;

	private boolean running = false;

	private boolean stopThread = false;

	/**
	 * Create an instance of the server
	 * 
	 * @param port
	 *            The port the server should listen on.
	 */
	protected XBMSPDiscoverServer(int port) {

		this.port = port;
		// get the local ip address
		serverIP = InternalIPFinder.getLocalIPAddress();

		Thread thread = this;
		thread.start();

	}

	public void run() {
		while (!stopThread) {
			if (running)
				initServer();
			try {
				Thread.sleep(1000);
			} catch (InterruptedException e) {
				// TODO Auto-generated catch block
				// e.printStackTrace();
			}
		}
	}

	/**
	 * Start the server.
	 * 
	 */
	public void startServer() {
		if (!running)
			running = true;

	}

	/**
	 * Stop the server
	 * 
	 */
	protected void stopServer() {
		running = false;

		socket.close();

	}

	/**
	 * Terminate the thread. If done this instance of XBMSPDiscoverServer should
	 * be destoryed.
	 * 
	 */
	protected void stopThread() {
		stopThread = true;
		running = false;
		socket.close();
	}

	/**
	 * Main loop of the server
	 * 
	 */
	private void initServer() {

		try {

			System.out.println("Server started...");
			DatagramPacket packet = new DatagramPacket(new byte[512], 512);
			socket = new DatagramSocket(null);
			// socket.setReuseAddress(true);
			socket.bind(new InetSocketAddress(port));

			while (running) {

				socket.receive(packet);

				process(packet);
			}

			// socket.send(packet);
		} catch (SocketException e) {
			// TODO Auto-generated catch block
			// e.printStackTrace();
			running = false;

		} catch (IOException e) {
			// TODO Auto-generated catch block
			// e.printStackTrace();
			running = false;

		}

		running = false;
	}

	/**
	 * Decodes incoming XBMSP Discovery protocol message.
	 * 
	 * @param packet
	 *            The datagram packet that was received.
	 */
	private void process(DatagramPacket packet) {

		ByteBuffer buffer = ByteBuffer.allocate(1500);

		buffer.put(packet.getData());

		buffer.flip(); // flip for reading

		int size = buffer.getInt();
		byte type = buffer.get();
		int messageID = buffer.getInt();
		int clientVersionSize = buffer.getInt();

		byte[] string = new byte[clientVersionSize];

		for (int i = 0; i < clientVersionSize; i++)
			string[i] = buffer.get();

		//String clientVersion = new String(string, 0, string.length);

		// System.out.println("Client Version: "+ clientVersion);

		// reset for reuse

		buffer.clear();
		// pack a response
		int lengthOfServerVersion = XBMSPEncoderDecoder.SERVER_VERSION.length();
		int lengthOfServerIP = serverIP.length();
		int lengthOfPort = String.valueOf(port).length();
		int serverComment = 0;
		int typeSize = 1;

		int messageSize = typeSize + 4 + lengthOfServerIP + lengthOfPort
				+ lengthOfServerVersion + serverComment + 16; // +4 for
																// messageID &
																// +16 for each
																// string size
																// field that is
																// packed

		// System.out.println("mss size:" + messageSize);

		buffer.putInt(messageSize);
		buffer.put(XBMSPEncoderDecoder.XBMSP_PACKET_SERVER_DISCOVERY_REPLY);
		buffer.putInt(messageID);
		buffer.putInt(lengthOfServerIP);
		buffer.put(serverIP.getBytes());
		buffer.putInt(lengthOfPort);
		buffer.put(String.valueOf(port).getBytes());
		buffer.putInt(lengthOfServerVersion);
		buffer.put(XBMSPEncoderDecoder.SERVER_VERSION.getBytes());
		buffer.putInt(serverComment);

		buffer.flip();
		byte data[] = new byte[buffer.remaining()];

		buffer.get(data);

		try {
			socket.send(new DatagramPacket(data, data.length, packet
					.getAddress(), packet.getPort()));
			// socket.send(packet);
		} catch (UnknownHostException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}

	}

	/**
	 * Set the port number of the server
	 * 
	 * @param port
	 *            the port number the server should listen on for the UDP
	 *            requests.
	 */
	protected void setPort(int port) {
		this.port = port;
	}

}
