/* XBFLASH - Xbox Flash image utility
 * Copyright (C) 2002  Paul Bartholomew
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * 
 * fbl_hash.c - Functions for hashing/signing the FBL region
 *
 * Special thanks to Franz/hamtitampti for the TEA-hack-compatible
 * RSA public/private key.
 * 
 */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "xbconfig.h"
#include "xbpack.h"
#include "cabio.h"
#include "cryptlib/hash/sha.h"
#include "cryptlib/crypt/rc4.h"
#include "giants.h"
#include "tea_hash.h"


#define	FLASHADDR_RSA_PUBLIC_KEY		0xffffd410
#define	RSA_PUBLIC_KEY_SIZE				0x114
#define	FLASHADDR_RSA_ENCRYPTED_DIGEST	0xfffffc80
#define	RSA_ENCRYPTED_DIGEST_SIZE		256

#define	FLASHADDR_HASH_DATA_TABLE_BASE	(FLASHADDR_RSA_ENCRYPTED_DIGEST + RSA_ENCRYPTED_DIGEST_SIZE)
#define	FLASHADDR_HASH_DATA_TABLE_SIZE	128
#define	FLASHADDR_PTR_XCODE_HASH_BASE	(FLASHADDR_HASH_DATA_TABLE_BASE)
#define	FLASHADDR_XCODE_HASH_SIZE		(FLASHADDR_HASH_DATA_TABLE_BASE+4)
#define	FLASHADDR_PTR_2BL_HASH_BASE		(FLASHADDR_HASH_DATA_TABLE_BASE+8)

#define	ADDRMASK(a,s)	((a) & ((s)-1))

void gigimport(giant g, unsigned char *buff,int len);

typedef struct tagRSA_PUBLIC_KEY
{
	char Magic[4];  		// "RSA1"
	unsigned int Bloblen; 		// 264 (Modulus + Exponent + Modulussize)
	unsigned int Bitlen;  	// 2048
	unsigned int ModulusSize;	// 255 (bytes in the Modulus)
	unsigned char Exponent[4];
	unsigned char Modulus[256];     // Bit endian style
	unsigned char Privatekey[256];  // Private Key .. we do not have it -- Big endian style
} RSA_PUBLIC_KEY;

RSA_PUBLIC_KEY	TEA_RSA_key = {
	{ 'R', 'S', 'A', '1' },

	264,
	2048,
	255,

	// Exponent
	{ 0x01, 0x00, 0x01, 0x00 },

	// Public Modulus
	{	0xD3, 0xD7, 0x4E, 0xE5, 0x66, 0x3D, 0xD7, 0x66, 0xC2, 0xD4, 0xA3, 0x21, 0xF2, 0x17, 0x36, 0x54, 
		0x2E, 0x52, 0xF6, 0x52, 0x02, 0x10, 0xF5, 0x64, 0x9C, 0x34, 0x7B, 0xFF, 0xEF, 0x7F, 0xC2, 0xEE, 
		0xBD, 0x05, 0x8B, 0xDE, 0x79, 0xB4, 0x77, 0x8E, 0x5B, 0x8C, 0x14, 0x99, 0xE3, 0xAE, 0xC6, 0x73, 
		0x72, 0x73, 0xB5, 0xFB, 0x01, 0x5B, 0x58, 0x46, 0x6D, 0xFC, 0x8A, 0xD6, 0x95, 0xDA, 0xED, 0x1B, 
		0x2E, 0x2F, 0xA2, 0x29, 0xE1, 0x3F, 0xF1, 0x39, 0x5B, 0x64, 0x51, 0xAE, 0xA2, 0xC0, 0xF7, 0x3A, 
		0xB3, 0x3E, 0x8A, 0xF5, 0xFF, 0x06, 0x92, 0x5C, 0x07, 0x26, 0x75, 0x79, 0x10, 0x5D, 0x47, 0x3E, 
		0xD1, 0x6A, 0x52, 0x10, 0x0B, 0xAE, 0x6A, 0x8B, 0x33, 0x44, 0x93, 0xDE, 0xF9, 0x9D, 0xFB, 0x15, 
		0xD9, 0xA4, 0x1C, 0xCF, 0x6F, 0xE4, 0x71, 0x94, 0xBE, 0x13, 0x00, 0xA8, 0x52, 0xCA, 0x07, 0xBD, 
		0x27, 0x98, 0x01, 0xA1, 0x9E, 0x4F, 0xA3, 0xED, 0x9F, 0xA0, 0xAA, 0x73, 0xC4, 0x71, 0xF3, 0xE9, 
		0x4E, 0x72, 0x42, 0x9C, 0xF0, 0x39, 0xCE, 0xBE, 0x03, 0x76, 0xFA, 0x2B, 0x89, 0x14, 0x9A, 0x81, 
		0x16, 0xC1, 0x80, 0x8C, 0x3E, 0x6B, 0xAA, 0x05, 0xEC, 0x67, 0x5A, 0xCF, 0xA5, 0x70, 0xBD, 0x60, 
		0x0C, 0xE8, 0x37, 0x9D, 0xEB, 0xF4, 0x52, 0xEA, 0x4E, 0x60, 0x9F, 0xE4, 0x69, 0xCF, 0x52, 0xDB, 
		0x68, 0xF5, 0x11, 0xCB, 0x57, 0x8F, 0x9D, 0xA1, 0x38, 0x0A, 0x0C, 0x47, 0x1B, 0xB4, 0x6C, 0x5A, 
		0x53, 0x6E, 0x26, 0x98, 0xF1, 0x88, 0xAE, 0x7C, 0x96, 0xBC, 0xF6, 0xBF, 0xB0, 0x47, 0x9A, 0x8D, 
		0xE4, 0xB3, 0xE2, 0x98, 0x85, 0x61, 0xB1, 0xCA, 0x5F, 0xF7, 0x98, 0x51, 0x2D, 0x83, 0x81, 0x76, 
		0x0C, 0x88, 0xBA, 0xD4, 0xC2, 0xD5, 0x3C, 0x14, 0xC7, 0x72, 0xDA, 0x7E, 0xBD, 0x1B, 0x4B, 0xA4
	},

	// Private key
	{
		0x07, 0x06, 0x34, 0xD1, 0xDC, 0xDA, 0x1C, 0xE6, 0x39, 0x58, 0xFA, 0x73, 0xDC, 0x70, 0x6A, 0x82, 
		0x42, 0x8F, 0x3B, 0x89, 0x75, 0x2F, 0x14, 0x95, 0xFA, 0x85, 0x59, 0x42, 0x82, 0x3C, 0xCB, 0x35, 
		0x48, 0x3A, 0xB7, 0xFA, 0x3E, 0x69, 0xD0, 0x73, 0xC7, 0xB0, 0xEA, 0xDD, 0xCE, 0xCB, 0xC1, 0x06, 
		0x77, 0x14, 0x59, 0x81, 0xFB, 0x23, 0x72, 0x31, 0xCD, 0x44, 0x0B, 0x90, 0x1E, 0xF0, 0x5C, 0x62, 
		0x0E, 0xFD, 0x31, 0xB6, 0xE8, 0x0C, 0x5A, 0x88, 0x44, 0x9A, 0x0A, 0x1B, 0x7A, 0x84, 0xB9, 0xCC, 
		0x92, 0xB8, 0x3A, 0xCB, 0x9D, 0x78, 0xC8, 0x47, 0x88, 0xCB, 0xC6, 0xED, 0x34, 0x7C, 0x4A, 0xA2, 
		0x31, 0x0B, 0xD0, 0xC5, 0xF1, 0x62, 0xED, 0xF2, 0x46, 0x15, 0xBA, 0x56, 0xC3, 0x15, 0x47, 0x06, 
		0x49, 0x02, 0x30, 0xF4, 0xD6, 0xB1, 0xEF, 0x56, 0x24, 0x44, 0x23, 0x05, 0x6A, 0xFF, 0xF3, 0xF0, 
		0x27, 0x5E, 0x0C, 0xF2, 0xA5, 0x46, 0x09, 0xD5, 0xC4, 0x01, 0x1C, 0x43, 0x8A, 0x85, 0x77, 0x8B, 
		0xA1, 0x6A, 0x3E, 0x70, 0xD2, 0x95, 0xBC, 0x0F, 0x61, 0x49, 0xA5, 0xC1, 0xBA, 0x26, 0xDC, 0x3C, 
		0x82, 0xC0, 0xF9, 0xF1, 0xA6, 0x64, 0x82, 0xF7, 0x96, 0x04, 0x3A, 0xFF, 0x54, 0xED, 0x5F, 0xFD, 
		0xCD, 0xCF, 0xA3, 0xDC, 0x0C, 0xD3, 0xD1, 0xD4, 0x9A, 0x5B, 0x83, 0xEB, 0x14, 0xA6, 0xF6, 0x3B, 
		0x73, 0x0F, 0x0E, 0x61, 0x67, 0x99, 0xCA, 0x35, 0x0E, 0xC8, 0x67, 0xCE, 0xEE, 0xCC, 0x54, 0x33, 
		0x5F, 0x79, 0xD0, 0xB0, 0xB2, 0x96, 0xA0, 0x7C, 0xEF, 0x4A, 0xAE, 0xD9, 0xE1, 0xE2, 0x79, 0x31, 
		0x3F, 0x85, 0xE6, 0x69, 0x9D, 0xAC, 0x56, 0x3E, 0x22, 0x34, 0xE4, 0x25, 0x74, 0x24, 0xE4, 0x45, 
		0x56, 0x7A, 0xB8, 0xE0, 0xC9, 0xC8, 0xDD, 0x50, 0x65, 0xCF, 0x6B, 0xC8, 0xB7, 0x37, 0xC8, 0x8F
	},

};




#define	RSA_SIG_SIZE	256

void
gen_fbl_signature(unsigned char *p_rom_image, unsigned int size_rom_image)
{
	SHA_CTX	sha_ctx;
	RC4_KEY	rc4_key;
	unsigned char	sha_digest[SHA_DIGEST_LENGTH];
	unsigned char	c_signature[RSA_SIG_SIZE];
	unsigned char	crypted_rsa_public_key[sizeof(TEA_RSA_key)];
	giant n = newgiant(INFINITY);	
	giant e = newgiant(INFINITY);
	giant sig = newgiant(INFINITY);
	int	i;
	unsigned int	xcode_region_base, xcode_region_size;
	unsigned int	f2bl_region_base, f2bl_region_size;

	if ((CFG_RC4_keymethod != 1) || (CFG_RC4_keylen != 16)) {
		return;
	}
	// RC4-encrypt the RSA public key
	RC4_set_key(&rc4_key, 12, CFG_RC4_key);
	RC4(&rc4_key, RSA_PUBLIC_KEY_SIZE, (void *)&TEA_RSA_key, crypted_rsa_public_key);
	// Copy crypted RSA public key into the FBL @ ffffd410
	memcpy(&p_rom_image[ADDRMASK(FLASHADDR_RSA_PUBLIC_KEY, size_rom_image)], crypted_rsa_public_key, RSA_PUBLIC_KEY_SIZE);

	// *fffffd80 = ptr to start of 'xcode hash region' (starts at base of Flash)
	xcode_region_base = *(unsigned int *)(&p_rom_image[ADDRMASK(FLASHADDR_PTR_XCODE_HASH_BASE, size_rom_image)]);
	xcode_region_base &= 0xfffc0000;
	xcode_region_base |= 0xfff00000;

	// *fffffd48 = size of 'xcode hash region'
	xcode_region_size = *(unsigned int *)(&p_rom_image[ADDRMASK(FLASHADDR_XCODE_HASH_SIZE, size_rom_image)]);

	// *fffffd88 = ptr to start of 'upper hash region' (starts at 2BL start)
	f2bl_region_base = *(unsigned int *)(&p_rom_image[ADDRMASK(FLASHADDR_PTR_2BL_HASH_BASE, size_rom_image)]);
	// 'upper hash region' goes up to 0xfffffc80 (where RSA-encrypted digest is stored)
	f2bl_region_size = (FLASHADDR_RSA_ENCRYPTED_DIGEST - f2bl_region_base);

	SHA1_Init(&sha_ctx);

	SHA1_Update(&sha_ctx, &p_rom_image[ADDRMASK(FLASHADDR_HASH_DATA_TABLE_BASE, size_rom_image)], FLASHADDR_HASH_DATA_TABLE_SIZE);
	SHA1_Update(&sha_ctx, &p_rom_image[ADDRMASK(xcode_region_base, size_rom_image)], xcode_region_size);
	SHA1_Update(&sha_ctx, &p_rom_image[ADDRMASK(f2bl_region_base, size_rom_image)], f2bl_region_size);
	SHA1_Final(sha_digest, &sha_ctx);

	memset(c_signature, 0xff, sizeof(c_signature));
	c_signature[(RSA_SIG_SIZE-1)] = 0;
	c_signature[(RSA_SIG_SIZE-1)-1] = 1;
	for (i = 0; i < SHA_DIGEST_LENGTH; i++) {
		c_signature[i] = sha_digest[(SHA_DIGEST_LENGTH-1)-i];
	}
	c_signature[SHA_DIGEST_LENGTH] = 0;

	gigimport(n, TEA_RSA_key.Modulus, 256);	//sizeof(TEA_RSA_key.Modulus));
	gigimport(e, TEA_RSA_key.Privatekey, 256);	//sizeof(TEA_RSA_key.Privatekey));
	gigimport(sig, c_signature, RSA_SIG_SIZE);

	powermodg(sig, e, n);

	memcpy(&p_rom_image[ADDRMASK(FLASHADDR_RSA_ENCRYPTED_DIGEST, size_rom_image)], sig->n, 256);

}


void gigimport(giant g, unsigned char *buff,int len)
{
	int count;
	memcpy(g->n,buff,len);
	g->sign = len/2;
	
	// Correcting to bits now
	for (count = g->sign ;count!=0;count--) {
		
		if (g->n[count] != 0x00) {
			count = count+1;
			break;      
		}
	}                                      

	g->sign= count;
	if (g->sign==0) g->sign = 1;
}

