/*
 * MP3 Tag library. It includes an implementation of the ID3 tags and Lyrics3
 * tags as they are defined at www.id3.org
 *
 * Copyright (C) Eric Farng 2003
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.farng.mp3;

import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.Iterator;

import org.farng.mp3.id3.AbstractID3v2;
import org.farng.mp3.id3.AbstractID3v2Frame;
import org.farng.mp3.id3.ID3v2_4;


/**
 * This class represents any type of tag in an MP3 file, including ID3 and
 * Lyrics and the file name.
 *
 * @author Eric Farng
 * @version $Revision: 1.1 $
 */
public abstract class AbstractMP3Tag
    extends AbstractMP3FileItem {
    /**
     * Appends this tag to the given file. Append means any information this
     * tag contains will be added to the file's corresponding tag, but it will
     * not replace any fields that the file already has. If the file does not
     * have this specific tag, a new one will be created.
     *
     * @param file MP3 file to append to.
     *
     * @throws IOException on any I/O error
     * @throws TagException on any exception generated by this library.
     */
    abstract public void append(RandomAccessFile file)
                         throws IOException, TagException;

    /**
     * removes the specific tag the easiest way. <BR> ID3v1 - cuts the length
     * of the tag <BR> lyrics3 -cuts the length of the tag, then writes the
     * id3v1 tag if it existed <BR> id3v2 - just overwrites the ID3 tag
     * indicator at the start of the tag<BR>
     *
     * @param file MP3 file to append to.
     *
     * @throws IOException on any I/O error
     */
    abstract public void delete(RandomAccessFile file)
                         throws IOException;

    /**
     * Overwrites this tag to the given file. Overwrite means any information
     * this tag contains will replace any existing fields in the file's
     * corresponding tag. If the file does not have this specific tag, a new
     * one will be created.
     *
     * @param file MP3 file to overwrite
     *
     * @throws IOException on any I/O error
     * @throws TagException on any exception generated by this library.
     */
    abstract public void overwrite(RandomAccessFile file)
                            throws IOException, TagException;

    /**
     * Looks for this tag. returns true if found. If found, the file pointer is
     * right after the tag start indicator i.e. "TAG" "LYRICSBEGIN" "ID3" + 2
     *
     * @param file MP3 file to overwrite
     *
     * @return returns true if found, false otherwise.
     *
     * @throws IOException on any I/O error
     */
    abstract public boolean seek(RandomAccessFile file)
                          throws IOException;

    /**
     * Returns true if this tag is a subset of the argument. Both tags are
     * converted into ID3v2_4 tags, and then compared frame by frame.
     *
     * @param tag superset tag
     *
     * @return true if this tag is a subset of the argument
     */
    public boolean isSubsetOf(AbstractMP3Tag tag) {
        AbstractID3v2      subset        = new ID3v2_4(this);
        AbstractID3v2      superset      = new ID3v2_4(tag);
        Iterator           iterator      = subset.iterator();
        AbstractID3v2Frame subsetFrame   = null;
        AbstractID3v2Frame supersetFrame = null;
        String             identifier    = null;

        while (iterator.hasNext()) {
            subsetFrame   = (AbstractID3v2Frame) iterator.next();
            identifier    = subsetFrame.getIdentifier();
            supersetFrame = (AbstractID3v2Frame) superset.getFrame(identifier);

            if (supersetFrame == null) {
                return false;
            }

            if (subsetFrame.isSubsetOf(supersetFrame) == false) {
                return false;
            }
        }

        return true;
    }

    /**
     * This method does nothing, but is called by subclasses for completeness
     *
     * @param tag tag to overwrite
     */
    public void append(AbstractMP3Tag tag) {}

    /**
     * Determines whether another object is equal to this tag. It just compares
     * if they are the same class, then calls <code>super.equals(obj)</code>.
     *
     * @param obj DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public boolean equals(Object obj) {
        if ((obj instanceof AbstractMP3Tag) == false) {
            return false;
        }

        return super.equals(obj);
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    abstract public Iterator iterator();

    /**
     * This method does nothing, but is called by subclasses for completeness
     *
     * @param tag tag to overwrite
     */
    public void overwrite(AbstractMP3Tag tag) {}

    /**
     * This method does nothing, but is called by subclasses for completeness
     *
     * @param tag tag to write to
     */
    public void write(AbstractMP3Tag tag) {}
}