/*
	extract-xiso.c

	an xdvdfs .iso file extraction tool by in <in@fishtank.com>
		written March 10, 2003

	last modified:
	
		03.29.03 in:	fixed a path display bug, changed the tree descent algorithm
						and added ftp to xbox support (rev to v1.2)
						
		04.04.03 in:	added a counter for total number of files in xiso (rev to
						v1.2.1)  THIS VERSION NOT FOR RELEASE!
						
		04.18.03 in:	added xoff_t typecasts for __u32 * __u32 manipulations.
						this fixed a bug with very large iso's where the directory
						table was at the end of the iso--duh! (rev to v1.3)
						
		04.19.03 in:	a user pointed out that the program is increasing its
						memory usage over large iso's.  I've tracked this to the buffer
						allocation in extract_file() during traverse_directory()
						recursions.  As a fix I've moved the copy buffer to a static
						variable.  Not as encapsulated as I'd like but hey, this is
						C after all.
						
						Also added support for FreeBSD (Intel) (rev to v1.4)

	notes:

	view this file with your tab stops set to 4 spaces or it it will look wacky.

	Regarding licensing:

	I think the GPL sucks!  (it stands for Generosity Poor License)

	My open-source code is really *FREE* so you can do whatever you want with it,
	as long as 1) you don't claim that you wrote my code and 2) you retain a notice
	that some parts of the code are copyright in@fishtank.com and 3) you understand
	there are no warranties.  I only guarantee that it will take up disk space!

	If you want to help out with this project it would be welcome, just email me at
	in@fishtank.com.

	This code is copyright in@fishtank.com and is licensed under a slightly modifified
	version of the Berkeley Software License, which follows:

	/*
	 * Copyright (c) 2003 in <in@fishtank.com>
	 * All rights reserved.
	 *
	 * Redistribution and use in source and binary forms, with or without
	 * modification, are permitted provided that the following conditions
	 * are met:
	 *
	 * 1. Redistributions of source code must retain the above copyright
	 *    notice, this list of conditions and the following disclaimer.
	 *
	 * 2. Redistributions in binary form must reproduce the above copyright
	 *    notice, this list of conditions and the following disclaimer in the
	 *    documentation and/or other materials provided with the distribution.
	 *
	 * 3. All advertising materials mentioning features or use of this software
	 *    must display the following acknowledgement:
	 *
	 *    This product includes software developed by in <in@fishtank.com>.
	 *
	 * 4. Neither the name of "in" nor the email address "in@fishtank.com"
	 *    may be used to endorse or promote products derived from this software
	 *    without specific prior written permission.
	 *
	 * THIS SOFTWARE IS PROVIDED `AS IS' AND ANY EXPRESS OR IMPLIED WARRANTIES
	 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
	 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE
	 * AUTHOR OR ANY CONTRIBUTOR(S) BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
	 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
	 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
	 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
	 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
	 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
	 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
	 *\
	
	enjoy!
	
	in
*/

#if defined( __LINUX__ )
	#define _LARGEFILE64_SOURCE
#endif


#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>


#include "libftp-5.0.1.modified.by.in/FtpLibrary.h"


#if defined( __DARWIN__ )
	#define target					"macos-x"

	#define READFLAGS				O_RDONLY
	#define WRITEFLAGS				O_WRONLY | O_CREAT | O_TRUNC
	
	#define big16(n)				( (n) = (n) << 8 | (n) >> 8 )
	#define big32(n)				( (n) = (n) << 24 | (n) << 8 & 0xff0000 | (n) >> 8 & 0xff00 | (n) >> 24 )

	typedef	off_t					xoff_t;
#elif defined( __FREEBSD__ )
	#define target					"freebsd"

	#define READFLAGS				O_RDONLY
	#define WRITEFLAGS				O_WRONLY | O_CREAT | O_TRUNC
	
	#define big16(n)
	#define big32(n)

	typedef	off_t					xoff_t;
#elif defined( __LINUX__ )
	#define target					"linux"
	
	#define READFLAGS				O_RDONLY | O_LARGEFILE
	#define WRITEFLAGS				O_WRONLY | O_CREAT | O_TRUNC | O_LARGEFILE

	#define lseek					lseek64
	
	#define big16(n)
	#define big32(n)

	typedef __off64_t				xoff_t;
#endif


#define DUMP_SECTORS				1


#ifndef __cplusplus
	typedef int bool;
	enum { false, true };
#endif


#define version						"1.4"

#define banner						"extract-xiso v" version " for " target " - written by in <in@fishtank.com>\n\n\
please:    feature requests, bug reports, or development inquiries *ONLY* to\n\
           this email address... technical support/help/howto requests will be\n\
           ignored (read the readme.txt file for more info)\n"
#define usage() 					fprintf( stderr, "%s\nusage: %s <options> file1.xiso file2.xiso ...\n\n\
    options:\n\
\n\
    -d <directory>      change to (local or remote, depending on mode)\n\
                        <directory> before extracting\n\
    -l                  only list files in xiso (don't extract)\n\
    -q                  quiet mode (suppress output)\n\
\n\
    -f <ftp_server>     extract to <ftp_server> instead of local filesystem\n\
    -u <user name>      ftp user name\n\
    -p <password>       ftp password\n\
", banner, argv[ 0 ] );
#define log							if ( ! quiet ) printf
#define Min( a , b )				( ( a ) < ( b ) ? ( a ) : ( b ) )

#define XISO_HEADER_SIZE			0x14
#define	XISO_HEADER_DATA			"MICROSOFT*XBOX*MEDIA"
#define XISO_HEADER_OFFSET			0x10000
#define XISO_SECTOR_SIZE			2048
#define XISO_SECTOR_OFFSET_SIZE		4
#define XISO_DIRTABLE_SIZE			4
#define XISO_FILETIME_SIZE			8
#define XISO_UNUSED_SIZE			0x7c8
#define XISO_TABLE_OFFSET_SIZE		2
#define XISO_FILESIZE_SIZE			4
#define XISO_ATTRIBUTES_SIZE		1
#define XISO_FILENAME_LENGTH_SIZE	1
#define XISO_ATTRIBUTE_RO			0x01
#define XISO_ATTRIBUTE_HID			0x02
#define XISO_ATTRIBUTE_SYS			0x04
#define XISO_ATTRIBUTE_DIR			0x10
#define	XISO_ATTRIBUTE_ARC			0x20
#define XISO_ATTRIBUTE_NOR			0x80
#define XISO_DWORD_SIZE				4

#define EXTRACT_BUFFER_SIZE			65536
#define DIR_NODE_FOR_CREATE_XISO	0			// one of these days I'll get around to writing an xiso creation tool ;)

enum errors {
	err_end_of_sector = -5001
};

typedef enum modes {
	kExtract,
	kUpload,
	kList
} modes;


static FTP						   *ftp = NULL;
static bool							quiet = false;
static xoff_t						total_bytes = 0;
static int							total_files = 0;
static char						   *copy_buffer = NULL;
static xoff_t						total_bytes_all_isos = 0;
static int							total_files_all_isos = 0;


typedef struct dir_node dir_node;

struct dir_node {
#if DIR_NODE_FOR_CREATE_XISO
	dir_node		   *left;
	dir_node		   *right;
	dir_node		   *subdirectory;
#endif

	char			   *filename;
	unsigned long		file_size;
	unsigned char		attributes;
	unsigned long		start_sector;
	unsigned char		filename_length;
};


int upload_file( int in_xiso, dir_node *in_file );
int decode_xiso( char *in_xiso, char *in_path, modes in_mode );
int extract_file( int in_xiso, dir_node *in_file, modes in_mode );
int traverse_directory( int in_xiso, xoff_t in_dir_start, char *in_path, modes in_mode );
int open_ftp_connection( char *in_host, char *in_user, char *in_password, FTP **out_ftp );


#if DEBUG
	void write_sector( int in_xiso, xoff_t in_start, unsigned long in_len, char *in_name );
#endif


int main( int argc, char **argv ) {
	bool		extract = true;
	char	   *server, *pass, *path, *user;
	int			i, opt_char, err = 0, isos = 0;

	if ( argc <= 1 ) { usage(); exit( 1 ); }
	
	server = pass = path = user = NULL;

	while ( ( opt_char = getopt( argc, argv, "d:f:lp:qu:" ) ) != -1 ) {
		switch ( opt_char ) {
			case 'd':		path = strdup( optarg );		break;
			case 'f':		server = strdup( optarg );		break;
			case 'l':		extract = false;				break;
			case 'p':		pass = strdup( optarg );		break;
			case 'q': 		quiet = true;					break;
			case 'u':		user = strdup( optarg );		break;

			default: {
				fprintf( stderr, "\n" );
				usage();
			} break;
		}
	}
	
	if ( optind >= argc ) { usage(); exit( 1 ); }

	if ( ! extract && server ) { free( server ); server = NULL; }
	
	log( "%s", banner );
	
	if ( ( server || extract ) && ( copy_buffer = (char *) malloc( EXTRACT_BUFFER_SIZE ) ) == NULL ) { fprintf( stderr, "unable to allocate memory\n" ); return 1; }
	
	if ( server ) err = open_ftp_connection( server, user, pass, &ftp );
	
	for ( i = optind; ! err && i < argc; ++i ) {
		++isos;
		log( "\n" );
		total_files = total_bytes = 0;

		if ( server && path ) {
			if ( FtpChdir( ftp, path ) < 0 ) err = 1;
			if ( err ) fprintf( stderr, "unable to change to remote directory %s\n", path );
		}
			
		if ( ! err ) err = decode_xiso( argv[ i ], path, server ? kUpload : extract ? kExtract : kList );
		if ( ! err ) { log( "\n%u files in %s total %llu bytes\n", total_files, argv[ i ], total_bytes ); }
	}
	
	if ( ! err && isos > 1 ) { log( "\n%u files in %u xiso's total %llu bytes\n", total_files_all_isos, isos, total_bytes_all_isos ); }
	
	if ( ftp && FtpBye( ftp ) < 0 ) FtpQuickBye( ftp );
	
	if ( copy_buffer ) free( copy_buffer );
	if ( server ) free( server );
	if ( pass ) free( pass );
	if ( path ) free( path );
	if ( user ) free( user );
		
	return err;
}


int decode_xiso( char *in_xiso, char *in_path, modes in_mode ) {
	int				xiso;
	char		   *name, *buf;
	dir_node	   *root = NULL;
	bool			chop = false;
	int				err = 0, len;
	char		   *cwd = NULL, buffer[ 4096 ];
	int				path_len = 0, add_slash = 0;
	unsigned long	root_dir_sect, root_dir_size;

	#define unchop() if ( chop ) { name[ len ] = '.'; chop = false; }

	// open xiso
	if ( ( xiso = open( in_xiso, READFLAGS, 0 ) ) == -1 ) {
		fprintf( stderr, "unable to open file %s: %s\n", in_xiso, strerror( errno ) );
		return 1;
	}

	cwd = getcwd( NULL, 0 );

	// change to extract directory
	if ( in_mode == kExtract && in_path ) {
		if ( chdir( in_path ) ) {
			fprintf( stderr, "unable to change to directory %s: %s\n", in_path, strerror( errno ) );
			if ( cwd ) free( cwd );
			exit( 1 );
		}
	}

	// strip any leading path information from the xiso name
	for ( name = &in_xiso[ len = strlen( in_xiso ) ]; name >= in_xiso && *name != '/'; --name ) ;	
	++name;
	
	log( "processing file %s:\n\n", name );
	
	// create a directory of the same name as the file we are working on, minus the "ISO" portion
	if ( ( len = strlen( name ) ) > 4 && name[ len - 4 ] == '.' &&
		( name[ len - 3 ] == 'i' || name[ len - 3 ] == 'I' ) &&
		( name[ len - 2 ] == 's' || name[ len - 2 ] == 'S' ) &&
		( name[ len - 1 ] == 'o' || name[ len - 1 ] == 'O' ) )
	{
		name[ len - 4 ] = 0;
		chop = true;
		len -= 4;
	}
	
	if ( ! len ) {
		fprintf( stderr, "invalid xiso image name: %s\n", in_xiso );
		err = 1;
	}
		
	if ( ! err ) {
		if ( in_mode == kExtract ) {
			if ( ( err = mkdir( name, 0755 ) ) ) fprintf( stderr, "unable to create directory %s: %s\n", name, strerror( errno ) );
			if ( ! err && ( err = chdir( name ) ) ) fprintf( stderr, "unable to change to directory %s: %s\n", name, strerror( errno ) );
		} else if ( in_mode == kUpload ) {
			if ( FtpMkdir( ftp, name ) < 0 ) { err = 1; fprintf( stderr, "unable to create remote directory %s\n", name ); }
			if ( ! err && FtpChdir( ftp, name ) < 0 ) { err = 1; fprintf( stderr, "unable to change to remote directory %s\n", name ); }
		}
	}
	
	// verify xiso header
	if ( ! err && lseek( xiso, XISO_HEADER_OFFSET, SEEK_SET ) == -1 ) err = 1;
	if ( ! err && read( xiso, buffer, XISO_HEADER_SIZE ) != XISO_HEADER_SIZE ) err = 1;
	if ( ! err && memcmp( buffer, XISO_HEADER_DATA, XISO_HEADER_SIZE ) ) {
		unchop();
		fprintf( stderr, "file %s does not appear to be a valid xbox iso image\n", name );
		err = 1;
	}
	
	// read root directory information
	if ( ! err && read( xiso, &root_dir_sect, XISO_SECTOR_OFFSET_SIZE ) != XISO_SECTOR_OFFSET_SIZE ) err = 1;
	if ( ! err && read( xiso, &root_dir_size, XISO_DIRTABLE_SIZE ) != XISO_DIRTABLE_SIZE ) err = 1;

	big32( root_dir_sect );
	big32( root_dir_size );
	
#if DEBUG && DUMP_SECTORS
	write_sector( xiso, (xoff_t) root_dir_sect * XISO_SECTOR_SIZE, root_dir_size, "root" );
#endif

	// seek to header tail and verify media tag
	if ( ! err && lseek( xiso, XISO_FILETIME_SIZE + XISO_UNUSED_SIZE, SEEK_CUR ) == -1 ) err = 1;
	if ( ! err && read( xiso, buffer, XISO_HEADER_SIZE ) != XISO_HEADER_SIZE ) err = 1;
	if ( ! err && memcmp( buffer, XISO_HEADER_DATA, XISO_HEADER_SIZE ) ) {
		unchop();
		fprintf( stderr, "file %s appears to be corrupt\n", name );
		err = 1;
	}

	// seek to root directory sector
	if ( ! err ) {
		if ( ! root_dir_sect && ! root_dir_size ) {
			unchop();
			log( "xbox image %s contains no files.\n", name );
		} else {
			if ( lseek( xiso, (xoff_t) root_dir_sect * XISO_SECTOR_SIZE, SEEK_SET ) == -1 ) err = 1;
						
			if ( in_path ) {
				path_len = strlen( in_path );
				if ( in_path[ path_len - 1 ] != '/' ) ++add_slash;
			}
			
			if ( ( buf = (char *) malloc( path_len + add_slash + strlen( name ) + 2 ) ) == NULL ) err = 1;
			
			if ( ! err ) {
				sprintf( buf, "%s%s%s/", in_path ? in_path : "", add_slash ? "/" : "", name );
				if ( ! err ) err = traverse_directory( xiso, (xoff_t) root_dir_sect * XISO_SECTOR_SIZE, buf, in_mode );
				free( buf );
			}
		}
	}
	
	if ( err ) {
		unchop();
		fprintf( stderr, "failed to extract xbox iso image %s\n", in_xiso );
	}
	
	if ( cwd ) {
		chdir( cwd );
		free( cwd );
	}
	
	close( xiso );
	unchop();

	#undef unchop

	return err;
}


#if DEBUG
void write_sector( int in_xiso, xoff_t in_start, unsigned long in_len, char *in_name ) {
	int				fp;
	ssize_t			wrote;
	xoff_t			curpos;
	char			buf[ 256 ], *sect;
	char		   *cwd = getcwd( NULL, 0 );

	chdir( "/Volumes/c/xbox/iso/exiso" );

	sprintf( buf, "%llu.%s", in_start, in_name );
	if ( ( fp = open( buf, WRITEFLAGS, 0644 ) ) != -1 ) {
		if ( ( curpos = lseek( in_xiso, 0, SEEK_CUR ) ) == -1 ) { fprintf( stderr, "unable to get current file position\n" ); return; }
		
		if ( lseek( in_xiso, in_start, SEEK_SET ) == -1 ) { fprintf( stderr, "unable to seek to offset %s\n", buf ); return; }
		if ( ( sect = (char *) malloc( in_len ) ) == NULL ) { fprintf( stderr, "unable to allocate memory\n" ); return; }
		
		if ( read( in_xiso, sect, in_len ) != in_len ) { fprintf( stderr, "unable to read sector\n", read ); return; }
		if ( ( wrote = write( fp, sect, in_len ) ) != in_len ) { fprintf( stderr, "unable to write sector (%u): %s\n", wrote, strerror( errno ) ); return; }
		
		if ( lseek( in_xiso, curpos, SEEK_SET ) == -1 ) { fprintf( stderr, "unable to seek to offset %llu\n", curpos ); return; }
		
		free( sect );
		close( fp );
	} else {
		fprintf( stderr, "error opening file %s for writing: %s\n", buf, strerror( errno ) );
	}

	chdir( cwd );
	free( cwd );
}
#endif


int traverse_directory( int in_xiso, xoff_t in_dir_start, char *in_path, modes in_mode ) {
	dir_node				dir;
	char				   *path;
	int						err = 0;
	unsigned short			l_offset, r_offset;
	
	memset( &dir, 0, sizeof(dir_node) );
	
	if ( read( in_xiso, &l_offset, XISO_TABLE_OFFSET_SIZE ) != XISO_TABLE_OFFSET_SIZE ) return 1;

	big16( l_offset );

	if ( l_offset == 0xffff ) return err_end_of_sector;

	if ( ! err && read( in_xiso, &r_offset, XISO_TABLE_OFFSET_SIZE ) != XISO_TABLE_OFFSET_SIZE ) err = 1;
	if ( ! err && read( in_xiso, &dir.start_sector, XISO_SECTOR_OFFSET_SIZE ) != XISO_SECTOR_OFFSET_SIZE ) err = 1;
	if ( ! err && read( in_xiso, &dir.file_size, XISO_FILESIZE_SIZE ) != XISO_FILESIZE_SIZE ) err = 1;
	if ( ! err && read( in_xiso, &dir.attributes, XISO_ATTRIBUTES_SIZE ) != XISO_ATTRIBUTES_SIZE ) err = 1;
	if ( ! err && read( in_xiso, &dir.filename_length, XISO_FILENAME_LENGTH_SIZE ) != XISO_FILENAME_LENGTH_SIZE ) err = 1;

	big16( r_offset );
	big32( dir.file_size );
	big32( dir.start_sector );

	if ( ! err && ( dir.filename = (char *) malloc( dir.filename_length + 1 ) ) == NULL ) err = 1;
	if ( ! err ) {
		if ( read( in_xiso, dir.filename, dir.filename_length ) != dir.filename_length ) err = 1;
		if ( ! err ) dir.filename[ dir.filename_length ] = 0;
	}

	if ( ! err && l_offset > 0 ) {
			if ( lseek( in_xiso, in_dir_start + l_offset * XISO_DWORD_SIZE, SEEK_SET ) == -1 ) err = 1;
			if ( ! err ) err = traverse_directory( in_xiso, in_dir_start, in_path, in_mode );
			if ( err == err_end_of_sector ) {
				l_offset = l_offset * XISO_DWORD_SIZE + ( XISO_SECTOR_SIZE - ( l_offset * XISO_DWORD_SIZE ) % XISO_SECTOR_SIZE );
				err = lseek( in_xiso, (xoff_t) in_dir_start + l_offset, SEEK_SET ) == -1 ? 1 : 0;
				if ( ! err ) err = traverse_directory( in_xiso, in_dir_start, in_path, in_mode );
			}
	}

	if ( ! err && dir.attributes & XISO_ATTRIBUTE_DIR && dir.start_sector && dir.file_size ) {
		if ( ( path = (char *) malloc( strlen( in_path ) + dir.filename_length + 2 ) ) == NULL ) err = 1;
		
		if ( ! err ) {
			sprintf( path, "%s%s/", in_path, dir.filename );
		
#if DEBUG && DUMP_SECTORS
			printf( "parsing directory tree for %s, sector is %llu, size is %u\n", dir.filename, (xoff_t) dir.start_sector * XISO_SECTOR_SIZE, dir.file_size );
			write_sector( in_xiso, (xoff_t) dir.start_sector * XISO_SECTOR_SIZE, dir.file_size, dir.filename );
#endif
			if ( lseek( in_xiso, (xoff_t) dir.start_sector * XISO_SECTOR_SIZE, SEEK_SET ) == -1 ) err = 1;
		}

		if ( ! err ) {
			if ( in_mode == kExtract ) {
				if ( ( err = mkdir( dir.filename, 0755 ) ) ) fprintf( stderr, "unable to make directory %s: %s\n", dir.filename, strerror( errno ) );
				if ( ! err && ( err = chdir( dir.filename ) ) ) fprintf( stderr, "unable to change to directory %s: %s\n", dir.filename, strerror( errno ) );
			} else if ( in_mode == kUpload ) {
				if ( FtpMkdir( ftp, dir.filename ) < 0 ) { err = 1; fprintf( stderr, "unable to create remote directory %s\n", dir.filename ); }
				if ( ! err && FtpChdir( ftp, dir.filename ) < 0 ) { err = 1; fprintf( stderr, "unable to change to remote directory %s\n", dir.filename ); }
			}
		}
		
		if ( ! err ) {
			err = traverse_directory( in_xiso, (xoff_t) dir.start_sector * XISO_SECTOR_SIZE, path, in_mode );

			if ( in_mode == kExtract && ( err = chdir( ".." ) ) ) fprintf( stderr, "could not change to parent directory\n" );
			if ( in_mode == kUpload && FtpChdir( ftp, ".." ) < 0 ) { err = 1; fprintf( stderr, "could not change to remote parent directory\n" ); }
		}

		if ( path ) free( path );
	} else {
		log( "%s%s%s (%u bytes)%s", in_mode == kExtract ? "extracting " : in_mode == kUpload ? "uploading " : "", in_path, dir.filename, dir.file_size , in_mode != kList ? " " : "" );
		if ( ! quiet ) fflush( stdout );
		if ( in_mode != kList && ( err = extract_file( in_xiso, &dir, in_mode ) ) ) { log( "failed!\n" ); }
		if ( ! err ) { log( "%s\n", in_mode != kList ? "[OK]" : "" ); }
		
		++total_files;
		++total_files_all_isos;
		total_bytes += dir.file_size;
		total_bytes_all_isos += dir.file_size;
	}

	if ( ! err && r_offset > 0 ) {
		if ( lseek( in_xiso, in_dir_start + r_offset * XISO_DWORD_SIZE, SEEK_SET ) == -1 ) err = 1;
		if ( ! err ) err = traverse_directory( in_xiso, in_dir_start, in_path, in_mode );
			if ( err == err_end_of_sector ) {
				r_offset = r_offset * XISO_DWORD_SIZE + ( XISO_SECTOR_SIZE - r_offset * XISO_DWORD_SIZE % XISO_SECTOR_SIZE );
				err = lseek( in_xiso, (xoff_t) in_dir_start + r_offset, SEEK_SET ) == -1 ? 1 : 0;
				if ( ! err ) err = traverse_directory( in_xiso, in_dir_start, in_path, in_mode );
			}
	}

	if ( ! err && lseek( in_xiso, in_dir_start, SEEK_SET ) == -1 ) err = 1;

	if ( dir.filename ) free( dir.filename );

	return err;
}


int extract_file( int in_xiso, dir_node *in_file, modes in_mode ) {
	int				out;
	unsigned long	i, size;
	int				err = 0;

	if ( in_mode == kExtract ) {
		if ( ( out = open( in_file->filename, WRITEFLAGS, 0644 ) ) == -1 ) err = 1;
	} else if ( in_mode == kUpload ) {
		if ( FtpOpenWrite( ftp, in_file->filename ) < 0 ) err = 1;
	} else err = 1;
	
	if ( ! err && lseek( in_xiso, (xoff_t) in_file->start_sector * XISO_SECTOR_SIZE, SEEK_SET ) == -1 ) err = 1;

	if ( ! err ) {
		if ( in_mode == kExtract ) {
			for ( i = 0, size = Min( in_file->file_size, EXTRACT_BUFFER_SIZE );
				  i < in_file->file_size && read( in_xiso, copy_buffer, size ) == size;
				  i += size, size = Min( in_file->file_size - i, EXTRACT_BUFFER_SIZE ) )
			{
				if ( write( out, copy_buffer, size ) != size ) {
					err = 1;
					break;
				}
			}
			
			close( out );
		} else {
			for ( i = 0, size = Min( in_file->file_size, EXTRACT_BUFFER_SIZE );
				  i < in_file->file_size && read( in_xiso, copy_buffer, size ) == size;
				  i += size, size = Min( in_file->file_size - i, EXTRACT_BUFFER_SIZE ) )
			{
				if ( FtpWriteBlock( ftp, copy_buffer, size ) != size ) {
					err = 1;
					break;
				}
			}
			
			FtpClose( ftp );
		}
	}
	
	return err;
}


int open_ftp_connection( char *in_host, char *in_user, char *in_password, FTP **out_ftp ) {
	STATUS			err = 0;
	
	log( "\nlogging in to ftp server %s... ", in_host );
	if ( ! quiet ) fflush( stdout );

	if ( FtpLogin( out_ftp, in_host, in_user, in_password, NULL ) < 0 ) err = 1;
	if ( ! err && FtpBinary( *out_ftp ) < 0 ) err = 1;

	log( "%s\n", err ? "failed!" : "[OK]" );

	if ( err && quiet ) fprintf( stderr, "unable to log in to ftp server %s\n", in_host );

	return err;
}
