unit IdURI;

{Details of implementation
-------------------------
2001-Nov Doychin Bondzhev
 - Fixes in URLEncode. There is difference when encoding Path+Doc and Params
2001-Oct-17 Peter Mee
 - Minor speed improvement - removed use of NormalizePath in SetURI.
 - Fixed bug that was cutting off the first two chars of the host when a
    username / password present.
 - Fixed bug that prevented username and password being updated.
 - Fixed bug that was leaving the bookmark in the document when no ? or =
    parameters existed.
2001-Feb-18 Doychin Bondzhev
 - Added UserName and Password to support URI's like
    http://username:password@hostname:port/path/document#bookmark
}

interface

Uses
  IdException;

type
  TIdURIOptionalFields = (ofAuthInfo, ofBookmark);
  TIdURIOptionalFieldsSet = set of TIdURIOptionalFields;

  TIdURI = class
  protected
    FDocument: string;
    FProtocol: string;
    FURI: String;
    FPort: string;
    Fpath: string;
    FHost: string;
    FBookmark: string;
    FUserName: string;
    FPassword: string;
    FParams: string;
    //
    procedure SetURI(const Value: String);
    function GetURI: String;
    procedure ParseURI(AURI: string; var VProtocol, VHost, VPath, VDocument, VParams, VPort, VBookmark,
      VUserName, VPassword: string); overload;
    procedure ParseURI(AURI: string; var VProtocol, VHost, VPath, VDocument, VParams, VPort, VBookmark
      : string); overload;
  public
    constructor Create(const AURI: string = ''); virtual;
    function GetFullURI(const AOptionalFileds: TIdURIOptionalFieldsSet = [ofAuthInfo, ofBookmark]): String;
    class procedure NormalizePath(var APath: string);
    class function URLDecode(ASrc: string): string;
    class function URLEncode(ASrc: string): string;
    class function ParamsEncode(ASrc: string): string;
    class function PathEncode(ASrc: string): string;    
    //
    property Bookmark : string read FBookmark write FBookMark;
    property Document: string read FDocument write FDocument;
    property Host: string read FHost write FHost;
    property Password: string read FPassword write FPassword;
    property Path: string read FPath write FPath;
    property Params: string read FParams write FParams;
    property Port: string read FPort write FPort;
    property Protocol: string read FProtocol write FProtocol;
    property URI: string read GetURI write SetURI;
    property Username: string read FUserName write FUserName;
  end;

  EIdURIException = class(EIdException);

implementation

uses
  IdGlobal, IdResourceStrings,
  SysUtils;

constructor TIdURI.Create(const AURI: string = '');
begin
  inherited Create;
  if length(AURI) > 0 then begin
    URI := AURI;
  end;
end;

class procedure TIdURI.NormalizePath(var APath: string);
var
  i: Integer;
begin
  // Normalize the directory delimiters to follow the UNIX syntax
  i := 1;
  while i <= Length(APath) do begin
    if APath[i] in LeadBytes then begin
      inc(i, 2)
    end else if APath[i] = '\' then begin
      APath[i] := '/';
      inc(i, 1);
    end else begin
      inc(i, 1);
    end;
  end;
end;

procedure TIdURI.ParseURI(AURI: string; var VProtocol, VHost, VPath, VDocument, VParams, VPort
 , VBookmark: string);
Var
  LUsername: String;
  LPassword: String;
begin
  ParseURI(AURI, VProtocol, VHost, VPath, VDocument, VParams, VPort, VBookmark, LUserName, LPassword);
end;

procedure TIdURI.ParseURI(AURI: string; var VProtocol, VHost, VPath,
  VDocument, VParams, VPort, VBookmark, VUserName, VPassword : string);
var
  LBuffer: string;
  LTokenPos, LPramsPos: Integer;
  LParams: String;
begin
  VHost := '';
  VProtocol := '';
  VPath := '';
  VDocument := '';
  VPort := '';
  VBookmark := '';
  VUsername := '';
  VPassword := '';
  NormalizePath(AURI);
  LTokenPos := IndyPos('://', AURI);
  if LTokenPos > 0 then begin
    // absolute URI
    // What to do when data don't match configuration ??
    // Get the protocol
    VProtocol := Copy(AURI, 1, LTokenPos  - 1);
    Delete(AURI, 1, LTokenPos + 2);
    // Get the user name, password, host and the port number
    LBuffer := Fetch(AURI, '/', True);
    // Get username and password
    LTokenPos := IndyPos('@', LBuffer);
    VPassword := Copy(LBuffer, 1, LTokenPos  - 1);
    if LTokenPos > 0 then
      Delete(LBuffer, 1, LTokenPos);
    VUserName := Fetch(VPassword, ':', True);
    // Ignore cases where there is only password (http://:password@host/pat/doc)
    if Length(VUserName) = 0 then VPassword := '';
    // Get the host and the port number
    VHost := Fetch(LBuffer, ':', True);
    VPort := LBuffer;
    // Get the path
    LPramsPos := IndyPos('?', AURI);
    if LPramsPos > 0 then begin // The case when there is parameters after the document name '?'
      LTokenPos := RPos('/', AURI, LPramsPos);
    end
    else begin
      LPramsPos := IndyPos('=', AURI);
      if LPramsPos > 0 then begin // The case when there is parameters after the document name '='
        LTokenPos := RPos('/', AURI, LPramsPos);
      end
      else begin
        LTokenPos := RPos('/', AURI, -1);
      end;
    end;

    VPath := '/' + Copy(AURI, 1, LTokenPos);
    // Get the document
    if LPramsPos > 0 then begin
      VDocument := Copy(AURI, 1, LPramsPos - 1);
      Delete(AURI, 1, LPramsPos - 1);
      LParams := AURI;
    end
    else
    VDocument := AURI;
    Delete(VDocument, 1, LTokenPos);

    VBookmark := VDocument;
    VDocument := Fetch(VBookmark, '#');
  end else begin
    // received an absolute path, not an URI
    LPramsPos := IndyPos('?', AURI);
    if LPramsPos > 0 then begin // The case when there is parameters after the document name '?'
      LTokenPos := RPos('/', AURI, LPramsPos);
    end else begin
      LPramsPos := IndyPos('=', AURI);
      if LPramsPos > 0 then begin // The case when there is parameters after the document name '='
        LTokenPos := RPos('/', AURI, LPramsPos);
      end else begin
        LTokenPos := RPos('/', AURI, -1);
      end;
    end;

    VPath := Copy(AURI, 1, LTokenPos);
    // Get the document
    if LPramsPos > 0 then begin
      VDocument := Copy(AURI, 1, LPramsPos - 1);
      Delete(AURI, 1, LPramsPos - 1);
      LParams := AURI;
    end else begin
      VDocument := AURI;
    end;
    Delete(VDocument, 1, LTokenPos);
  end;

  // Parse the # bookmark from the document
  if Length(VBookmark) = 0 then
  begin
    VBookmark := LParams;
    LParams := Fetch(VBookmark, '#');
  end;

  VParams := LParams;
end;

procedure TIdURI.SetURI(const Value: String);
begin
  FURI := Value;
  NormalizePath(FURI);
  ParseURI(FURI, FProtocol, FHost, FPath, FDocument, FParams, FPort, FBookmark,
    FUserName, FPassword);
end;

function TIdURI.GetURI: String;
begin
  FURI := GetFullURI;
  // result must contain only the proto://host/path/document
  // If you need the full URI then you have to call GetFullURI
  result := GetFullURI([]);
end;

class function TIdURI.URLDecode(ASrc: string): string;
var
  i: integer;
  ESC: string[2];
  CharCode: integer;
begin
  Result := '';
  ASrc := StringReplace(ASrc, '+', ' ', [rfReplaceAll]);  {do not localize}
  i := 1;
  while i <= Length(ASrc) do begin
    if ASrc[i] <> '%' then begin  {do not localize}
      Result := Result + ASrc[i]
    end else begin
      Inc(i); // skip the % char
      ESC := Copy(ASrc, i, 2); // Copy the escape code
      Inc(i, 1); // Then skip it.
      try
        CharCode := StrToInt('$' + ESC);  {do not localize}
        if (CharCode > 0) and (CharCode < 256) then begin
          Result := Result + Char(CharCode);
        end;
      except end;
    end;
    Inc(i);
  end;
end;

class function TIdURI.ParamsEncode(ASrc: string): string;
const
  UnsafeChars = ['*', '#', '%', '<', '>', '+'];  {do not localize}
var
  i: Integer;
begin
  Result := '';
  for i := 1 to Length(ASrc) do begin
    if ASrc[i] = ' ' then begin    {do not localize}
      Result := Result + '+';    {do not localize}
    end else if (ASrc[i] in UnsafeChars) or (ASrc[i] >= #$80) then begin
      Result := Result + '%' + IntToHex(Ord(ASrc[i]), 2);  {do not localize}
    end else begin
      Result := Result + ASrc[i];
    end;
  end;
end;

class function TIdURI.PathEncode(ASrc: string): string;
const
  UnsafeChars = ['*', '#', '%', '<', '>', '+', ' '];  {do not localize}
var
  i: Integer;
begin
  Result := '';
  for i := 1 to Length(ASrc) do begin
    if (ASrc[i] in UnsafeChars) or (ASrc[i] >= #$80) then begin
      Result := Result + '%' + IntToHex(Ord(ASrc[i]), 2);  {do not localize}
    end else begin
      Result := Result + ASrc[i];
    end;
  end;
end;

class function TIdURI.URLEncode(ASrc: string): string;
Var
  LURI: TIdURI;
begin
  LURI := TIdURI.Create(ASrc);
  try
    LURI.Path := PathEncode(LURI.Path);
    LURI.Document := PathEncode(LURI.Document);
    LURI.Params := ParamsEncode(LURI.Params);
  finally
    result := LURI.URI;
    LURI.Free;
  end;
end;

function TIdURI.GetFullURI(
  const AOptionalFileds: TIdURIOptionalFieldsSet): String;
Var
  LURI: String;
begin
  if Length(FProtocol) = 0 then
    raise EIdURIException.Create(RSURINoProto);

  LURI := FProtocol + '://';

  if (Length(FUserName) > 0) and (ofAuthInfo in AOptionalFileds) then begin
    LURI := LURI + FUserName;

    if Length(FPassword) > 0 then begin
      LURI := LURI + ':' + FPassword;
    end;

    LURI := LURI + '@';
  end;

  if Length(FHost) = 0 then
    raise EIdURIException.Create(RSURINoHost);
  LURI := LURI + FHost;
  if Length(FPort) > 0 then begin
    LURI := LURI + ':' + FPort;
  end;
  LURI := LURI + FPath + FDocument + FParams;
  if (Length(FBookmark) > 0) and (ofBookmark in AOptionalFileds) then begin
    LURI := LURI + '#' + FBookmark;
  end;
  result := LURI;
end;

end.
