package com.superhac.JXBStreamer.Core;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.SocketChannel;
import java.util.ArrayList;
import java.util.logging.Logger;

/**
 * Handles all the messaging for client communication with the server.
 * <p>
 * <p>
 * More information can be found at <a
 * href="http://www.superhac.com">Superhac.com</a>
 * 
 * @author Steven Scott (superhac007@gmail.com)
 */

public class XBMSPServerMessageHandler {

	/**
	 * the currentPath where the client is sitting in the file system. Will be
	 * null for the Virtual Root
	 */
	private File currentPath;

	/** All the Directory Handlers for this client * */
	private ArrayList<DirectoryHandlerObject> directoryHandlers = new ArrayList<DirectoryHandlerObject>();

	/** all the file handlers for this client */
	private ArrayList<FileHandlerObject> fileHandlers = new ArrayList<FileHandlerObject>();

	/** the directorys that make up the virtual root */
	private ArrayList<File> virtualRoot;

	/** a state var for checking if the client is at the virtual root * */
	private boolean atVirtualRoot = true;

	/** the socket channel for the client */
	private SocketChannel sc;

	/**
	 * The directory handler counter.. Handles the numbering of the handles.
	 * Always points at the next avaiable handler number
	 */
	private int directoryNextHandlerCount = 0;

	/**
	 * The file handler counter.. Handles the numbering of the handles. Always
	 * points at the next avaiable handler number
	 */
	private int fileNextHandlerCount = 10000;// start these at 10k

	/**
	 * a state varible for marking this a new connection. Used in sending the
	 * initial server response messages for new connections.
	 */
	private boolean newConnection = true;

	/** The connected clients software version */
	private String clientVersion = "Not Set";

	/** the last command the client sent to the server */
	private int lastClientCommand = 0;

	/** debug logger */
	private static Logger logger;

	/**
	 * last time the client transmitted a message to the server. used in check
	 * for in active clients.
	 */
	private long lastTransmissionTime;

	/**
	 * 
	 * @param sc
	 *            The socket channel for this client
	 * @param virtualRoot
	 *            the shared directorys for the virtual roots.
	 */
	protected XBMSPServerMessageHandler(SocketChannel sc,
			ArrayList<File> virtualRoot) {

		this.currentPath = null;
		this.sc = sc;
		this.virtualRoot = virtualRoot;

		// setup logger
		logger = com.superhac.JXBStreamer.Core.Debug.getLogger();

	}

	/**
	 * Processes the incoming messages
	 * 
	 * @param message
	 *            the buffer from socket.
	 */
	protected void IncommingMessage(ByteBuffer message) {

		// catch if we dropped the connection
		if (message.remaining() == 0) {
			if (com.superhac.JXBStreamer.Core.Debug.debug)
				logger
						.info("Message is empty... Probably disconnected client.");
			return;
		}

		// update the last transmission time
		lastTransmissionTime = System.currentTimeMillis();

		if (newConnection) // first packet is the version info
		{
			byte[] version = new byte[message.remaining()];
			int i = 0;
			for (i = 0; i < version.length; i++) {

				version[i] = message.get();
				if (version[i] == (char) 0x0A) // LF
					break;

			}
			clientVersion = new String(version, 0, i - 1); // string the
			// captured bytes -
			// the LF

			if (com.superhac.JXBStreamer.Core.Debug.debug)
				logger
						.info("New connection XBMSP client version packet. Client Version: "
								+ clientVersion);

			newConnection = false;

	
		}

		/**
		 * Sometimes the buffer is returned with 2 packets in one buffer.
		 * When this occurs we just process the rest of buffer. 
		 */
		while (message.hasRemaining())
		{
		XBMSPEncoderDecoder packet = new XBMSPEncoderDecoder();
		packet.decodepacket(message);
		handleMessage(packet);
		//System.out.println("More data? "+message.hasRemaining());
		}
	}

	/**
	 * Gets the socket channel that assigned to this message handler.
	 * 
	 * @return the socket channel
	 */
	protected SocketChannel getSocketChannel() {
		return sc;
	}

	/**
	 * Handles the message routing and the response to be sent to the client.
	 * Uses the socket channel to transmit the response packet.
	 * 
	 * @param decodedMessage
	 *            the decoded message
	 */
	private void handleMessage(XBMSPEncoderDecoder decodedMessage) {

		XBMSPEncoderDecoder response = null;
		boolean error = false;

		if (com.superhac.JXBStreamer.Core.Debug.debug)
			logger.info("Routing decoded packet by type: Type: "
					+ decodedMessage.getType());

		lastClientCommand = decodedMessage.getType();

		switch (decodedMessage.getType()) {

		/* server sends packet okay message */
		case XBMSPEncoderDecoder.XBMSP_PACKET_NULL:
			response = decodedMessage.encodePacketOk();
			break;
		case XBMSPEncoderDecoder.XBMSP_PACKET_SETCWD:
			error = changeDir(decodedMessage.getPayloadString());
			if (error)
				response = decodedMessage.encodePacketError(
						XBMSPEncoderDecoder.XBMSP_ERROR_FAILURE,
						"Invalid Directory Change");
			else
				response = decodedMessage.encodePacketOk();
			break;
		case XBMSPEncoderDecoder.XBMSP_PACKET_UPCWD:
			error = upDirectory(decodedMessage.getPayloadInt());
			if (error)
				response = decodedMessage.encodePacketError(
						XBMSPEncoderDecoder.XBMSP_ERROR_FAILURE,
						"Invalid Directory Change");
			else
				response = decodedMessage.encodePacketOk();
			break;
		case XBMSPEncoderDecoder.XBMSP_PACKET_FILELIST_OPEN:
			int handle = getDirectoryHandle();
			response = decodedMessage.encodePacketHandle(handle);
			break;
		case XBMSPEncoderDecoder.XBMSP_PACKET_FILELIST_READ:
			DirectoryHandlerObject handleObject = getDirectoryHandle(decodedMessage
					.getPayloadInt());
			if (handleObject == null)// invalid!
				response = decodedMessage.encodePacketError(
						XBMSPEncoderDecoder.XBMSP_ERROR_INVALID_HANDLE,
						"Invalid Handle");
			else {
				String[] strings = handleObject.getNextEntry();
				response = decodedMessage.encodePacketFileData(strings[0],
						strings[1]);
			}
			break;
		case XBMSPEncoderDecoder.XBMSP_PACKET_FILE_INFO:
			DirectoryHandlerObject dho = getCWDDirectoryHandler();
			if (dho != null) {
				int index = dho.getFileIndexNumber(decodedMessage
						.getPayloadString());
				if (index != -1) // file not found in directory
				{
					String[] strings = dho.getEntryDataByIndex(index);
					response = decodedMessage.encodePacketFileData(strings[0],
							strings[1]);
				} else {
					response = decodedMessage.encodePacketError(
							XBMSPEncoderDecoder.XBMSP_ERROR_NO_SUCH_FILE,
							"No such File...");
				}
			} else
				response = decodedMessage.encodePacketError(
						XBMSPEncoderDecoder.XBMSP_ERROR_NO_SUCH_FILE,
						"No such File..");
			break;
		case XBMSPEncoderDecoder.XBMSP_PACKET_FILE_OPEN:
			DirectoryHandlerObject dho2 = getCWDDirectoryHandler();
			if (dho2 != null) {
				int index = dho2.getFileIndexNumber(decodedMessage
						.getPayloadString());
				if (index == -1) // invalid file handler
					response = decodedMessage.encodePacketError(
							XBMSPEncoderDecoder.XBMSP_ERROR_INVALID_HANDLE,
							"Invalid Handle");
				else {
					File file = dho2.getFileByIndex(index);
					int handle2 = getFileHandle(file);
					response = decodedMessage.encodePacketHandle(handle2);
				}
			} else
				response = decodedMessage.encodePacketError(
						XBMSPEncoderDecoder.XBMSP_ERROR_NO_SUCH_FILE,
						"No such File or Directory");
			break;

		case XBMSPEncoderDecoder.XBMSP_PACKET_FILE_READ:

			FileHandlerObject fho = getFileHandler(decodedMessage
					.getPayloadInt());
			if (fho != null) {
				response = decodedMessage.encodePacketFileContents(fho
						.read(decodedMessage.getPayloadInt2()));
			} else {
				response = decodedMessage.encodePacketError(
						XBMSPEncoderDecoder.XBMSP_ERROR_INVALID_HANDLE,
						"Invalid Handle");
			}
			break;
		case XBMSPEncoderDecoder.XBMSP_PACKET_FILE_SEEK:
			FileHandlerObject fho2 = getFileHandler(decodedMessage
					.getPayloadInt());
			if (fho2 != null) {
				boolean error2 = fho2.seekFile(
						decodedMessage.getPayloadInt64(), decodedMessage
								.getPayloadByte());
				if (!error2) // seek was good
					response = decodedMessage.encodePacketOk();
				else
					// seek was bad
					response = decodedMessage.encodePacketError(
							XBMSPEncoderDecoder.XBMSP_ERROR_ILLEGAL_SEEK,
							"Illegal Seek!");
			} else {
				response = decodedMessage.encodePacketError(
						XBMSPEncoderDecoder.XBMSP_ERROR_INVALID_HANDLE,
						"Invalid Handle");
			}
			break;
		case XBMSPEncoderDecoder.XBMSP_PACKET_CLOSE:
			if (closeHandle(decodedMessage.getPayloadInt())) // removed!
				response = decodedMessage.encodePacketOk();
			else
				response = decodedMessage.encodePacketError(
						XBMSPEncoderDecoder.XBMSP_ERROR_INVALID_HANDLE,
						"Invalid Handle");

			break;

		case XBMSPEncoderDecoder.XBMSP_PACKET_CLOSE_ALL:
			closeAllHandles();
			response = decodedMessage.encodePacketOk();
			break;

		}

		// send the message out if not errored...or response is null
		if (response != null)
			try {
				if (com.superhac.JXBStreamer.Core.Debug.debug)
					logger.info("Sending response packet..");

				// get the bytebuffer that contains the packet
				ByteBuffer buf = response.getPacket();

				/**
				 * This you have to do because the bytebuffer you put in my not
				 * be fully writen to socket. THis will keep looping until it
				 * has been comepletly written out. *** HACK LESSON***
				 */
				while (buf.hasRemaining())
					sc.write(buf);

			} catch (IOException e) {
				System.out.println("ERROR: " + e);
			}
	}

	/**
	 * Handles the client's request for changing directories
	 * 
	 * @param newDirectory
	 *            the directory that the currentPath should be changed to.
	 * @return true if error occured and false if not
	 */
	private boolean changeDir(String newDirectory) {
		boolean error = false;
		boolean matched = false;

		// check for all the symbolic directory moves
		if (newDirectory == null) // just keep the same currentPath
		{
			if (com.superhac.JXBStreamer.Core.Debug.debug)
				logger
						.info("Client requested directory change of null.  Directory not changed.");
			matched = true;
			return error;
		} else if (newDirectory.startsWith("/", 0)) // set current path to root
		// path
		{
			if (com.superhac.JXBStreamer.Core.Debug.debug)
				logger.info("Client requested directory change to : <"
						+ newDirectory + ">.");

			atVirtualRoot = true;
			currentPath = null;
			return false;
		} else if (newDirectory.startsWith("..", 0)) // move up 1 level
		{ // if were already at root... stay at root
			if (com.superhac.JXBStreamer.Core.Debug.debug)
				logger.info("Client requested directory change to : <"
						+ newDirectory + ">.");
			if (atVirtualRoot) {
				return true;

			} else // move one directory up
			{
				System.out.println("Move up .. ran");

				File currentPathTemp = new File(currentPath.getParent());

				for (File root : virtualRoot) {
					// parent of the virtual root
					File parent = new File(root.getParent());
					if (parent.getAbsolutePath().compareTo(
							currentPathTemp.getAbsolutePath()) == 0) {
						atVirtualRoot = true;
						currentPath = null;
						break;
					}
					currentPath = currentPathTemp;
				}
				return false;
			}

		}

		if (atVirtualRoot) {
			for (File path : virtualRoot) {
				if ((path.getName().compareTo(newDirectory)) == 0) {
					currentPath = new File(path.getAbsolutePath());
					atVirtualRoot = false;
					matched = true;
					break;
				}

			}

		} else {
			// get the listing of the current directory
			String[] children = currentPath.list();

			// check the current directory for a matching directory
			for (int i = 0; i < children.length; i++) {
				if (children[i].compareTo(newDirectory) == 0) {
					String newPath = currentPath + "/" + newDirectory;

					if (new File(newPath).isDirectory()) {
						currentPath = new File(newPath); // set the new path
						// to
						// the currentpath
						atVirtualRoot = false;
						if (com.superhac.JXBStreamer.Core.Debug.debug)
							logger
									.info("Client requested directory change to : <"
											+ newDirectory + ">.");
						matched = true;
						break;
					}
				}
			}
		}

		if (!matched) // we did not match the newDirectory in the current path
			error = true;

		if (com.superhac.JXBStreamer.Core.Debug.debug)
			logger.info("ERROR: Client requested directory change to : <"
					+ newDirectory + ">.");
		return error;
	}

	/**
	 * Handles directory changes UP from currentPath by a number to move.
	 * 
	 * @param count
	 *            the number of directories to move UP
	 * @return true if successfull and false if not.
	 */
	private boolean upDirectory(int count) {

		// this needs to take into account the virutal ROOT! FIXME

		if (com.superhac.JXBStreamer.Core.Debug.debug)
			logger.info("Client requested directory change up " + count
					+ " levels.");

		boolean error = false;

		if (count == 0) // nop operation
			return false;
		else if (count == 0xFFFFFFFF) // move to root
		{
			currentPath = null;
			atVirtualRoot = true;
			return false;
		}

		// for all other numbers
		for (int i = 0; i < count; i++) {
			// were at root and the count is still greater... ERROR!
			if (atVirtualRoot)
				return true;

			File currentPathTemp = new File(currentPath.getParent());

			for (File root : virtualRoot) {
				// parent of the virtual root
				File parent = new File(root.getParent());
				if (parent.getAbsolutePath().compareTo(
						currentPathTemp.getAbsolutePath()) == 0) {
					atVirtualRoot = true;
					currentPath = null;
					break;
				}
				currentPath = currentPathTemp;
			}
			// compare new directoy to virutal root

		}

		return error;
	}

	/**
	 * Handle client requests for a Directory handle.
	 * 
	 * @return a handle
	 */
	private int getDirectoryHandle() {

		// if (com.superhac.JXBStreamer.Core.Debug.debug)
		// logger.info("Getting directory handle for <"
		// + currentPath.getAbsolutePath() + ">");

		int handle = 0;
		boolean match = false;

		// Check if handler already exists
		if (atVirtualRoot) {
			for (int i = 0; i < directoryHandlers.size(); i++) {
				if (directoryHandlers.get(i).getPath() == null) {
					handle = directoryHandlers.get(i).getHandle();
					match = true;
					if (com.superhac.JXBStreamer.Core.Debug.debug)
						logger
								.info("Directory handler already created.. Returning the handler.");
				}
			}
		} else {
			for (int i = 0; i < directoryHandlers.size(); i++) {
				if (directoryHandlers.get(i).getPath() != null)
					if (directoryHandlers.get(i).getPath().getName().compareTo(
							currentPath.getName()) == 0) {
						handle = directoryHandlers.get(i).getHandle();
						match = true;
						if (com.superhac.JXBStreamer.Core.Debug.debug)
							logger
									.info("Directory handler already created.. Returning the handler.");
					}
			}
		}

		// handle does not exisit create a new handlerObject
		if (!match) {

			if (com.superhac.JXBStreamer.Core.Debug.debug)
				logger.info("Directoy handler does not exist.. Creating one..");

			// create it
			if (atVirtualRoot) {
				DirectoryHandlerObject dirHandler = new DirectoryHandlerObject(
						directoryNextHandlerCount, virtualRoot);

				handle = dirHandler.getHandle();

				directoryHandlers.add(dirHandler);

				// increment handlerCount
				directoryNextHandlerCount++;

			} else {
				DirectoryHandlerObject dirHandler = new DirectoryHandlerObject(
						directoryNextHandlerCount, currentPath);
				handle = dirHandler.getHandle();

				directoryHandlers.add(dirHandler);

				// increment handlerCount
				directoryNextHandlerCount++;
			}

		}

		return handle;

	}

	/**
	 * Handle client requests for a file handle.
	 * 
	 * @param file
	 *            the File object for the requested handle.
	 * @return a handle
	 */
	private int getFileHandle(File file) {

		int handle = 0;
		boolean match = false;

		if (com.superhac.JXBStreamer.Core.Debug.debug)
			logger.info("Get File handler for <" + file.getName() + ">.");

		// Check if handler already exists
		for (int i = 0; i < fileHandlers.size(); i++) {
			if (fileHandlers.get(i).getFileName().compareTo(file.getName()) == 0) {
				handle = fileHandlers.get(i).getHandle();
				match = true;
				if (com.superhac.JXBStreamer.Core.Debug.debug)
					logger.info("File handler exist.. returning it.");
			}
		}

		// handle does not exisit create a new handlerObject
		if (!match) {
			if (com.superhac.JXBStreamer.Core.Debug.debug)
				logger.info("File handler does not exist.. Creating one."
						+ " Handler #:" + fileNextHandlerCount);
			// create it
			FileHandlerObject fileHandler = new FileHandlerObject(
					fileNextHandlerCount, file);
			handle = fileHandler.getHandle();

			fileHandlers.add(fileHandler);

			// increment handlerCount
			fileNextHandlerCount++;

		}

		return handle;

	}

	/**
	 * gets the current working DirectoryHandlerObject. e.g. currentPath.
	 * 
	 * @return the DirectoryHandlerObject for the currentPath
	 */
	private DirectoryHandlerObject getCWDDirectoryHandler() {

		if (com.superhac.JXBStreamer.Core.Debug.debug)
			logger.info("Geting CWD handler...");

		// just in case a handle to the current dir was not created.
		// XBMC does this without doing a change dir first on a new
		// connection...
		getDirectoryHandle();

		if (atVirtualRoot) {
			// match handle to virtual handle
			for (int i = 0; i < directoryHandlers.size(); i++) {

				// matched to a file to the virtual root
				if (directoryHandlers.get(i).getPath() == null)
					return directoryHandlers.get(i);

			}

		} else {
			// match handle to known handler
			for (int i = 0; i < directoryHandlers.size(); i++) {

				// the virutal root with have a path of null... we skip it
				if (directoryHandlers.get(i).getPath() != null)
					// matched to a file in the currentpath
					if (directoryHandlers.get(i).getPath().getName().compareTo(
							currentPath.getName()) == 0)
						return directoryHandlers.get(i);

			}
		}
		if (com.superhac.JXBStreamer.Core.Debug.debug)
			logger
					.info("ERROR: Can't find working directory handler.  Should not happen!");

		return null;
	}

	/**
	 * Gets the DirectoryHandlerObject for the specified handle.
	 * 
	 * @param handle
	 *            the handle for the directory
	 * @return DirectoryHandlerObject of the requested handle.
	 */
	private DirectoryHandlerObject getDirectoryHandle(int handle) {

		if (com.superhac.JXBStreamer.Core.Debug.debug)
			logger.info("Client directory read for handle #: " + handle);
		// match handle to known handler
		for (int i = 0; i < directoryHandlers.size(); i++) {
			if (directoryHandlers.get(i).getHandle() == handle)
				return directoryHandlers.get(i);
		}

		return null; // this is an invalid handle!

	}

	/**
	 * Gets the FileHandlerObject for the specified handle
	 * 
	 * @param handle
	 *            the handle number
	 * @return FileHandlerObject for the specified handle
	 */
	private FileHandlerObject getFileHandler(int handle) {

		// check for matching handle!

		// match handle to known handler
		for (int i = 0; i < fileHandlers.size(); i++) {
			if (fileHandlers.get(i).getHandle() == handle) {
				if (com.superhac.JXBStreamer.Core.Debug.debug)
					logger.info("Client file read for handle #: " + handle
							+ " filename:" + fileHandlers.get(i).getFileName());

				return fileHandlers.get(i);
			}

		}
		if (com.superhac.JXBStreamer.Core.Debug.debug)
			logger.info("Client file read for handle #: " + handle
					+ " is invalid!");
		return null; // this is an invalid handle!

	}

	/**
	 * Used when a client wants to close a handle.
	 * 
	 * @param handle
	 *            the handle to be closed.
	 * @return true if good, false if failed.
	 */
	private boolean closeHandle(int handle) {

		if (com.superhac.JXBStreamer.Core.Debug.debug)
			logger
					.info("Client request handle #: " + handle
							+ " to be closed.");

		// match File handle to known handler
		for (int i = 0; i < fileHandlers.size(); i++) {
			if (fileHandlers.get(i).getHandle() == handle) {
				fileHandlers.remove(i);
				return true;
			}
		}

		// match Directory handle to known handler
		for (int i = 0; i < directoryHandlers.size(); i++) {
			if (directoryHandlers.get(i).getHandle() == handle) {
				directoryHandlers.remove(i);
				return true;
			}
		}

		return false; // could not match!

	}

	/**
	 * Closes all open handles
	 * 
	 * @return true if good, false if failed.
	 */
	private boolean closeAllHandles() {

		if (com.superhac.JXBStreamer.Core.Debug.debug)
			logger.info("Client requested all handles" +
					" to be closed..");

		fileHandlers.removeAll(fileHandlers);
		directoryHandlers.removeAll(directoryHandlers);

		return true;
	}

	/**
	 * Gets the clients status for this message handler.
	 * 
	 * @return a ConnectedClientStatus
	 */
	public ConnectedClientStatus getStatus() {

		return new ConnectedClientStatus(sc.socket().getRemoteSocketAddress()
				.toString(), clientVersion, lastClientCommand, fileHandlers
				.size(), directoryHandlers.size(), getFileTransferRates());
	}

	/**
	 * Gets the file transfer rates for all open FileHandler objects.
	 * 
	 * @return the total number of bytes per second transfered.
	 */
	private long getFileTransferRates() {
		long fileTransferRate = 0;

		for (FileHandlerObject file : fileHandlers)
			fileTransferRate += file.getFileTransferRate();

		return fileTransferRate;
	}

	/**
	 * Gets the number of seconds since client last sent a message.
	 * 
	 * @return the number of seconds since the client sent a message.
	 */
	protected long getLastTransmissionDelay() {
		// in seconds!
		long delay = (System.currentTimeMillis() - lastTransmissionTime) / 1000;
		return delay;
	}

}
