#include "jaxstream.h"

#define do_log (strlen(logfile)>1)

FILE *lf=NULL;
int s=-1,done=0;
XSClient *clients=NULL;
int num_clients=0;
int do_debug=0, quiet=0, bg=0;
short port=XS_PORT;
char ip[16]=XS_IP, fromip[16]="", rootdir[1024]="",user[65]=XS_USER,group[65]=XS_GROUP,
	 logfile[1024]=XS_LOGFILE, pidfile[1024]=XS_PIDFILE, cfgfile[1024]=XS_CFGFILE;

XSCmd cmds[]={
	{ "*CAT", xs_cat },		// dir
	{ "RCAT", xs_cat },		// recursive dir?
	{ "BACK", xs_back },	// cd ..
	{ "OPEN", xs_open },	// open(file)
	{ "READ", xs_read },	// read(file)
	{ "TELL", xs_tell },	// tell(file)
	{ "CLSE", xs_clse },	// close(file)
	{ "DBUG", xs_dbug },	// debug (return "0 OK")
	{ "INFO", xs_unkn },	// depreciated? (return "-1 ERROR")
	{ NULL, NULL }
};

XSCfg cfg[]={
	{ "IP",			CFG_STRING,	ip,			16,		1 },
	{ "FROMIP",		CFG_STRING,	fromip,		16,		1 },
	{ "CFGFILE",	CFG_STRING,	cfgfile,	1024,	1 },
	{ "LOGFILE",	CFG_STRING,	logfile,	1024,	0 },
	{ "PIDFILE",	CFG_STRING,	pidfile,	1024,	0 },
	{ "ROOTDIR",	CFG_STRING,	rootdir,	1024,	0 },
	{ "USER",		CFG_STRING,	user,		65,		0 },
	{ "GROUP",		CFG_STRING,	group,		65,		0 },
	{ "QUIET",		CFG_INT,	&quiet,		4,		0 },
	{ "DEBUG",		CFG_INT,	&do_debug,	4,		1 },
	{ "BACKGROUND",	CFG_INT,	&bg,		4,		0 },
	{ "PORT",		CFG_SHORT,	&port,		2,		1 },
	{ NULL,			CFG_INVALID,NULL,		0,		0 }
};

void cleanup(int sig)
{
	if(sig)
		log("WE GET SIGNAL %d\n",sig);
	if(s!=-1)
		close(s);
	if(sig==SIGHUP)
	{
		readcfg(1);
		return;
	}
	remove(pidfile);
	log("Process exitting.\n");
	if(lf!=stdout && lf!=stderr && lf)
		fclose(lf);
	if(sig)
		exit(sig);
	s=-1;
}

char *fdgets(char *str, int max_len, int fd)
{
	char c;
	int i;

	debug("fdgets(%p,%d,%d)\n",str,max_len,fd);
	if(!str || !max_len)
		return NULL;
	i=0;
	do
	{
		if(read(fd,&c,1)<0)
		{
			perror("read");
			return NULL;
		}
		str[i++]=c;
		debug("%c",c);
	} while(c!='\n' && i<max_len-1);
	str[i]=0;
	debug("fdgets:\n%s\n",str);
	return str;
}

int fdNprintf(int fd, int N, char *format, ...)
{
	va_list ap;
	char *str, str0[1];
	int n;
	
	va_start(ap,format);
	if((n=vsnprintf(str0,1,format,ap))<=0)
		return n;
	if(n<32)
		n=N;
	else
		n++;
	str=malloc(n);
	str[n-1]=0;
	if(!str)
		return -1;
	n=vsnprintf(str,n,format,ap);
	while(n<N)
		str[n++]=' ';
	str[N-1]=0;
	va_end(ap);
	debug("fdNprintf:\n%s|\n",str);
	n=write(fd,str,n);
	free(str);
	return n;
}

int fdprintf(int fd, char *format, ...)
{
	va_list ap;
	char *str, str0[1];
	int n,i;
	
	va_start(ap,format);
	if((n=vsnprintf(str0,1,format,ap))<=0)
		return n;
	n++;
	str=malloc(n);
	if(!str)
	{
		perror("fdprintf:malloc");
		return -1;
	}
	n=vsnprintf(str,n,format,ap);
	va_end(ap);
	debug("fdprintf:\n%s\n",str);
	i=write(fd,str,n);
	if(i<0)
		perror("fdprintf:write");
	free(str);
	debug("fdprintf: wrote %d/%d bytes\n",i,n);
	return i;
}

int mprintf(char **dest, char *format, ...)
{
	va_list ap;
	char *str, str0[1];
	int n;
	
	va_start(ap,format);
	if((n=vsnprintf(str0,1,format,ap))<=0)
		return n;
	n++;
	str=malloc(n);
	if(!str)
		return -1;
	n=vsnprintf(str,n,format,ap);
	va_end(ap);
	free(*dest);
	*dest=str;
	return n;
}

char *strip(char *str, char *remove)
{
	while(*str && strchr(remove,str[strlen(str)-1]))
		str[strlen(str)-1]=0;
	return str;
}

char *chomp(char *str, char *remove)
{
	char *s;
	
	strip(str,remove);
	for(s=str;*s && strchr(remove,*s);s++);
	if(s && s!=str)
		memmove(str,s,strlen(s)+1);
	return str;
}

char *up(char *str)
{
	char *s;
	for(s=str;*s;s++)
		*s&=~0x20;
	return str;
}

void add_client(int sock, struct sockaddr_in *addr)
{
	XSClient *c;
	int i;
	unsigned long expected;
	
	if(sock<0)
		return;
	expected=inet_addr(fromip);
	if(strlen(fromip) && expected!=addr->sin_addr.s_addr)
	{
		log("Refused connection from %s:%hd (!=%s)\n",
				inet_ntoa(addr->sin_addr),ntohs(addr->sin_port),fromip);
		shutdown(sock,SHUT_RDWR);
		close(sock);
		return;
	}
	log("Accepted connection from %s:%hd\n",
			inet_ntoa(addr->sin_addr),ntohs(addr->sin_port));

	clients=realloc(clients,sizeof(XSClient)*(num_clients+1));
	
	c=&clients[num_clients];
	memset(c,0,sizeof(XSClient));
	
	i=1;
	setsockopt(sock, IPPROTO_TCP, TCP_NODELAY, (void*)&i, sizeof(i));
	i=2048;
	setsockopt(sock, IPPROTO_TCP, TCP_MAXSEG, (void*)&i, sizeof(i));
	
	c->sock=sock;
	memcpy(&c->addr,addr,sizeof(struct sockaddr_in));
	c->run=client_run_hello;
	c->fd=-1;
	
	num_clients++;
}

void free_xsclient(XSClient *c)
{
	if(!c)
		return;
	if(c->fd!=-1)
	{
		close(c->fd);
		c->fd=0;
	}
	if(c->sock)
	{
		shutdown(c->sock, SHUT_RDWR);
		close(c->sock);
		c->sock=0;
	}
}

void remove_client_ptr(XSClient *c)
{
	int i;
	for(i=0;i<num_clients;i++)
		if(c==&clients[i])
			remove_client(i);
}

void remove_client(int i)
{
	XSClient *c;
	
	if(i>=num_clients)
		return;
	c=&clients[i];
	log("Disconnecting from %s:%hd\n",
			inet_ntoa(c->addr.sin_addr),ntohs(c->addr.sin_port));
	free_xsclient(&clients[i]);
	num_clients--;
	if(i<num_clients)
		memmove(&clients[i],&clients[i+1],sizeof(XSClient)*(num_clients-i));
	if(num_clients)
		clients=realloc(clients,sizeof(XSClient)*num_clients);
	else
	{
		free(clients);
		clients=NULL;
	}
}

void redraw()
{
	int i;
	char cwd[1024];

#define EOL "\033[K"
	if(do_debug || quiet)
		return;
	getcwd(cwd,1024);
	printf("\033[H\033[1mJAXStream - "VERSION" : by LIM <jcatki@jonatkins.org>\033[0m"EOL"\n"
			"Serving at %s:%d as %s:%s%s%s"EOL"\n"
			"%-79.79s"EOL"\n"
			"\033[7mCn V IP/Command/File                                                            \033[0m"EOL"\n"
			,ip,port,user,group,
			strlen(fromip)?" only from ":"",
			strlen(fromip)?fromip:"",
			cwd);
	for(i=0;i<num_clients;i++)
	{
		XSClient *c=&clients[i];
		char *fname="";

		if(c->state&XS_CLIENT_PLAYING && c->fname)
		{
			char *s=strrchr(c->fname,'/');
			if(s)
				fname=s+1;
			else
				fname=c->fname;
		}
		
		printf("%2d %c %s:%hd\033[K\n",
				i,
				c->state&XS_CLIENT_VALID?'Y':'N',
				inet_ntoa(c->addr.sin_addr),
				ntohs(c->addr.sin_port));
		printf("     %.75s\033[K\n",
				c->cmd?c->cmd:"");
		if(c->state&XS_CLIENT_PLAYING)
			printf("     %3d%% %.70s\033[K\n",
					(int)(c->pos*100.0/(float)c->len),
					fname);
	}
	printf("\033[J");
	fflush(stdout);
}

void init(int inhup)
{
	int i;
	struct sockaddr_in addr;

	if(do_debug)
		lf=stderr;
	if(bg && !inhup)
	{
		pid_t pid;
		if((pid=fork())<0)
		{
			perror("fork");
			bg=0;
		}
		else
		{
			FILE *f;
			
			if(pid)
				exit(0);
			setsid();
			quiet=1;
			if(do_log)
			{
				lf=fopen(logfile,"a");
				if(!lf)
					perror(logfile);
				log("JAXStream - "VERSION" : by LIM <jcatki@jonatkins.org>\n");
				log("Process started.\n");
			}
			close(0);
			if(lf)
			{
				dup2(fileno(lf),1);
				dup2(fileno(lf),2);
			}
			else
			{
				close(1);
				close(2);
			}
			if(!(f=fopen(pidfile,"w")))
				perror(pidfile);
			else
			{
				fprintf(f,"%d",getpid());
				fclose(f);
				log("wrote pid to %s\n",pidfile);
			}
		}
	}
	{
		char cwd[1024];
		getcwd(cwd,1024);
		log("cwd=%.1024s\n",cwd);
	}
	signal(SIGINT, cleanup);
	signal(SIGTERM, cleanup);
	signal(SIGHUP, cleanup);
	if((s=socket(AF_INET, SOCK_STREAM, 0))<0)
	{
		perror("socket");
		exit(2);
	}
	i=1;
	setsockopt(s, SOL_SOCKET, SO_REUSEADDR, (void*)&i, sizeof(i));
	setsockopt(s, IPPROTO_TCP, TCP_NODELAY, (void*)&i, sizeof(i));

	addr.sin_family = AF_INET;
	addr.sin_addr.s_addr = inet_addr(ip);
	addr.sin_port = htons(port);
	if(bind(s,(struct sockaddr*)&addr,sizeof(addr))<0)
	{
		perror("bind");
		exit(3);
	}
	if(listen(s,5)<0)
	{
		perror("listen");
		exit(4);
	}
	debug("listening on %s:%d\n",ip,port);
	if(!bg && !quiet && !do_debug)
		printf("\033[H\033[J");
}

void readcfg(int inhup)
{
	FILE *f;
	int i;
	char line[4096],*s;
	XSCfg *c;
	
	if(!(f=fopen(cfgfile,"r")))
		perror(cfgfile);
	else
	{
		while(fgets(line,4096,f))
		{
			chomp(line,"\n\r \t");
			//debug("readcfg: '%s'\n",line);
			if((s=strchr(line,'#')))
				*s=0;
			if(!strlen(line))
				continue;
			if((s=strchr(line,'=')))
			{
				*s=0;
				s++;
				chomp(s,"\n\r \t");
				if(*s=='"')
				{
					char *e=strrchr(s,'"');
					if(e && e>s)
					{
						*e=0;
						s++;
					}
				}
			}
			else
				s=0;
			strip(line,"\n\r \t");
			debug("readcfg: opt='%s' arg='%s'\n",line,s?s:"(null)");
			if(!s)
				continue;
			up(line);
			for(i=0;cfg[i].opt && strcmp(line,cfg[i].opt); i++);
			c=&cfg[i];
			if(!c->opt || (inhup && !c->onhup))
				continue;
			debug("readcfg: cfg[%d] %s, %d, %p, %d\n",i,c->opt,c->type,c->var,c->len);
			switch(c->type)
			{
				case CFG_SHORT:
					*((short*)c->var)=atoi(s);
					break;
				case CFG_INT:
					*((int*)c->var)=atoi(s);
					break;
				case CFG_STRING:
					if(strlen(s)<c->len)
						strncpy(c->var,s,c->len);
					break;
				default:
					break;
			}
		}
		fclose(f);
	}
	if(inhup)
		init(inhup);
}

////////////////////////////////////////////////////////////////////////////////

int main(int argc, char **argv)
{
	int i,c,n;
	struct sockaddr_in addr;
	
	readcfg(0);
	while((c=getopt(argc,argv,"c:bd:Dqi:p:f:hu:g:l:P:"))>0)
	{
		debug("option: %c",c)
		switch(c)
		{
			case 'c':
				strncpy(cfgfile,optarg,1024);
				debug("=%s",cfgfile);
				readcfg(0);
				break;
			case 'D':
				do_debug=!do_debug;
				if(do_debug)
					bg=0;
				break;
			case 'q':
				quiet=!quiet;
				break;
			case 'd':
				strncpy(rootdir,optarg,1024);
				debug("=%s",rootdir);
				break;
			case 'u':
				strncpy(user,optarg,65);
				debug("=%s",user);
				break;
			case 'g':
				strncpy(group,optarg,65);
				debug("=%s",group);
				break;
			case 'i':
				strncpy(ip,optarg,16);
				debug("=%s",ip);
				break;
			case 'p':
				port=atoi(optarg);
				debug("=%hd",port);
				break;
			case 'f':
				strncpy(fromip,optarg,16);
				debug("=%s",fromip);
				break;
			case 'l':
				strncpy(logfile,optarg,16);
				debug("=%s",logfile);
				break;
			case 'P':
				strncpy(pidfile,optarg,16);
				debug("=%s",pidfile);
				break;
			case 'b':
				bg=!bg;
				if(bg)
					do_debug=0;
				break;
			case 'h':
			default:
				fprintf(stderr,"JAXStream - "VERSION" : by LIM <jcatki@jonatkins.org>\n"
					   "%s [-Dbhq] [-i bindIP] [-p port] [-f fromIP] [-d path] [-c file]\n"
					   "     [-l file] [-P file] [-u user] [-g group]\n"
					   "  -i bindIP  IP Address to bind to (default="XS_IP")\n"
					   "  -p port    port to bind to (default=%hd)\n"
					   "  -f fromIP  IP Address allowed to connect (default=blank)\n"
					   "  -c file    config filename (default="XS_CFGFILE")\n"
					   "  -l file    log filename (default="XS_LOGFILE")\n"
					   "  -P file    pid filename (default="XS_PIDFILE")\n"
					   "  -d path    chroot path\n"
					   "  -u user    username to run as\n"
					   "  -g group   groupname to run as\n"
					   "  -b         daemonize (fork into background)\n"
					   "  -D         debug\n"
					   "  -q         quiet\n"
					   "  -h         this help\n"
					   ,argv[0],XS_PORT);
				return 0;
		}
		debug("\n");
	}
	init(0);
	{
		struct passwd *u;
	    struct group *g;

		if(!(u=getpwnam(user)))
		{
			if(errno)
				perror(user);
			else
				fprintf(stderr,"%s is not a valid username!",user);
			return 5;
		}
		if(setuid(u->pw_uid)<0)
		{
			perror(user);
			return 6;
		}
		if(!(g=getgrnam(group)))
		{
			if(errno)
				perror(group);
			else
				fprintf(stderr,"%s is not a valid username!",user);
			return 6;
		}
		if(setgid(g->gr_gid)<0)
		{
			perror(group);
			return 6;
		}
	}
	if(strlen(rootdir))
	{
		debug("chroot to %s\n",rootdir);
		if(chdir(rootdir)<0 || chroot(rootdir)<0)
		{
			perror(rootdir);
			return 1;
		}
	}
	while(!done)
	{
		fd_set fds;
		int max_fd;
		
		redraw();
		
		FD_ZERO(&fds);
		FD_SET(s,&fds);
		max_fd=s;
		for(i=0; i<num_clients; i++)
		{
			FD_SET(clients[i].sock, &fds);
			if(clients[i].sock>max_fd)
				max_fd=clients[i].sock;
		}
		
		debug("selecting on %d clients and server...\n",num_clients);
		if((n=select(max_fd+1,&fds,0,0,0))<0)
		{
			perror("select");
		}
		else if(n)
		{
			if(FD_ISSET(s,&fds))
			{
				debug("server accept...\n");
				i=sizeof(addr);
				c=accept(s,(struct sockaddr*)&addr,&i);
				if(c>=0 && i==sizeof(addr))
					add_client(c,&addr);
				else
					perror("accept");
				n--;
			}
			for(i=0; n && i<num_clients; i++)
			{
				if(FD_ISSET(clients[i].sock,&fds))
				{
					debug("client #%d run...\n",i);
					if(clients[i].run(&clients[i])<0)
						remove_client(i);
					n--;
				}
			}
		}
	}
	close(s);
	return 0;
}

////////////////////////////////////////////////////////////////////////////////

int client_run_hello(XSClient *c)
{
	char line[XS_LEN];

	debug("hello?\n");
	if(!fdgets(line,XS_LEN,c->sock))
	{
		perror("client_run_hello:fgets");
		return -1;
	}
	strip(line,"\r\n ");
	strncpy(c->cmd,line,XS_LEN);
	c->cmd[XS_LEN-1]=0;
	if(strcmp(line,XS_HELLO0))
	{
		debug("invalid hello: %s\n",line);
		return -1;
	}
	if(!fdprintf(c->sock,XS_HELLO1))
	{
		perror("client_run_hello:fdprintf");
		return -1;
	}
	c->run=client_run_xstream_60;
	c->state|=XS_CLIENT_VALID;
	return 0;
}

int client_run_xstream_60(XSClient *c)
{
	char line[XS_LEN];
	int i;
	
	memset(line,0,XS_LEN);
	//if(!fdgets(line,XS_LEN,c->sock))
	if((i=read(c->sock,line,XS_LEN))<=0)
	{
		perror("client_run_hello:read");
		return -1;
	}
	//strip(line,"\r\n ");
	strncpy(c->cmd,line,XS_LEN);
	c->cmd[XS_LEN-1]=0;
	debug("client_run_xstream_60: parsing: %s\n",line);
	for(i=0; cmds[i].str; i++)
		if(!strncmp(line, cmds[i].str, 4))
			return cmds[i].func(c,line);
	debug("client_run_xstream_60: invalid commandline '%s'",line);
	fdprintf(c->sock,XS_ERROR);
	return 0;
}

int xs_cat(struct XSClient *c, char *line)
{
	int n,i,fd,attr;
	struct dirent **des;
	char path[XS_LEN], fname[XS_LEN], *msg=NULL;
	struct stat st;
	
	line=strchr(line,',');
	if(line)
	{
		line++;
		if(!strcmp(line,XS_CAT_BACK))
		{
			chdir("..");
			line+=4;
		}
		else if(!strcmp(line,XS_CAT_ROOT))
		{
			chdir("/");
			line+=6;
		}
		else if(*line==XS_CAT_CURRENT)
		{
			line++;
		}
		if(strlen(line))
			chdir(line);
	}
	getcwd(path,XS_LEN);
	if(path[strlen(path)-1]!='/')
		strcat(path,"/");
	n=scandir(path, &des, 0, alphasort);
	if(n<0)
	{
		perror(path);
		fdprintf(c->sock,XS_ERROR_UNKNOWN);
		log("%s:%hd LIST FAILED %s\n",
				inet_ntoa(c->addr.sin_addr),ntohs(c->addr.sin_port),line);
		return 0;
	}
	log("%s:%hd LIST %s\n",
			inet_ntoa(c->addr.sin_addr),ntohs(c->addr.sin_port),path);
	mprintf(&msg,XS_SHARES0);
	for(i=0; i<n; i++)
	{
		if(*des[i]->d_name=='.')
			continue;
		snprintf(fname, XS_LEN, "%s%s", path, des[i]->d_name);
		if(stat(fname,&st)<0)
		{
			perror(fname);
			continue;
		}
		attr=0;
		if(S_ISDIR(st.st_mode))
			attr|=XS_ATTR_DIR;
		if(S_ISREG(st.st_mode))
			attr|=XS_ATTR_FILE;
		if(attr && (fd=open(fname,O_RDONLY))!=-1)
		{
			close(fd);
			mprintf(&msg,"%s"XS_ITEM0"%d"XS_ITEM1"%s"XS_ITEM2,msg, attr, fname);
		}
		free(des[i]);
	}
	free(des);
	mprintf(&msg,"%s%s",msg,XS_SHARES1);
	fdNprintf(c->sock,XS_SLEN,"%d",strlen(msg));
	debug("%s\n",msg);
	i=write(c->sock,msg,strlen(msg));
	free(msg);
	return i<0?-1:0;
}

int xs_back(struct XSClient *c, char *line)
{
	chdir("..");
	fdNprintf(c->sock,XS_SLEN,"0 OK");
	return 0;
}

int xs_open(struct XSClient *c, char *line)
{
	off_t len;
	line=strchr(line,',');
	if(!line)
	{
		fdNprintf(c->sock,XS_SLEN,XS_ERROR);
		return 0;
	}
	if(c->fd!=-1)
		close(c->fd);
	line++;
	c->fd=open(line,O_RDONLY);
	strncpy(c->fname,line,XS_LEN);
	c->fname[XS_LEN-1]=0;
	if(c->fd<0)
	{
		log("%s:%hd OPEN FAILED %s\n",
				inet_ntoa(c->addr.sin_addr),ntohs(c->addr.sin_port),line);
		fdNprintf(c->sock,XS_SLEN,XS_ERROR_OPEN);
		return 0;
	}
	log("%s:%hd OPEN %s\n",
			inet_ntoa(c->addr.sin_addr),ntohs(c->addr.sin_port),line);
	len=lseek(c->fd,0,SEEK_END); // this can fail...?
	if(len<0)
		perror("lseek(beginning)");
	if(lseek(c->fd,0,SEEK_SET)<0)
		perror("lseek(beginning)");
	c->state|=XS_CLIENT_PLAYING;
	c->len=len;
	c->pos=0;
	fdNprintf(c->sock,XS_SLEN,"%ld",len);
	return 0;
}

int xs_read(struct XSClient *c, char *line)
{
	off_t pos=-1;
	unsigned long len=0,n;

	line+=5;
	n=sscanf(line,"%ld,%lu",&pos,&len);
	debug("scanned %ld elements\n",n);
	if(c->fd==-1 || !len || pos==-1 || n!=2)
	{
		fdNprintf(c->sock,XS_SLEN,XS_ERROR);
		return 0;
	}
	pos=lseek(c->fd,pos,SEEK_SET);
	if(pos==-1)
		fdNprintf(c->sock,XS_SLEN,"0");
	else
	{
		unsigned char buf[len];

		c->pos=pos;
		n=read(c->fd,buf,len);
		if(n<0)
		{
			perror("xs_read:read");
			fdNprintf(c->sock,XS_SLEN,XS_ERROR_READ);
			return 0;
		}
		fdNprintf(c->sock,XS_SLEN,"%lu",n);
		c->pos+=n;
		if(write(c->sock,buf,n)<0)
		{
			perror("xs_read:write");
			return -1;
		}
	}
	return 0;
}

int xs_tell(struct XSClient *c, char *line)
{
	off_t pos;
	
	pos=lseek(c->fd,0,SEEK_CUR);
	if(pos<0)
		fdNprintf(c->sock,XS_SLEN,XS_ERROR_TELL);
	else
	{
		fdNprintf(c->sock,XS_SLEN,"%ld",pos);
		c->pos=pos;
	}
	return 0;
}

int xs_clse(struct XSClient *c, char *line)
{
	log("%s:%hd CLOSE\n",
			inet_ntoa(c->addr.sin_addr),ntohs(c->addr.sin_port));
	remove_client_ptr(c);
	return 0;
}

int xs_dbug(struct XSClient *c, char *line)
{
	fdNprintf(c->sock,XS_SLEN,"0 OK");
	return 0;
}

int xs_unkn(struct XSClient *c, char *line)
{
	fdNprintf(c->sock,XS_SLEN,XS_ERROR_UNKNOWN);
	return 0;
}
