package com.superhac.JXBStreamer.Core;

import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.util.logging.Logger;

/**
 * This handles all the XMBSP packet Decoding and Encoding. This is used by the
 * XBMSPServerMessageHandler's to decode and code messages.
 * <p>
 * <p>
 * The protocol is documented <a
 * href="http://multivac.fatburen.org/localdoc/ccxstream/xbmsp.txt">here</a>.
 * <p>
 * More information can be found at <a
 * href="http://www.superhac.com">Superhac.com</a>
 * 
 * @author Steven Scott (superhac007@gmail.com)
 */

public class XBMSPEncoderDecoder {

	/* Indentifier message on inital connection */
	protected static final String SERVER_VERSION = "XBMSP-1.0 1.0 JXBStreamer Media Server 1.0\n";

	/* Packet types (e.g. type field) for CLIENT's */
	protected static final byte XBMSP_PACKET_NULL = 10; // no payload

	protected static final byte XBMSP_PACKET_SETCWD = 11;

	protected static final byte XBMSP_PACKET_FILELIST_OPEN = 12; // no payload

	protected static final byte XBMSP_PACKET_FILELIST_READ = 13;

	protected static final byte XBMSP_PACKET_FILE_INFO = 14;

	protected static final byte XBMSP_PACKET_FILE_OPEN = 15;

	protected static final byte XBMSP_PACKET_FILE_READ = 16;

	protected static final byte XBMSP_PACKET_FILE_SEEK = 17;

	protected static final byte XBMSP_PACKET_CLOSE = 18;

	protected static final byte XBMSP_PACKET_CLOSE_ALL = 19; // no payload

	protected static final byte XBMSP_PACKET_SET_CONFIGURATION_OPTION = 20;

	protected static final byte XBMSP_PACKET_AUTHENTICATION_INIT = 21;

	protected static final byte XBMSP_PACKET_AUTHENTICATE = 22;

	protected static final byte XBMSP_PACKET_UPCWD = 23;

	/* Packet types (e.g. type field) for Server */
	protected static final byte XBMSP_PACKET_OK = 1;

	protected static final byte XBMSP_PACKET_ERROR = 2;

	protected static final byte XBMSP_PACKET_HANDLE = 3;

	protected static final byte XBMSP_PACKET_FILE_DATA = 4;

	protected static final byte XBMSP_PACKET_FILE_CONTENTS = 5;

	protected static final byte XBMSP_PACKET_AUTHENTICATION_CONTINUE = 6;

	/** Packet types (e.g. type field) for Server discovery protocol */
	protected static final byte XBMSP_PACKET_SERVER_DISCOVERY_QUERY = 90;

	protected static final byte XBMSP_PACKET_SERVER_DISCOVERY_REPLY = 91;

	/* Reponse ERROR CODES */

	protected static final byte XBMSP_ERROR_OK = 0; /* Reserved */

	protected static final byte XBMSP_ERROR_FAILURE = 1;

	protected static final byte XBMSP_ERROR_UNSUPPORTED = 2;

	protected static final byte XBMSP_ERROR_NO_SUCH_FILE = 3;

	protected static final byte XBMSP_ERROR_INVALID_FILE = 4;

	protected static final byte XBMSP_ERROR_INVALID_HANDLE = 5;

	protected static final byte XBMSP_ERROR_OPEN_FAILED = 6;

	protected static final byte XBMSP_ERROR_TOO_MANY_OPEN_FILES = 7;

	protected static final byte XBMSP_ERROR_TOO_LONG_READ = 8;

	protected static final byte XBMSP_ERROR_ILLEGAL_SEEK = 9;

	protected static final byte XBMSP_ERROR_OPTION_IS_READ_ONLY = 10;

	protected static final byte XBMSP_ERROR_INVALID_OPTION_VALUE = 11;

	protected static final byte XBMSP_ERROR_AUTHENTICATION_NEEDED = 12;

	protected static final byte XBMSP_ERROR_AUTHENTICATION_FAILED = 13;

	/* seek types for FILE_SEEK */
	protected static final byte XBMSP_FILE_SEEK_TYPE_FWDFROMBEG = 0;

	protected static final byte XBMSP_FILE_SEEK_TYPE_BCKFROMEND = 1;

	protected static final byte XBMSP_FILE_SEEK_TYPE_FWDFROMCUR = 2;

	protected static final byte XBMSP_FILE_SEEK_TYPE_BCKFROMCUR = 3;

	/* Fields within the packet */

	/** Message length of msglen field. excludes the length field itself!! */
	private int msgLen;

	/** Message Type of msgType field. */
	private byte type;

	/**
	 * Message ID of the msgID field. msg identifier to keep req / resp insync,
	 * cannot use 0x00000000 OR 0xFFFFFFFF
	 */
	private int id;

	/** complete raw payload of Decoding message */
	private ByteBuffer payload;

	/** complete raw payload of Encoded message */
	private ByteBuffer packet;

	/**
	 * payload specific fields: used in the decoding process for the DATA part
	 * of the packet.
	 */
	private String payloadString;

	/**
	 * payload specific fields: used in the decoding process for the DATA part
	 * of the packet.
	 */
	private String payloadString2;

	/**
	 * payload specific fields: used in the decoding process for the DATA part
	 * of the packet.
	 */
	private int payloadInt;

	/**
	 * payload specific fields: used in the decoding process for the DATA part
	 * of the packet.
	 */
	private int payloadInt2;

	/**
	 * payload specific fields: used in the decoding process for the DATA part
	 * of the packet.
	 */
	private long payloadInt64;

	/**
	 * payload specific fields: used in the decoding process for the DATA part
	 * of the packet.
	 */
	private byte payloadByte;

	/**
	 * payload specific fields: used in the decoding process for the DATA part
	 * of the packet.
	 */
	private byte[] payloadData;

	/*
	 * Static header size in bytes (msglen, TYPE, ID) -- excludes the length
	 * field itself
	 */
	private int headerSize = 5;

	/** debug logger */
	private static Logger logger;

	protected XBMSPEncoderDecoder() {
		// if (com.superhac.JXBStreamer.Core.Debug.debug)
		logger = com.superhac.JXBStreamer.Core.Debug.getLogger();
	}

	/**
	 * Call this after the message is ready to be transmitted.
	 * 
	 * @return a buffer containing the encoded packet.
	 */
	protected ByteBuffer getPacket() {
		if (com.superhac.JXBStreamer.Core.Debug.debug)
			logger.info("Packing packet for transmission..");

		// calculate the size of the packet
		if (payload != null) {

			payload.flip();
			this.msgLen = headerSize + payload.remaining();
		} else
			this.msgLen = headerSize;
		// System.out.println("msglen: "+this.msgLen);

		// encode the message
		this.packet = ByteBuffer.allocate(this.msgLen + 4); // + 4 for the
		// msglen field

		// header
		packet.putInt(this.msgLen);
		packet.put(this.type);
		packet.putInt(this.id);

		if (payload != null)
			packet.put(payload);

		// flip the buffer 
		packet.flip();

		return packet;

	}

	/**
	 * Gets the msgID of the packet. (msgID)
	 * 
	 * @return msgID
	 */
	private int getId() {
		return id;
	}

	/**
	 * Sets the message ID of the packet (msgID)
	 * 
	 * @param id
	 */
	private void setId(int id) {
		this.id = id;
	}

	/**
	 * Sets the message length of the packet. (msgLen)
	 * 
	 * @param msgLen
	 */
	private void setMsgLen(int msgLen) {
		this.msgLen = msgLen;
	}

	/**
	 * Returns the packet type (type field)
	 * 
	 * @return
	 */
	protected byte getType() {
		return type;
	}

	/**
	 * sets the type
	 * 
	 * @param type
	 */
	private void setType(byte type) {
		this.type = type;
	}

	/**
	 * Takes the socket buffer thats passed in and decodes the message.
	 * 
	 * @param buff
	 *            the Buffer from the socket.
	 */
	protected void decodepacket(ByteBuffer buff) // this should throw an
	// exception when the packet
	// cannot be decoded
	{

		if (com.superhac.JXBStreamer.Core.Debug.debug)
			logger.info("Decoding packet.. Socket Buffer size: "
					+ buff.remaining());

		/** couldn's match packet type */
		// boolean typeNotIdentified = false;
		// start decoding header
		this.msgLen = buff.getInt();
		this.type = buff.get();
		this.id = buff.getInt();

		if (this.msgLen == headerSize) // the size of the header only or
			// greater then packet size
			return; // theres not payload or the msglen is greater then packet
		// size! e.g. XBMSP_PACKET_NULL

		if (com.superhac.JXBStreamer.Core.Debug.debug)
			logger
					.info("Decoding packet.. Socket Buffer size after XBMSP header: "
							+ buff.remaining());

		// allocate the payload
		this.payload = ByteBuffer.allocate(msgLen); // the size

		while (buff.hasRemaining()) {
			this.payload.put(buff.get());
		}

		this.payload.flip(); // flip for reading!

		// start the parsing of the payload
		switch (this.type) {
		case XBMSP_PACKET_SETCWD: // change dir
			if (com.superhac.JXBStreamer.Core.Debug.debug)
				logger.info("<XBMSP_PACKET_SETCWD Decoded>");
			this.payloadString = decodePayloadString();
			break;
		case XBMSP_PACKET_UPCWD:
			if (com.superhac.JXBStreamer.Core.Debug.debug)
				logger.info("<XBMSP_PACKET_UPCWD>");
			this.payloadInt = decodePayloadInt();
			break;
		case XBMSP_PACKET_FILELIST_READ:
			if (com.superhac.JXBStreamer.Core.Debug.debug)
				logger.info("<XBMSP_PACKET_FILELIST_READ>");
			this.payloadInt = decodePayloadInt();
			break;
		case XBMSP_PACKET_FILE_INFO:
			if (com.superhac.JXBStreamer.Core.Debug.debug)
				logger.info("<XBMSP_PACKET_FILE_INFO>");
			this.payloadString = decodePayloadString();
			break;
		case XBMSP_PACKET_FILE_OPEN:
			if (com.superhac.JXBStreamer.Core.Debug.debug)
				logger.info("<XBMSP_PACKET_FILE_OPEN>");
			this.payloadString = decodePayloadString();
			break;
		case XBMSP_PACKET_FILE_READ:
			if (com.superhac.JXBStreamer.Core.Debug.debug)
				logger.info("<XBMSP_PACKET_FILE_READ>");
			this.payloadInt = decodePayloadInt();
			this.payloadInt2 = decodePayloadInt();
			break;
		case XBMSP_PACKET_FILE_SEEK:
			if (com.superhac.JXBStreamer.Core.Debug.debug)
				logger.info("<XBMSP_PACKET_FILE_SEEK>");
			this.payloadInt = decodePayloadInt();
			this.payloadByte = decodePayloadByte();
			this.payloadInt64 = decodePayloadInt64();
			break;
		case XBMSP_PACKET_CLOSE:
			if (com.superhac.JXBStreamer.Core.Debug.debug)
				logger.info("<XBMSP_PACKET_CLOSE>");
			this.payloadInt = decodePayloadInt();
			break;
		case XBMSP_PACKET_SET_CONFIGURATION_OPTION:
			if (com.superhac.JXBStreamer.Core.Debug.debug)
				logger.info("<XBMSP_PACKET_SET_CONFIGURATION_OPTION>");
			this.payloadString = decodePayloadString();
			this.payloadString2 = decodePayloadString();
			break;
		case XBMSP_PACKET_AUTHENTICATION_INIT:
			if (com.superhac.JXBStreamer.Core.Debug.debug)
				logger.info("<XBMSP_PACKET_AUTHENTICATION_INIT>");
			this.payloadString = decodePayloadString();
			break;
		case XBMSP_PACKET_AUTHENTICATE:
			if (com.superhac.JXBStreamer.Core.Debug.debug)
				logger.info("<XBMSP_PACKET_AUTHENTICATE>");
			this.payloadInt = decodePayloadInt();
			this.payloadData = decodePayloadData();
			break;

		default:
			if (com.superhac.JXBStreamer.Core.Debug.debug)
				logger.info("Packet not identifed... Hacker? lmfao!");
		// typeNotIdentified = true;
		}

		// if typeNotIdentified is true throw exception.. TODO!

	}

	/**
	 * Decodes a String at the current payload buffer position.
	 * 
	 * @return
	 */
	private String decodePayloadString() {
		int stringSize = payload.getInt();

		// if its zero then the string is null
		if (stringSize == 0)
			return null;

		byte[] dir = new byte[stringSize];

		// get all the string bytes
		for (int i = 0; i < stringSize; i++)
			dir[i] = payload.get();

		// copy the bytes to a new string!
		try {
			return new String(dir, "US-ASCII");
		} catch (UnsupportedEncodingException e) {
			System.out.println("ERROR: " + e);
			System.exit(1);
		}

		return null;
	}

	/**
	 * Decodes a Int at the current payload buffer position.
	 * 
	 * @return
	 */
	private int decodePayloadInt() {
		return payload.getInt();

	}

	/**
	 * Decodes a Byte at the current payload buffer position.
	 * 
	 * @return
	 */
	private byte decodePayloadByte() {
		return payload.get();

	}

	/**
	 * Decodes a Long(64 bits) at the current payload buffer position.
	 * 
	 * @return
	 */
	private long decodePayloadInt64() {
		return payload.getLong();

	}

	/**
	 * Decodes a Byte Array at the current payload buffer position for whatever
	 * remains in the buffer.
	 * 
	 * @return
	 */
	private byte[] decodePayloadData() {
		byte[] payloadData = new byte[payload.remaining()];
		int size = payload.limit();

		for (int i = 0; i < size; i++)
			payloadData[i] = payload.get();

		return payloadData;

	}

	/**
	 * Encodes a packet OK response to this instance of XBMSPEncoderDecoder. An
	 * encoded message is always based on a message recieved from the client.
	 * Hence we use the orginal XBMSPEncoderDecoder instance of the recieved
	 * message to encode the response.
	 * 
	 * @return a new XBMSPEncoderDecoder with the response packet.
	 */
	protected XBMSPEncoderDecoder encodePacketOk() {

		if (com.superhac.JXBStreamer.Core.Debug.debug)
			logger.info("Encoded Packet Okay response.");
		// create a new packet
		XBMSPEncoderDecoder outgoingPacket = new XBMSPEncoderDecoder();

		// set the msg length
		outgoingPacket.setMsgLen(headerSize);

		// set the type
		outgoingPacket.setType(XBMSP_PACKET_OK);

		// set message ID to the received message ID
		outgoingPacket.setId(this.getId());

		return outgoingPacket;

	}

	/**
	 * Encodes a packet ERROR response to this instance of XBMSPEncoderDecoder.
	 * An encoded message is always based on a message recieved from the client.
	 * Hence we use the orginal XBMSPEncoderDecoder instance of the recieved
	 * message to encode the response.
	 * 
	 * @param errorCode
	 *            the error code as defined by the XBMSP protocol.
	 * @param errorMessage
	 *            an optional TEXT message
	 * @return a new XBMSPEncoderDecoder with the response packet.
	 */
	protected XBMSPEncoderDecoder encodePacketError(byte errorCode,
			String errorMessage) {
		if (com.superhac.JXBStreamer.Core.Debug.debug)
			logger.info("Encoded Packet Error response. Code:" + errorCode);

		XBMSPEncoderDecoder outgoingPacket = new XBMSPEncoderDecoder();

		ByteBuffer payload = ByteBuffer.allocate(2048);
		// create a new packet

		// set the type
		outgoingPacket.setType(XBMSP_PACKET_ERROR);

		// set message ID to the received message ID
		outgoingPacket.setId(this.getId());

		// set error code in payload section
		payload.put(errorCode);

		// optionally pack an error string. int32 length+string
		if (errorMessage != null) {
			// prepend the string length
			int stringLength = errorMessage.length();
			payload.putInt(stringLength);

			// add string to payload.. Convert string to ACSII bytes
			try {
				payload.put(errorMessage.getBytes("US-ASCII"));
			} catch (UnsupportedEncodingException e) {
				System.out.println("ERROR: " + e);
				System.exit(1);
			}
		}
		// set the payload on the outgoing packet
		outgoingPacket.setPayload(payload);

		return outgoingPacket;
	}

	/**
	 * Encodes a packet PacketHandle response to this instance of
	 * XBMSPEncoderDecoder. An encoded message is always based on a message
	 * recieved from the client. Hence we use the orginal XBMSPEncoderDecoder
	 * instance of the recieved message to encode the response.
	 * 
	 * @param handle
	 *            the handle number of the file or directory
	 * @return a new XBMSPEncoderDecoder with the response packet.
	 */
	protected XBMSPEncoderDecoder encodePacketHandle(int handle) {
		if (com.superhac.JXBStreamer.Core.Debug.debug)
			logger.info("Encoded Packet Handle response.");

		XBMSPEncoderDecoder outgoingPacket = new XBMSPEncoderDecoder();

		ByteBuffer payload = ByteBuffer.allocate(100); // only 1 int
		// create a new packet

		// set the type
		outgoingPacket.setType(XBMSP_PACKET_HANDLE);

		// set message ID to the received message ID
		outgoingPacket.setId(this.getId());

		// set the handle in the payload
		payload.putInt(handle);

		// set the payload on the outgoing packet
		outgoingPacket.setPayload(payload);

		return outgoingPacket;
	}

	/**
	 * Encodes a packet PacketFileData response to this instance of
	 * XBMSPEncoderDecoder. An encoded message is always based on a message
	 * recieved from the client. Hence we use the orginal XBMSPEncoderDecoder
	 * instance of the recieved message to encode the response.
	 * 
	 * @param name
	 *            The name of the file or handle(file.getName())
	 * @param data
	 *            The XML encoded String with the file data. (e.g. file size,
	 *            accessTime, etc..)
	 * @return a new XBMSPEncoderDecoder with the response packet.
	 */
	protected XBMSPEncoderDecoder encodePacketFileData(String name, String data) {
		if (com.superhac.JXBStreamer.Core.Debug.debug)
			logger.info("Encoded File data response. (XML Data).");
		XBMSPEncoderDecoder outgoingPacket = new XBMSPEncoderDecoder();
		byte bytes[];

		ByteBuffer payload = ByteBuffer.allocate(2000);
		// create a new packet

		// set the type
		outgoingPacket.setType(XBMSP_PACKET_FILE_DATA);

		// set message ID to the received message ID
		outgoingPacket.setId(this.getId());

		if (name != null && data != null) // a entry
		{
			// pack string 1
			payload.putInt(name.length());
			bytes = name.getBytes();
			payload.put(bytes);

			// pack string 2
			payload.putInt(data.length());
			bytes = data.getBytes();
			payload.put(bytes);
		} else // the end of the list was hit. set lengths to zero on both
		// strings
		{
			payload.putInt(0x00000000);
			payload.putInt(0x00000000);
		}

		// set the payload on the outgoing packet
		outgoingPacket.setPayload(payload);

		return outgoingPacket;

	}

	/**
	 * Encodes a packet encodePacketFileContents response to this instance of
	 * XBMSPEncoderDecoder. An encoded message is always based on a message
	 * recieved from the client. Hence we use the orginal XBMSPEncoderDecoder
	 * instance of the recieved message to encode the response.
	 * 
	 * @param data
	 *            the raw file data for the request
	 * @return a new XBMSPEncoderDecoder with the response packet.
	 */
	protected XBMSPEncoderDecoder encodePacketFileContents(ByteBuffer data) {
		if (com.superhac.JXBStreamer.Core.Debug.debug)
			logger.info("Encoded File Contents.");
		XBMSPEncoderDecoder outgoingPacket = new XBMSPEncoderDecoder();

		// set the type
		outgoingPacket.setType(XBMSP_PACKET_FILE_CONTENTS);

		// set message ID to the received message ID
		outgoingPacket.setId(this.getId());

		if (data != null) {

			outgoingPacket.setPayload(data);

		} else // no more data
		{
			ByteBuffer payload = ByteBuffer.allocate(4);
			payload.putInt(0x00000000);
			outgoingPacket.setPayload(payload);
		}

		return outgoingPacket;

	}

	/**
	 * Sets the payload of the packet
	 * 
	 * @param payload
	 */
	private void setPayload(ByteBuffer payload) {
		this.payload = payload;
	}

	/**
	 * Gets the first byte from the payload.. (message data section)
	 * 
	 * @return
	 */
	protected byte getPayloadByte() {
		return payloadByte;
	}

	/**
	 * Gets the first byte[] from the payload.. (message data section)
	 * 
	 * @return
	 */
	protected byte[] getPayloadData() {
		return payloadData;
	}

	/**
	 * Gets the first Int from the payload.. (message data section)
	 * 
	 * @return
	 */
	protected int getPayloadInt() {
		return payloadInt;
	}

	/**
	 * Gets the second Int from the payload.. (message data section)
	 * 
	 * @return
	 */
	protected int getPayloadInt2() {
		return payloadInt2;
	}

	/**
	 * Gets the first long(INT64) from the payload.. (message data section)
	 * 
	 * @return
	 */
	protected long getPayloadInt64() {
		return payloadInt64;
	}

	/**
	 * Gets the first String from the payload.. (message data section)
	 * 
	 * @return
	 */
	protected String getPayloadString() {
		return payloadString;
	}

	/**
	 * Gets the Second String from the payload.. (message data section)
	 * 
	 * @return
	 */
	protected String getPayloadString2() {
		return payloadString2;
	}
}
