package com.superhac.JXBStreamer.Core;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.util.logging.Logger;

/**
 * The FileHandlerObject maps Files to handles and provides services for
 * fulfilling client requests on Files. E.g. get file data.
 * <p>
 * <p>
 * More information can be found at <a
 * href="http://www.superhac.com">Superhac.com</a>
 * 
 * @author Steven Scott (superhac007@gmail.com)
 */

public class FileHandlerObject {

	/** The handle represting the file */
	private int handle;

	/** The file */
	private File file;

	/** The max position of the File */
	private long maxPosition; // the real file size in bytes

	/** The current position in the File */
	private long currentFilePosition; // starts at zero and always at the

	// position of the next read.

	/**
	 * The number of bytes transfered from file. Running Total. inc on file
	 * reads!
	 */
	private long fileTransferBytes = 0;

	/** The starting time of the handle being opened... */
	private long startTime;

	/** the File Stream representing the File */
	private RandomAccessFile fstream;

	/** Debug Logger */
	private static Logger logger;

	protected FileHandlerObject(int handle, File file) {

		this.handle = handle;
		this.file = file;

		// init start time... This is used in calculating the transfer time
		startTime = System.currentTimeMillis();

		// open the file
		try {
			fstream = new RandomAccessFile(file, "r");
			currentFilePosition = fstream.getFilePointer();
			this.maxPosition = fstream.length();

		} catch (FileNotFoundException e) {
			System.out.println("Should not have happend(FIleHandler): " + e);
			System.exit(1);
		} catch (IOException e) {
			System.out.println("Should not have happend(FIleHandler): " + e);
			System.exit(1);
		}

		logger = com.superhac.JXBStreamer.Core.Debug.getLogger();
	}

	/**
	 * Read a number of bytes from file
	 * 
	 * @param numberOfBytes
	 *            the number of bytes to read
	 * @return A bytebuffer containing the read bytes.
	 */
	protected ByteBuffer read(int numberOfBytes) {

		if (com.superhac.JXBStreamer.Core.Debug.debug)
			logger.info("Reading and packing file data. Read bytes req: "
					+ numberOfBytes + " Current POS: " + currentFilePosition
					+ " Max Pos: " + maxPosition);

		try {

			byte data[] = new byte[numberOfBytes];
			int bytesRead = fstream.read(data);
			ByteBuffer buf = ByteBuffer.allocate(20000); // at most 20k...
			// add check later

			if (bytesRead == -1) // eof
			{
				if (com.superhac.JXBStreamer.Core.Debug.debug)
					logger.info("End of File... Sending null. "
							+ "Current POS: " + currentFilePosition
							+ " Max Pos: " + maxPosition);
				return null;
			} else {
				currentFilePosition = fstream.getFilePointer(); // increment
				fileTransferBytes += bytesRead;
				buf.putInt(bytesRead);
				return buf.put(data, 0, bytesRead);

			}

		} catch (IOException e) {
			System.out.println("Should not have happend(FIleHandler read): "
					+ e);
			System.exit(1);
		}

		// should never get here
		return null;

	}

	/**
	 * Seek to a location withing the file...
	 * 
	 * @param offset
	 *            an amount to seek from the SeekType
	 * @param seekType
	 *            (0 = forward from the beginning of the file, 1 = backward from
	 *            the end of the file, 2 = forward from the current position, 3 =
	 *            backwards from the current position)
	 * @return True if its a success or false if not.
	 */
	protected boolean seekFile(long offset, byte seekType) {

		if (com.superhac.JXBStreamer.Core.Debug.debug)
			logger.info("Seeking file data: offset: " + offset + " SeekType:"
					+ seekType + " Current POS: " + currentFilePosition
					+ " Max Pos: " + maxPosition);

		boolean error = false;
		long newOffset;

		try {
			switch (seekType) {
			case XBMSPEncoderDecoder.XBMSP_FILE_SEEK_TYPE_FWDFROMBEG:
				if (offset <= maxPosition) {
					fstream.seek(offset);

				} else {
					error = true;
				}
				break;
			case XBMSPEncoderDecoder.XBMSP_FILE_SEEK_TYPE_BCKFROMEND:
				newOffset = maxPosition - offset;

				if (newOffset >= 0 && newOffset <= maxPosition) {
					fstream.seek(newOffset);
				} else {
					error = true;
				}
				break;
			case XBMSPEncoderDecoder.XBMSP_FILE_SEEK_TYPE_FWDFROMCUR:
				newOffset = currentFilePosition + offset;
				if (newOffset <= maxPosition) {
					fstream.seek(newOffset);
				} else {
					error = true;
				}
				break;
			case XBMSPEncoderDecoder.XBMSP_FILE_SEEK_TYPE_BCKFROMCUR:
				newOffset = currentFilePosition - offset;
				if (newOffset >= 0) {
					fstream.seek(newOffset);
				} else {
					error = true;
				}
				break;
			default:
				error = true;
			}
		} catch (IOException e) {
			System.out.println("Should not have happend(FIleHandler seek): "
					+ e);
			System.exit(1);
		}

		// System.out.println("Offset: "+currentFilePosition+" Error: "+error +
		// "Type: "+seekType);
		return error;
	}

	/**
	 * Get the file name represented by FileHandleObject
	 * 
	 * @return filename
	 */
	protected String getFileName() {
		return file.getName();
	}

	/**
	 * Gets the File object of the file represented by FileHandleObject
	 * 
	 * @return the File object
	 */
	protected File getFile() {
		return file;
	}

	/**
	 * Gets the handle number
	 * 
	 * @return handle number
	 */
	protected int getHandle() {
		return handle;
	}

	/**
	 * Gets the file transfer rate in seconds. Based on the creation time of the
	 * object , the current system time and the number of bytes transfered. E.g.
	 * reads.
	 * 
	 * @return The number of bytes per second transfered.
	 */
	protected long getFileTransferRate() {

		long duration = (System.currentTimeMillis() - startTime) / 1000; // in
		// secs
		long bytesPerSec = 0;

		if (fileTransferBytes > 0 && duration > 0)
			bytesPerSec = fileTransferBytes / duration;

		return bytesPerSec;
	}

}
