/*
	Copyright 2003 Chris Cavey
	
	This file is part of XID.
	
	XID is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.
	
	XID is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
	
	You should have received a copy of the GNU General Public License
	along with XID; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "xidioctl.h"

#pragma code_seg("PAGE") // start PAGE section
NTSTATUS XidIoctlGetDeviceDescriptor(PDEVICE_OBJECT DeviceObject, PIRP Irp)
{
	PHID_DESCRIPTOR pHidDescriptor; /* Hid descriptor for this device */
    NTSTATUS ntStatus = STATUS_SUCCESS;
    DEVICE_EXTENSION *DeviceExtension;
    PIO_STACK_LOCATION IrpStack;

    IrpStack = IoGetCurrentIrpStackLocation(Irp);
    DeviceExtension = GET_MINIDRIVER_DEVICE_EXTENSION (DeviceObject);

    // Get a pointer to the HID_DESCRIPTOR
    pHidDescriptor = (PHID_DESCRIPTOR)Irp->UserBuffer;

    if(IrpStack->Parameters.DeviceIoControl.OutputBufferLength < sizeof(*pHidDescriptor) )
    {
        ntStatus = STATUS_BUFFER_TOO_SMALL;
    } 
	else
    {
  		RtlZeroMemory( pHidDescriptor, sizeof(*pHidDescriptor) );
        
        // Copy device descriptor to HIDCLASS buffer
        pHidDescriptor->bLength                         = sizeof(*pHidDescriptor);
        pHidDescriptor->bDescriptorType                 = HID_HID_DESCRIPTOR_TYPE;
        pHidDescriptor->bcdHID                          = HID_REVISION;
        pHidDescriptor->bCountry                        = 0;
        pHidDescriptor->bNumDescriptors                 = 1;
        pHidDescriptor->DescriptorList[0].bReportType   = HID_REPORT_DESCRIPTOR_TYPE;
		pHidDescriptor->DescriptorList[0].wReportLength = (USHORT)DeviceExtension->reportGen.getReportDescSize();

		//Report how many bytes were copied
		Irp->IoStatus.Information = sizeof(*pHidDescriptor);
    }

	return ntStatus;
}

NTSTATUS XidIoctlGetReportDescriptor(PDEVICE_OBJECT DeviceObject, PIRP Irp)
{
	DEVICE_EXTENSION *DeviceExtension;
    PIO_STACK_LOCATION IrpStack;
    NTSTATUS ntStatus;
 	DeviceExtension = GET_MINIDRIVER_DEVICE_EXTENSION (DeviceObject);
    IrpStack = IoGetCurrentIrpStackLocation(Irp);

    if(DeviceExtension->reportGen.getReportDescSize() > (USHORT)IrpStack->Parameters.DeviceIoControl.OutputBufferLength )
    {
    	ntStatus = STATUS_BUFFER_TOO_SMALL;
    } 
	else
    {
		//RtlCopyMemory(Irp->UserBuffer, XidHidReportDesc, sizeof(XidHidReportDesc));
		DeviceExtension->reportGen.generateReportDescriptor((unsigned char *)Irp->UserBuffer);

		// Report how many bytes were copied
        Irp->IoStatus.Information = DeviceExtension->reportGen.getReportDescSize(); 
        ntStatus = STATUS_SUCCESS;
    }

	return ntStatus;
}

NTSTATUS XidIoctlGetAttributes(PDEVICE_OBJECT DeviceObject, PIRP Irp)
{
	NTSTATUS ntStatus = STATUS_SUCCESS;
    PIO_STACK_LOCATION IrpStack;

    IrpStack = IoGetCurrentIrpStackLocation(Irp);

    if( IrpStack->Parameters.DeviceIoControl.OutputBufferLength < sizeof (HID_DEVICE_ATTRIBUTES))
    {
        ntStatus = STATUS_BUFFER_TOO_SMALL;
    } 
	else
    {
        DEVICE_EXTENSION *DeviceExtension;
        PHID_DEVICE_ATTRIBUTES  DeviceAttributes;

        //Get a pointer to the device extension
        DeviceExtension = GET_MINIDRIVER_DEVICE_EXTENSION(DeviceObject);
        DeviceAttributes = (PHID_DEVICE_ATTRIBUTES)Irp->UserBuffer;

		//Zero memory so we can fill it with good values
        RtlZeroMemory(DeviceAttributes, sizeof(*DeviceAttributes));

        DeviceAttributes->Size          = sizeof(*DeviceAttributes);
        DeviceAttributes->VendorID      = DeviceExtension->VendorId;
        DeviceAttributes->ProductID     = DeviceExtension->ProductId;
        DeviceAttributes->VersionNumber = 1;
        
         //  Report how many bytes were copied
        Irp->IoStatus.Information = sizeof(*DeviceAttributes);
    }

	return ntStatus;
}

#pragma code_seg() //NOTE: The ReadReport code cannot be segmented, must not be swapped out, EVER.

NTSTATUS XidIoctlSetFeature(PDEVICE_OBJECT DeviceObject, PIRP Irp)
{
	NTSTATUS ntStatus = STATUS_SUCCESS;
	DEVICE_EXTENSION *DeviceExtension = GET_MINIDRIVER_DEVICE_EXTENSION(DeviceObject);
    PIO_STACK_LOCATION IrpStack = IoGetCurrentIrpStackLocation(Irp);
	HID_XFER_PACKET *xfer = (HID_XFER_PACKET *)Irp->UserBuffer;
	KIRQL irql;

	if( DeviceExtension->fStarted == FALSE )
    {
        ntStatus = STATUS_DEVICE_NOT_READY;
    }

    if(NT_SUCCESS(ntStatus))
    {
		KeAcquireSpinLock(&DeviceExtension->lockReport, &irql);
		DeviceExtension->reportGen.dynConfig.updateInternalConfigDesc(xfer->reportBuffer, xfer->reportBufferLen);
		KeReleaseSpinLock(&DeviceExtension->lockReport, irql);

		Irp->IoStatus.Information = xfer->reportBufferLen;
	}
	else
	{
		Irp->IoStatus.Information = 0;
	}

	Irp->IoStatus.Status = ntStatus;

	return ntStatus;
}

NTSTATUS XidIoctlGetFeature(PDEVICE_OBJECT DeviceObject, PIRP Irp)
{
	NTSTATUS ntStatus = STATUS_SUCCESS;
	DEVICE_EXTENSION *DeviceExtension = GET_MINIDRIVER_DEVICE_EXTENSION(DeviceObject);
    PIO_STACK_LOCATION IrpStack = IoGetCurrentIrpStackLocation(Irp);
	HID_XFER_PACKET *xfer = (HID_XFER_PACKET *)Irp->UserBuffer;
	KIRQL irql;
	
	if( DeviceExtension->fStarted == FALSE )
    {
        ntStatus = STATUS_DEVICE_NOT_READY;
    }

	DbgPrint("get feature: %d buffer size", xfer->reportBufferLen);

	/*
    if(NT_SUCCESS(ntStatus))
    {
		KeAcquireSpinLock(&DeviceExtension->lockReport, &irql);
		DeviceExtension->reportGen.dynConfig.UpdateInternalConfigDesc(xfer->reportBuffer, xfer->reportBufferLen);
		KeReleaseSpinLock(&DeviceExtension->lockReport, irql);

		Irp->IoStatus.Information = xfer->reportBufferLen;
	}
	else
	//{*/
		Irp->IoStatus.Information = 0;
	//}

	Irp->IoStatus.Status = ntStatus;

	return ntStatus;
}

NTSTATUS XidIoctlReadReport(PDEVICE_OBJECT DeviceObject, PIRP Irp)
{
	NTSTATUS ntStatus = STATUS_SUCCESS;
	KIRQL irql;
    DEVICE_EXTENSION *DeviceExtension;
    PIO_STACK_LOCATION  IrpStack;

    DeviceExtension = GET_MINIDRIVER_DEVICE_EXTENSION(DeviceObject);
    IrpStack = IoGetCurrentIrpStackLocation(Irp);
 
    //First check the size of the output buffer (there is no input buffer)
    if( IrpStack->Parameters.DeviceIoControl.OutputBufferLength < DeviceExtension->reportGen.getInputReportSize())
    {
        ntStatus = STATUS_BUFFER_TOO_SMALL;
    }

    if( DeviceExtension->fStarted == FALSE )
    {
        ntStatus = STATUS_DEVICE_NOT_READY ;
    }

    //If all's well, translate device specific data to HID report
    if(NT_SUCCESS(ntStatus))
    {
    	KeAcquireSpinLock(&DeviceExtension->lockReport, &irql);
		DeviceExtension->reportGen.generateInputReport((unsigned char *)Irp->UserBuffer);
		KeReleaseSpinLock(&DeviceExtension->lockReport, irql);
		
        Irp->IoStatus.Information = DeviceExtension->reportGen.getInputReportSize();
    } 
    else
    {
        Irp->IoStatus.Information = 0;
    }

    Irp->IoStatus.Status = ntStatus;

	return ntStatus;
}

NTSTATUS XidIoctlWriteReport(PDEVICE_OBJECT DeviceObject, PIRP Irp)
{
	NTSTATUS ntStatus = STATUS_SUCCESS;
	KIRQL irql;
    DEVICE_EXTENSION *DeviceExtension;
    PIO_STACK_LOCATION  IrpStack;
	HID_XFER_PACKET *inputReport;
	BYTE *temp = (BYTE *)(Irp->AssociatedIrp.SystemBuffer), 
		 out[6] = { 0 };
	ULONG outSize = 6;

    DeviceExtension = GET_MINIDRIVER_DEVICE_EXTENSION(DeviceObject);
    IrpStack = IoGetCurrentIrpStackLocation(Irp);
	Irp->IoStatus.Information = 0;

	// TODO: something with the report input buffer length
	//IrpStack->Parameters.DeviceIoControl.InputBufferLength);

    if( DeviceExtension->fStarted == FALSE )
    {
        ntStatus = STATUS_DEVICE_NOT_READY ;
    }

    //If all's well, translate device specific data to HID report
    if(NT_SUCCESS(ntStatus))
    {
		out[0] = 0x00;
		out[1] = 0x06; // 6 byte report size
		out[2] = 0x00;
		out[3] = temp[1]; // low frequency motor motor
		out[4] = 0x00;
		out[5] = temp[2]; // high frequency motor
		ntStatus = UsbDoInterruptTransfer(GET_HID_DEVICE_EXTENSION(DeviceObject), (void*)out, outSize, TRANSFER_OUT);
    } 

    Irp->IoStatus.Status = ntStatus;

	return ntStatus;
}