/*
	Copyright 2003 Chris Cavey
	
	This file is part of XID.
	
	XID is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.
	
	XID is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
	
	You should have received a copy of the GNU General Public License
	along with XID; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "xidcontroller.h"
/*
	XIDCONTROLLER is used to parse the raw data from the physical device
	The class provides functions to store report data, manipulate contols via configuration structures, and simply access individual controls

   TODO:
		must check values before use ( pointers ), but will this impact speed?
		use macros to parse button data, this may be faster ... is speed an issue?
*/

// copy the raw bytes read from the usb device ( controller ) into a local buffer for later manipulation
// "rawIn" is a pointer to this buffer, copy "size" bytes into local buffer "internalReport"
void XIDCONTROLLER::UpdateInternalReport(void *rawIn, unsigned char size)
{
	memcpy(&internalReport, rawIn, size);
}

#define short_to_word(x) ((signed int)x + (signed int)32768)
#define KeAbs(x) ((x >= 0 ) ? x : -x)

DWORD isqrt(DWORD input)
{
	DWORD retVal;
	__asm
	{
		mov eax, input
      xor ecx, ecx
      mov ebx, 1
lp:   cmp eax, ebx
      jb  ex
      sub eax, ebx
      add ebx, 2
      add ecx, 1
      jmp lp
ex:   mov retVal, ecx
	}
	return retVal;
}

// use the configuration info stored in buttonMapping combined with the type to accurately fill the report out
BYTE XIDCONTROLLER::Button(const XIDFEATURE_BUTTON *buttonMapping)
{
	BYTE retVal = 0;
	WORD threshold = buttonMapping->threshold;
	WORD deltaX = 0, deltaY = 0, dist;

	switch(buttonMapping->physicalControl)
	{
		case 1:
			retVal = A();
			if(buttonMapping->type == 1) // digital button?
					retVal = retVal >= threshold ? 255 : 0;
			break;
		case 2:
			retVal = B();
			if(buttonMapping->type == 1)
					retVal = retVal >= threshold ? 255 : 0;
			break;
		case 3:
			retVal = X();
			if(buttonMapping->type == 1)
					retVal = retVal >= threshold ? 255 : 0;
			break;
		case 4:
			retVal = Y();
			if(buttonMapping->type == 1)
					retVal = retVal >= threshold ? 255 : 0;
			break;
		case 5:
			retVal = White();
			if(buttonMapping->type == 1)
					retVal = retVal >= threshold ? 255 : 0;
			break;
		case 6:
			retVal = Black();
			if(buttonMapping->type == 1)
					retVal = retVal >= threshold ? 255 : 0;
			break;
		case 7:
			retVal = Back() ? 255 : 0;
			break;
		case 8:
			retVal = Start() ? 255 : 0;
			break;
		case 9:
			retVal = LStick() ? 255 : 0;
			break;
		case 10:
			retVal = RStick() ? 255 : 0;
			break;
		case 11:
			retVal = DPadU() ? 255 : 0;
			break;
		case 12:
			retVal = DPadUR() ? 255 : 0;
			break;
		case 13:
			retVal = DPadR() ? 255 : 0;
			break;
		case 14:
			retVal = DPadDR() ? 255 : 0;
			break;
		case 15:
			retVal = DPadD() ? 255 : 0;
			break;
		case 16:
			retVal = DPadDL() ? 255 : 0;
			break;
		case 17:
			retVal = DPadL() ? 255 : 0;
			break;
		case 18:
			retVal = DPadUL() ? 255 : 0;
			break;
		case 19:
			retVal = LTrig();
			if(buttonMapping->type == 1)
					retVal = retVal >= threshold ? 255 : 0;
			break;
		case 20:
			retVal = RTrig();
			if(buttonMapping->type == 1)
					retVal = retVal >= threshold ? 255 : 0;
			break;
		case 21: //left stick

			//DbgPrint("check is value %d is between %d and %d, check if %d is between %d and %d", LStickX(), buttonMapping->x1, buttonMapping->x2,  LStickY(), buttonMapping->y1, buttonMapping->y2);

			if(buttonMapping->type == 1)
			{
				if((buttonMapping->x1 <= LStickX()) && (buttonMapping->x2 >= LStickX()) && (buttonMapping->y2 <= LStickY()) && (buttonMapping->y1 >= LStickY()))
					retVal = 255;
				else
					retVal = 0;
			}
			else
			{
				if((buttonMapping->x1 <= LStickX()) && (buttonMapping->x2 >= LStickX()) && (buttonMapping->y2 <= LStickY()) && (buttonMapping->y1 >= LStickY()))
				{
					// TODO: need to implement the hot spot distance calculation
					retVal = 255;
				}
				else
					retVal = 0;
			}
			break;
		case 22: //right stick
			if(buttonMapping->type == 1)
			{
				if((buttonMapping->x1 <= RStickX()) && (buttonMapping->x2 >= RStickX()) && (buttonMapping->y2 <= RStickY()) && (buttonMapping->y1 >= RStickY()))
					retVal = 255;
				else
					retVal = 0;
			}
			else
			{
				if((buttonMapping->x1 <= RStickX()) && (buttonMapping->x2 >= RStickX()) && (buttonMapping->y2 <= RStickY()) && (buttonMapping->y1 >= RStickY()))
				{
					retVal = 255;
				}
				else
					retVal = 0;
			}
			break;
		default:
			break;
	}
	
	return retVal;
}

SHORT XIDCONTROLLER::Axis(const XIDFEATURE_AXIS *axisMapping)
{
	SHORT retVal = 0;
	BYTE flags = axisMapping->flags;

	switch(axisMapping->physicalControl)
	{
		case 23:
			retVal = LStickY();
			if(flags & XIDFEATUREFLAG_INVERT)
				retVal = -1 - retVal;
			break;
		case 24:
			retVal = LStickX();
			if(flags & XIDFEATUREFLAG_INVERT)
				retVal = -1 - retVal;
			break;
		case 25:
			retVal = RStickY();
			if(flags & XIDFEATUREFLAG_INVERT)
				retVal = -1 - retVal;
			break;
		case 26:
			retVal = RStickX();
			if(flags & XIDFEATUREFLAG_INVERT)
				retVal = -1 - retVal;
			break;
		case 27:
			if(DPadU())
				retVal = 32767;
			else if(DPadD())
				retVal = -32768;
			if(flags & XIDFEATUREFLAG_INVERT)
				retVal = -1 - retVal;
			break;
		case 28:
			if(DPadR())
				retVal = 32767;
			else if(DPadL())
				retVal = -32768;
			if(flags & XIDFEATUREFLAG_INVERT)
				retVal = -1 - retVal;
			break;
	}

	return retVal;
}

BYTE XIDCONTROLLER::Hat(const XIDFEATURE_HAT *hat)
{
	BYTE retVal = 0, x, tempVal;

	for(x = 1; x < 8; x+=2)
	{
		tempVal = Button((const XIDFEATURE_BUTTON *)&hat[x]);
		if(tempVal == 255)
		{
			retVal = x;
			break;
		}
	}

	if(retVal == 0)
	{
		retVal = -1;
		for(x = 0; x < 8; x+=2)
		{
			tempVal = Button((const XIDFEATURE_BUTTON *)&hat[x]);
			if(tempVal == 255)
			{
				retVal = x;
				break;
			}
		}
	}
	
	return retVal;
}


// simply accessor functions for various buttons of the controller
// perhaps macros would be faster

bool XIDCONTROLLER::Start()
{
	return internalReport.bmButtonsBits.S == 1;
}

bool XIDCONTROLLER::Back()
{
	return internalReport.bmButtonsBits.B == 1;
}

bool XIDCONTROLLER::LStick()
{
	return internalReport.bmButtonsBits.LT == 1;
}

bool XIDCONTROLLER::RStick()
{
	return internalReport.bmButtonsBits.RT == 1;
}

bool XIDCONTROLLER::DPadU()
{
	return internalReport.bmButtonsBits.U == 1;
}

bool XIDCONTROLLER::DPadUR()
{
	return (internalReport.bmButtonsBits.U == 1) && (internalReport.bmButtonsBits.R == 1);
}

bool XIDCONTROLLER::DPadR()
{
	return internalReport.bmButtonsBits.R == 1;
}

bool XIDCONTROLLER::DPadDR()
{
	return (internalReport.bmButtonsBits.D == 1) && (internalReport.bmButtonsBits.R == 1);
}

bool XIDCONTROLLER::DPadD()
{
	return internalReport.bmButtonsBits.D == 1;
}

bool XIDCONTROLLER::DPadDL()
{
	return (internalReport.bmButtonsBits.D == 1) && (internalReport.bmButtonsBits.L == 1);
}

bool XIDCONTROLLER::DPadL()
{
	return internalReport.bmButtonsBits.L == 1;
}

bool XIDCONTROLLER::DPadUL()
{
	return (internalReport.bmButtonsBits.U == 1) && (internalReport.bmButtonsBits.L == 1);
}


BYTE XIDCONTROLLER::A()
{
	return internalReport.analogButtons[XINPUT_GAMEPAD_A];
}

BYTE XIDCONTROLLER::B()
{
	return internalReport.analogButtons[XINPUT_GAMEPAD_B];
}

BYTE XIDCONTROLLER::X()
{
	return internalReport.analogButtons[XINPUT_GAMEPAD_X];
}

BYTE XIDCONTROLLER::Y()
{
	return internalReport.analogButtons[XINPUT_GAMEPAD_Y];
}

BYTE XIDCONTROLLER::Black()
{
	return internalReport.analogButtons[XINPUT_GAMEPAD_BLACK];
}

BYTE XIDCONTROLLER::White()
{
	return internalReport.analogButtons[XINPUT_GAMEPAD_WHITE];
}

BYTE XIDCONTROLLER::LTrig()
{
	return internalReport.analogButtons[XINPUT_GAMEPAD_LEFT_TRIGGER];
}

BYTE XIDCONTROLLER::RTrig()
{
	return internalReport.analogButtons[XINPUT_GAMEPAD_RIGHT_TRIGGER];
}

SHORT XIDCONTROLLER::LStickX()
{
	return internalReport.sThumbLX;
}

SHORT XIDCONTROLLER::LStickY()
{
	return internalReport.sThumbLY;
}


SHORT XIDCONTROLLER::RStickX()
{
	return internalReport.sThumbRX;
}


SHORT XIDCONTROLLER::RStickY()
{
	return internalReport.sThumbRY;
}
