#include <stdio.h>
#include <stdlib.h>
#include <sys/io.h>
#include <sys/types.h>
#include <unistd.h>
#include <getopt.h>
#include <string.h>
#include "port_io.h"

// special commands in programmer Mode
// write config
// eg SetNibble(LPT,8,11,0) = enable programmer mode
// eg SetNibble(LPT,8,not 11,0) = disable programmer mode
// eg SetNibble(LPT,9,DontCare,0) = set to adresse to adresse +1
// 0x378 'lpt1
// 0x278 'lpt2
// 0x3BC 'lpt3

#define ANZ_FS_ENTRIES 40
#define MAXDEVICES 4 			// Anzahl der bekannten Typen

DWORD dwFlashSectors[ANZ_FS_ENTRIES];

typedef struct Device {
	DWORD dwFlashDeviceDescriptor;		// Enthlt die Manufacturer und die Device ID
	DWORD dwFlashSectorsOffset;		// Offset in der Sectorstabel
	DWORD dwFlashNameOffset;		// Offset in der Namenstabelle
	char *szFlashName;
	DWORD dwBank;
	DWORD dwRangeStart;
	DWORD dwRangeEnde;
	DWORD dwRangeSize;
	DWORD dwBufferBank;
	char *szBuffer;
	char *szVBuffer;
} *PDevice;

struct Device dev[MAXDEVICES];

DWORD checkVersion(DWORD value);
DWORD checkSwitches(DWORD value);
DWORD checkDeviceX(WORD wPortAddr,BYTE conf,struct Device *device_info);
bool activate(WORD wPortAddr, BYTE conf, bool status);
bool readX(WORD wPortAddr,BYTE conf, struct Device *dev_info, int nBuffer);
bool eraseX(WORD wPortAddr,BYTE conf, struct Device *dev,bool sector,DWORD dwSector);
bool programX(WORD wPortAddr,BYTE conf, struct Device *dev);
void setBankRange(struct Device *dev, int nBankIndex);

#define DEBUG

void initDevStru() {

	dev[0].dwFlashDeviceDescriptor = 0x4DA;
	dev[0].dwFlashSectorsOffset = 0;
	dev[0].dwFlashNameOffset = 0;
	dev[0].szFlashName = "Fujitsu MBM29LV800TA\0";

	dev[1].dwFlashDeviceDescriptor = 0x45B;
	dev[1].dwFlashSectorsOffset = 20;
	dev[1].dwFlashNameOffset = 1;
	dev[1].szFlashName = "Fujitsu MBM29LV800BA\0";

	dev[2].dwFlashDeviceDescriptor = 0x1DA;
	dev[2].dwFlashSectorsOffset = 0;
	dev[2].dwFlashNameOffset = 2;
	dev[2].szFlashName = "AMD Am29LV800BT\0";

	dev[3].dwFlashDeviceDescriptor = 0x15B;
	dev[3].dwFlashSectorsOffset = 20;
	dev[3].dwFlashNameOffset = 3;
	dev[3].szFlashName = "AMD Am29LV800BB\0";

	dwFlashSectors[0] = 0x0;
	dwFlashSectors[1] = 0x10000;
	dwFlashSectors[2] = 0x20000;
	dwFlashSectors[3] = 0x30000;
	dwFlashSectors[4] = 0x40000;
	dwFlashSectors[5] = 0x50000;
	dwFlashSectors[6] = 0x60000;
	dwFlashSectors[7] = 0x70000;
	dwFlashSectors[8] = 0x80000;
	dwFlashSectors[9] = 0x90000;
	dwFlashSectors[10] = 0xA0000;
	dwFlashSectors[11] = 0xB0000;
	dwFlashSectors[12] = 0xC0000;
	dwFlashSectors[13] = 0xD0000;
	dwFlashSectors[14] = 0xE0000;
	dwFlashSectors[15] = 0xF0000;
	dwFlashSectors[16] = 0xF8000;
	dwFlashSectors[17] = 0xFA000;
	dwFlashSectors[18] = 0xFC000;
	dwFlashSectors[19] = -1;
	dwFlashSectors[20] = 0x0;
	dwFlashSectors[21] = 0x4000;

	// TEST TEST TEST Damit wird nur der erste Sektor gewhlt
	//dwFlashSectors[22] = -1;
	dwFlashSectors[22] = 0x6000;
	// Ein Trick gegen berlufe
	dwFlashSectors[23] = 0x8000;
	dwFlashSectors[24] = 0x10000;
	dwFlashSectors[25] = 0x20000;
	dwFlashSectors[26] = 0x30000;
	dwFlashSectors[27] = 0x40000;
	dwFlashSectors[28] = 0x50000;
	dwFlashSectors[29] = 0x60000;
	dwFlashSectors[30] = 0x70000;
	dwFlashSectors[31] = 0x80000;
	dwFlashSectors[32] = 0x90000;
	dwFlashSectors[33] = 0xA0000;
	dwFlashSectors[34] = 0xB0000;
	dwFlashSectors[35] = 0xC0000;
	dwFlashSectors[36] = 0xD0000;
	dwFlashSectors[37] = 0xE0000;
	dwFlashSectors[38] = 0xF0000;
	dwFlashSectors[39] = -1;
	

}
//' Vorbelegung der Combo Box
//ComboBank.AddItem ("1MB Bank (Block 1 - 4)")
//ComboBank.AddItem ("512kb Bank (Block 1 - 2)")
//ComboBank.AddItem ("512kb Bank (Block 3 - 4)")
//ComboBank.AddItem ("256kb Bank (Block 1)")
//ComboBank.AddItem ("256kb Bank (Block 2)")
//ComboBank.AddItem ("256kb Bank (Block 3)")
//ComboBank.AddItem ("256kb Bank (Block 4)")

//' Positionierung auf das erste Element
//ComboBank.ListIndex = 0
//ComboBank.Enabled = False

void print_help() {
	printf("bioxx usage : \n");
	printf("bioox [OPTION]\n");
	printf("      -h   help\n");
	printf("      -p   programmer mode\n");
	printf("      -r   read mode\n");
	printf("      -f   filename\n");
	printf("      -v   verify\n");
	printf("      -b   bank\n");
	printf("           0 -> 1MB Bank (Block 1 - 4)\n");
	printf("           1 -> 512kb Bank (Block 1 - 2)\n");
	printf("           2 -> 512kb Bank (Block 3 - 4)\n");
	printf("           3 -> 256kb Bank (Block 1)\n");
	printf("           4 -> 256kb Bank (Block 2)\n");
	printf("           5 -> 256kb Bank (Block 3)\n");
	printf("           6 -> 256kb Bank (Block 4)\n");
	printf("\n");
	printf("Examples :\n\n");
	printf("Programm : bioxx -p -b 0 -f test.bin\n");
	printf("Read     : bioxx -r -v -f test.bin\n");
}

#define M_NONE 0
#define M_PROG 1
#define M_READ 2

int main(int argc, char* argv[]) {

	WORD wPortAddr=0x378;
	DWORD dwVal;
	BYTE conf = 1;
	DWORD dwVersion;
	DWORD dwSwitches;
	DWORD dwFlashID;
	DWORD dwBank = 0;
	DWORD dwVerify = false;
	DWORD dwError = false;
	DWORD dwFileLength = 0;
	struct Device dev_info;
	int mode = M_NONE;
	char szFileName[2048];
	FILE *fp;
	int c;
	
	printf("\nBioXX Flasher 0.1\n\nLinux port by Edgar Hucek <hostmaster@ed-soft.at>\n\n");

	memset(szFileName,0,2048);
	initDevStru();

	while((c = getopt(argc, argv, "prvb:f:")) != -1) {
		switch(c) {
			case 'r':
				if(mode == M_NONE) {
					mode = M_READ;
				} else {
					printf("bioxx can be only in programmer or read mode\n");
					exit(1);
				}
				break;
			case 'p':
				if(mode == M_NONE) {
					mode = M_PROG;
				} else {
					printf("bioxx can be only in programmer or read mode\n");
					exit(1);
				}
				break;
			case 'b':
				if (optarg) {
					dwBank = atoi((const char *)optarg);
				} else {
					dwBank = 0;
				}
				break;
			case 'f':
				if (optarg) {
					strcpy(szFileName,optarg);
				} else {
					strcpy(szFileName,"image.bin");
				}
				break;
			case 'v':
				dwVerify = true;
				break;
			case 'h':
				print_help();
				exit(0);
				break;
		}
	}
	
	if(mode == M_NONE) {
		print_help();
		exit(0);
	}
	if((mode != M_NONE) && szFileName[0] == 0) {
		printf("No file to process\n");
	}

	if (iopl(3) == -1) {
		printf("Error init port 0x%03x\n",wPortAddr);
		printf("BioXX Flasher must be run with root rights\n");
		exit(1);
	}

	if(ioperm(wPortAddr,5,1) == -1) {
		printf("Error init port 0x%03x\n",wPortAddr);
		printf("BioXX Flasher must be run with root rights\n");
		exit(1);
	}

	if(!SetPortVal(wPortAddr, 255, 1)) { //init port , all pins high)
		printf("Error init port 0x%03x\n",wPortAddr);
		exit(1);
	}

	if(!GetHWVer(wPortAddr, &dwVal, conf)) {
		printf("Error geting HW Version\n");
		exit(1);
	}

	dwVersion = checkVersion(dwVal);
	dwSwitches = checkSwitches(dwVal);

	dwFlashID = checkDeviceX(wPortAddr,conf,&dev_info);
	if(dwFlashID == -1) {
		printf("Unknown Flash Type\n");
		exit(1);
	}

	printf("BioXX Board on LPT             : 0x%03x\n",wPortAddr);
	printf("Hardware Revision              : V%d\n",dwVersion);
	printf("Flash Device                   : 0x%04x - %s\n",dev_info.dwFlashDeviceDescriptor , dev_info.szFlashName);

	setBankRange(&dev_info,dwBank);
	printf("Bank                           : %d\n",dev_info.dwBank);
	printf("Bank Start                     : 0x%08x\n",dev_info.dwRangeStart);
	printf("Bank End                       : 0x%08x\n",dev_info.dwRangeEnde);
	printf("Bank Size                      : 0x%08x\n",dev_info.dwRangeSize);

	dev_info.szBuffer = malloc(dev_info.dwRangeSize);
	dev_info.szVBuffer = malloc(dev_info.dwRangeSize);

	memset(dev_info.szBuffer,0,dev_info.dwRangeSize);
	memset(dev_info.szVBuffer,0,dev_info.dwRangeSize);

	if(mode == M_READ) {
		readX(wPortAddr,conf,&dev_info,0);
		if(dwVerify) {
			readX(wPortAddr,conf,&dev_info,1);
			if(memcmp(dev_info.szBuffer,dev_info.szVBuffer,dev_info.dwRangeSize) != 0) {
				printf("Verify error\n");
				dwError = true;
			}
		}
		if(!dwError) {
			fp = fopen(szFileName,"w+");
			if(fp == NULL) {
				printf("Error open outputfile : %s\n",szFileName);
			} else {
				fwrite(dev_info.szBuffer,1,dev_info.dwRangeSize,fp);
				fclose(fp);
			}
		}
	}
	
	if(mode == M_PROG) {
		printf("Entering programmer mode\n");
		fp = fopen(szFileName,"r");
		if(fp == NULL) {
			printf("Error in opening file : %s\n",szFileName);
			exit(1);
		}
		fseek(fp,0L,SEEK_END);
		dwFileLength = ftell(fp);
		fseek(fp,0L,SEEK_SET);
		printf("Open file : %s\n",szFileName);
		printf("File size : %d\n",dwFileLength);
		if(dwFileLength != dev_info.dwRangeSize) {
			printf("Filesize not matching bank size\n");
		} else {
			fread(dev_info.szBuffer,dwFileLength,1,fp);
			if(dwBank == 0) {
				eraseX(wPortAddr,conf, &dev_info ,false,0);
			} else {
				{
					int i = dev_info.dwFlashSectorsOffset;
					DWORD dwStart;
					DWORD dwEnd;

					dwStart = dev_info.dwFlashSectorsOffset;
					dwEnd = 0;

					while(dwFlashSectors[i] != -1) {
						if(dev_info.dwRangeStart >= dwFlashSectors[i]) dwStart = i;
						if(abs(dev_info.dwRangeStart + dev_info.dwRangeSize) <= abs(dwFlashSectors[i])) break;
						i++;
					}

					dwEnd = i;

					for(i = dwStart; i < dwEnd;i++) {
						eraseX(wPortAddr,conf, &dev_info ,true,dwFlashSectors[i]);
					}
				}
			}
			programX(wPortAddr,conf, &dev_info);
			if(dwVerify) {
	                        readX(wPortAddr,conf,&dev_info,1);
	                        if(memcmp(dev_info.szBuffer,dev_info.szVBuffer,dev_info.dwRangeSize) != 0) {
	                                printf("Verify error\n");
	                        }
			}
		}
		fclose(fp);
	}
	
	free(dev_info.szBuffer);
	free(dev_info.szVBuffer);
	
	printf("\n");
	ioperm(wPortAddr,5,0);

	exit(0);
}

DWORD checkVersion(DWORD value)
{
	if(value & 0x8) {
		return 8;
	} else {
		return value & 0x7;
	}
}

DWORD checkSwitches(DWORD value)
{
	if(value == 8) {
		return value & 0x7;
	} else {
		return 2;
	}
}

// GOE START
// Bestimmt den Flashtype ber Manufactorer ID und Device ID
DWORD checkDeviceX(WORD wPortAddr,BYTE conf, struct Device *dev_info) {

	DWORD dwVal_Manu;
	DWORD dwVal_Dev;
	int i;
	DWORD dwFlashDeviceDescriptor;		// Enthlt die Manufacturer und die Device ID
	DWORD dwFlashSectorsOffset;		// Offset in der Sectorstabel

	activate(wPortAddr,conf,true);

	// Set to Autoselect Mode
	SetAdr(wPortAddr, 0xAAA, conf);
	SetDat(wPortAddr, 0xAA, conf);
	SetAdr(wPortAddr, 0x555, conf);
	SetDat(wPortAddr, 0x55, conf);
	SetAdr(wPortAddr, 0xAAA, conf);
	SetDat(wPortAddr, 0x90, conf);

	// Read ID
	SetAdr(wPortAddr, 0, conf);
	GetDat(wPortAddr, &dwVal_Manu, conf);

	SetAdr(wPortAddr, 2, conf);
	GetDat(wPortAddr, &dwVal_Dev, conf);

	dwFlashDeviceDescriptor = -1;
	dwFlashSectorsOffset = -1;

	dwFlashDeviceDescriptor = dwVal_Manu * 0x100;
	dwFlashDeviceDescriptor = dwFlashDeviceDescriptor + dwVal_Dev;

	// Offset bestimmen
	for(i=0;i < MAXDEVICES;i++) {
		if (dev[i].dwFlashDeviceDescriptor == dwFlashDeviceDescriptor) {
			memcpy(dev_info,&dev[i],sizeof(struct Device));
			SetAdr(wPortAddr, 0x0, conf);
    			SetDat(wPortAddr, 0xF0, conf);
			return  dwFlashDeviceDescriptor;
		}
	}

	SetAdr(wPortAddr, 0x0, conf);
	SetDat(wPortAddr, 0xF0, conf);

	activate(wPortAddr,conf,false);
	return -1;
}

bool activate(WORD wPortAddr, BYTE conf, bool status) {
    if(status == true) {
        SetNibble(wPortAddr, 8, 11, conf);		// enable programmer mode, disable LPC bus on flash
        return true;
    } else {
        SetNibble(wPortAddr, 8, 0, conf); 		// enable LPC bus on flash, disable programer mode
	return true;
    }
}

bool readX(WORD wPortAddr,BYTE conf, struct Device *dev,int nBuffer) {
	int i;
	int nProz;
	DWORD dwVal;
	
	activate(wPortAddr,conf,true);
	
	SetAdr(wPortAddr, dev->dwRangeStart, conf);
	
	nProz = 0;

	for(i = 0;i < dev->dwRangeSize;i++) {
		if((i%(dev->dwRangeSize/100)) == 0) {
			printf("Read %03d%%\n",nProz);
			nProz ++;
		}
		GetDat(wPortAddr, &dwVal, conf);
		if(nBuffer == 0) {
			dev->szBuffer[i] = (BYTE)dwVal;
		}
		if(nBuffer == 1) {
			dev->szVBuffer[i] = (BYTE)dwVal;
		}
		SetNibble(wPortAddr, 9, 9, conf);
	}
	
	activate(wPortAddr,conf,false);
	return true;
}

bool eraseX(WORD wPortAddr,BYTE conf, struct Device *dev,bool sector,DWORD dwSector) {
	
	DWORD dwvaltmp = 1;
	DWORD dwVal = 0;
	
	activate(wPortAddr,conf,true);
	
	SetAdr(wPortAddr, 0xAAA, conf);
	SetDat(wPortAddr, 0xAA, conf);
	SetAdr(wPortAddr, 0x555, conf);
	SetDat(wPortAddr, 0x55, conf);
	SetAdr(wPortAddr, 0xAAA, conf);
	SetDat(wPortAddr, 0x80, conf);
	SetAdr(wPortAddr, 0xAAA, conf);
	SetDat(wPortAddr, 0xAA, conf);
	SetAdr(wPortAddr, 0x555, conf);
	SetDat(wPortAddr, 0x55, conf);
	if(sector) {
		// GOE START
		SetAdr(wPortAddr, dwSector, conf);
		// GOE ENDE
		SetDat(wPortAddr, 0x30, conf);
						
	} else {
		SetAdr(wPortAddr, 0xAAA, conf);
		SetDat(wPortAddr, 0x10, conf);
	}

	printf("Erasing \n");
	while ((dwvaltmp != dwVal) && (dwVal != 255)) {
		dwvaltmp = dwVal;
		GetDat(wPortAddr, &dwVal, conf);
	}
	printf(" done\n");

	activate(wPortAddr,conf,false);
	return true;

}

bool programX(WORD wPortAddr,BYTE conf, struct Device *dev) {

	int i;
	int nProz = 0;
	DWORD dwVal;
	
	activate(wPortAddr,conf,true);	

	SetAdr(wPortAddr, 0xAAA, conf);
	SetDat(wPortAddr, 0xAA, conf);
	SetAdr(wPortAddr, 0x555, conf);
	SetDat(wPortAddr, 0x55, conf);
	SetAdr(wPortAddr, 0xAAA, conf);
	SetDat(wPortAddr, 0x20, conf);

	SetAdr(wPortAddr, dev->dwRangeStart, conf);
	
	for(i = 0; i < dev->dwRangeSize;i++) {
		if((i%(dev->dwRangeSize/100)) == 0) {
			printf("Write %03d%%\n",nProz);
                        nProz ++;
                }
		SetDat(wPortAddr, 0xA0, conf);             //short write command (see datasheet of FlashChip)
		SetDat(wPortAddr, dev->szBuffer[i], conf);    //write data

read_again_prg:
		GetDat(wPortAddr, &dwVal, conf);
		if(((dwVal & 128) ^ (dev->szBuffer[i] & 128)) == 0) {
			goto do_next_prg; // 'next byte to program
		}
		if((dwVal & 32) != 32) {
			goto read_again_prg;
		}
		GetDat(wPortAddr, &dwVal, conf);
		if(((dwVal & 128) ^ (dev->szBuffer[i] & 128)) == 0) {
			goto do_next_prg; // 'next byte to program
		}
		if((dwVal & 32) != 32) {
			goto read_again_prg;
		}

		printf("Programm error in Address 0x%08x\n",dev->dwRangeStart + i);
		goto programm_end;

do_next_prg:
		SetNibble(wPortAddr, 9, 9, conf);
	}
	
programm_end:
	SetDat(wPortAddr, 0x90, conf);
	SetDat(wPortAddr, 0xF0, conf);

	activate(wPortAddr,conf,true);	
	return true;

}

void setBankRange(struct Device *dev, int nBankIndex) {
// Bestimmt die untere und obere Grenze der selektierten Bank
// WICHTIG: ohne die Abs Funktionen kommen negative Werte vor
//          und zwar immer dann, wenn das MSB gesetzt ist (0x8xxxxx)

	DWORD dwRangeStart = 0;
	DWORD dwRangeEnde = 0;

	switch(nBankIndex) {
		case 0:
			dwRangeStart = 0x0;
			dwRangeEnde = 0x100000;
			break;
		case 1:
			dwRangeStart = 0x0;
			dwRangeEnde = 0x80000;
			break;
		case 2:
			dwRangeStart = 0x80000;
			dwRangeEnde = 0x80000;
			break;
		case 3:
			dwRangeStart = 0x0;
			dwRangeEnde = 0x40000;
			break;
		case 4:
			dwRangeStart = 0x40000;
			dwRangeEnde = 0x40000;
			break;
		case 5:
			dwRangeStart = 0x80000;
			dwRangeEnde = 0x40000;
			break;
		case 6:
			dwRangeStart = 0xC0000;
			dwRangeEnde = 0x40000;
			break;
	}
	
	dev->dwBank = nBankIndex;
	dev->dwRangeStart = dwRangeStart;
	dev->dwRangeSize = dwRangeEnde;
	dev->dwRangeEnde = abs(dwRangeStart + dwRangeEnde);
}

