unit XDFSMakerUnit;

{
  XBOX DISC FILE SYSTEM MAKER (aka xISO) Unit v1.0
  Copyright (C) 2003 Yursoft

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

  ------------------------------------------------------------------------------

  Historial

  19 de Marzo de 2003
  Escrito y publicado.    
}

interface

uses Windows, SysUtils, Classes;

const
  XBOX_MEDIA_ID = 'MICROSOFT*XBOX*MEDIA';
  
  XBOX_FILE_ATTRIBUTE_READONLY = $01;
  XBOX_FILE_ATTRIBUTE_HIDDEN   = $02;
  XBOX_FILE_ATTRIBUTE_SYSTEM   = $04;
  XBOX_FILE_ATTRIBUTE_DIRECTORY= $10;
  XBOX_FILE_ATTRIBUTE_FILE     = $20;
  XBOX_FILE_ATTRIBUTE_NORMAL   = $80;

type
{$A-}
  TXBOX_FS_VOLUME_DESCRIPTOR = record
    IDIn: array[0..19] of Char;
    RootSector: Integer;
    RootSize: Integer;
    FileTime: FILETIME;
    Blank: array[0..1991] of Byte;
    IDOut: array[0..19] of Char;
  end;

  PXBOX_FS_ENTRY = ^TXBOX_FS_ENTRY;
  TXBOX_FS_ENTRY = record
    LNode: Word;
    RNode: Word;
    EntrySector: Integer;
    EntrySize: Cardinal;
    Attributes: Byte;
    LongFileName: Byte;
    FileName: array[0..255] of Char;
  end;
{$A+}

  // Type of the event procedure
  TMessage = procedure(MessageText: string);

  TDirectoryList = class;

  // Entry record of the TDirectoryList class. 
  PFile = ^TFile;
  TFile = record
    FileName: string;
    Size: Integer;
    Attributes: Integer;
    DirectoryPointer: TDirectoryList;

    RNode: Word;
    RecordSize: Word;
  end;

  TDirectoryList = class(TList)
  public
    // Points to the parent folder.
    Parent: TDirectoryList;
    // Constructor of the object. 
    constructor Create(ParentDirectory: TDirectoryList);
    destructor Destroy(); override;
    // Return if the directory entry is a directory or not.
    function IsDirectory(FileName: PFile): Boolean;
    // Add a new entry in this directory.
    function AddEntry(FileName: PFile): integer;
    // Return the directory entry at i position.
    function Entry(i: integer): PFile;
    // Return if this directory is empty or not.
    function Empty(): Boolean;
  end;

  TXBOX_FILESYSTEM = class
  private
    // Buffer interno de objeto.
    // Internal buffer of the object.
    Buffer: array[0..65535] of Byte;
    // Padre del arbol de ficheros.
    // Root of the treefiles.
    Root: TDirectoryList;
    // Devuelve el tamao de una entrada de directorio.
    // Return the size of the directory entry.
    function SizeDirEntry(DirectoryList: TDirectoryList): integer;
    // Crea apartir del parametro Directory la tabla de directorios y ficheros en DirectoryList como padre.
    // Make a tree directory of the Directory passed.
    procedure MakeFileList(Directory: string; DirectoryList: TDirectoryList);
    // Genera la tabla de ficheros de la imagen.
    // Makes the files table of the image.
    function NMakeISO(ISOStream: TFilestream; DirectoryList: TDirectoryList; var NextSectorAvailable: integer): integer;
    // Genera el VD y llama a NMakeISO para generar el resto de la imagen.
    // Make the Volume Descriptor and call to NMakeISO.
    procedure MakeISO(ISOName: string);
  public
    RootSector: Integer;
    OnProcess: TMessage;
    constructor Create;
    destructor Destroy; override;
    // Genera el fichero ISO apartir del directorio y el nombre de la ISO especificado.
    // This public function makes the ISO of the directory specified.
    function Make(Directory: string; ISOName: string): Boolean;
  end;

implementation

///////////// Common Functions //////////////

// Devuelve el offset del sector siguiente al pasado.
// Return the offset of the next sector. 
function NextSector(Offset: int64): int64;
begin
  Result := ((Offset div 2048) * 2048) + 2048;
end;

// Redondea el valor pasado a multiplo de un sector 2048 bytes.
// Round the offset to be a multiple of 2048.
function OffsetToSector(Offset: int64): int64;
begin
  if (Offset mod 2048) <> 0 then
    Result := (Offset div 2048) + 1
  else
    Result := Offset div 2048;
end;

///////////// Class TDirectoryList //////////////
function Compare(Item1, Item2: Pointer): Integer;
begin
  Result := CompareText(PFile(Item1)^.FileName, PFile(Item2)^.FileName);
end;

constructor TDirectoryList.Create(ParentDirectory: TDirectoryList);
begin
  Parent := ParentDirectory;
  inherited Create;
end;

destructor TDirectoryList.Destroy;
var
   i: integer;
begin
  for i := 0 to Count-1 do
  begin
    if IsDirectory(PFile(Items[i])) then
      if PFile(Items[i]).DirectoryPointer <> nil then
      begin
        PFile(Items[i]).DirectoryPointer.Free;
        Dispose(PFile(Items[i]));
      end;
  end;   
  inherited;
end;

function TDirectoryList.IsDirectory(FileName: PFile): Boolean;
begin
  Result := (FileName.Attributes and XBOX_FILE_ATTRIBUTE_DIRECTORY) = XBOX_FILE_ATTRIBUTE_DIRECTORY;
end;

function TDirectoryList.AddEntry(FileName: PFile): integer;
begin
  Result := Add(FileName);
end;

function TDirectoryList.Entry(i: integer): PFile;
begin
  Result := PFile(Items[i]);
end;

function TDirectoryList.Empty: Boolean;
begin
  Result := Count = 0;
end;


///////////// Class TXBOX_FILESYSTEM //////////////
constructor TXBOX_FILESYSTEM.Create;
begin
  Root := TDirectoryList.Create(nil);
  RootSector := 264;
  inherited;
end;

destructor TXBOX_FILESYSTEM.Destroy;
begin
  Root.Free;
  inherited;
end;


function TXBOX_FILESYSTEM.SizeDirEntry(DirectoryList: TDirectoryList): integer;
var
   i,PositionDirEntry,SizeEntry: integer;
   ListEntry: PFile;
begin
  Result := 0;
  if DirectoryList = nil then Exit;  

  PositionDirEntry := 0;
  for i := 0 to DirectoryList.Count-1 do
  begin
    ListEntry := DirectoryList.Entry(i);

    SizeEntry := 14 + Length(ListEntry.FileName);

    if (2048 - (PositionDirEntry mod 2048)) < SizeEntry then
      PositionDirEntry := NextSector(PositionDirEntry);

    PositionDirEntry := PositionDirEntry + SizeEntry;

    if (PositionDirEntry mod 4) <> 0 then
      PositionDirEntry := PositionDirEntry + 4 - (PositionDirEntry mod 4);

    if i <> DirectoryList.Count-1 then
    begin
      SizeEntry := 14 + Length(ExtractFileName(DirectoryList.Entry(i+1).FileName));
      if (2048 - (PositionDirEntry mod 2048)) < SizeEntry then
        PositionDirEntry := NextSector(PositionDirEntry);
    end;      
  end;
  Result := PositionDirEntry;
end;


procedure TXBOX_FILESYSTEM.MakeFileList(Directory: string; DirectoryList: TDirectoryList);
var
   SR: TSearchRec;
   Entry: PFile;
begin
  if Directory[Length(Directory)] <> '\' then
    Directory := Directory + '\';

  if (FindFirst(Directory+'*.*',faArchive or faDirectory or faHidden or faSysFile or faReadOnly,SR) = 0) then
  begin
    repeat
      if (SR.Name[1] = '.') then Continue;

      New(Entry);
      Entry.FileName := Directory+SR.Name;
      Entry.Size := SR.Size;
      Entry.Attributes := SR.Attr;
      DirectoryList.AddEntry(Entry);
      
      if (Entry.Attributes and faDirectory) = faDirectory then
      begin
        Entry.DirectoryPointer := TDirectoryList.Create(DirectoryList);
        MakeFileList(Entry.FileName,Entry.DirectoryPointer);
      end;
    until (FindNext(SR) <> 0);
    FindClose(SR);
    DirectoryList.Sort(Compare);      
  end;
end;


function TXBOX_FILESYSTEM.NMakeISO(ISOStream: TFilestream; DirectoryList: TDirectoryList; var NextSectorAvailable: integer): integer;
var
   i,j: integer;
   PositionDirEntry,SizeEntry,ReadIt: int64;
   OffsetDirEntry: Int64;
   F: TFilestream;
   ListEntry: PFile;
   Entry: TXBOX_FS_ENTRY;
   s: string;
begin
  if DirectoryList = nil then Exit;

  // Realizamos la asignacin del Nodo Derecho de la entrada.
  // We do the assign of the right node to the entry.
  PositionDirEntry := 0;  
  for i := 0 to DirectoryList.Count-1 do
  begin
    ListEntry := DirectoryList.Entry(i);

    SizeEntry := 14 + Length(ExtractFileName(ListEntry.FileName));

    if (2048 - (PositionDirEntry mod 2048)) < SizeEntry then
      PositionDirEntry := NextSector(PositionDirEntry);

    PositionDirEntry := PositionDirEntry + SizeEntry;

    if (PositionDirEntry mod 4) <> 0 then
    begin
      ListEntry.RecordSize := SizeEntry + 4 - (PositionDirEntry mod 4);
      PositionDirEntry := PositionDirEntry + 4 - (PositionDirEntry mod 4);
    end
    else
      ListEntry.RecordSize := SizeEntry;

    if i <> DirectoryList.Count-1 then
    begin
      SizeEntry := 14 + Length(ExtractFileName(DirectoryList.Entry(i+1).FileName));
      if (2048 - (PositionDirEntry mod 2048)) < SizeEntry then
        PositionDirEntry := NextSector(PositionDirEntry);
    end;

    // Si es la ultima entrada del directorio debemos asignarle 0 al nodo derecho.
    // If this entry is the last, we must assign the entry 0 to the Right Node.
    if i = DirectoryList.Count-1 then
      ListEntry.RNode := 0
    else
      ListEntry.RNode := PositionDirEntry div 4;      
  end;

  // Desplazamiento del directorio en la ISO.
  // Offset of the directory in the ISO:
  OffsetDirEntry := ISOStream.Position;
  NextSectorAvailable := NextSectorAvailable + OffsetToSector(SizeDirEntry(DirectoryList));
  FillChar(Buffer,SizeOf(Buffer),$FF);
  ISOStream.Write(Buffer,OffsetToSector(SizeDirEntry(DirectoryList))*2048);
  ISOStream.Seek(NextSectorAvailable*2048,soBeginning);
  SetEndOfFile(ISOStream.Handle);

  for i := 0 to DirectoryList.Count-1 do
  begin
    ListEntry := DirectoryList.Entry(i);
   
    Entry.LNode := 0;
    Entry.RNode := ListEntry.RNode;

    // Si es un directorio le asignamos los valores segun deba ser.
    // If the entry is a directory, we are going to write the correct values in
    // the record.
    if DirectoryList.IsDirectory(ListEntry) then
    begin
      // Si no est vacio escrimos los valores, si no es asi por estandar debemos
      // ponerlo a el tamao y el sector a que apunta.
      // If its a empty directory we must zeroed the size and the sector pointer
      // of the entry.
      if not ListEntry.DirectoryPointer.Empty then
      begin
        Entry.EntrySize := SizeDirEntry(ListEntry.DirectoryPointer);
        Entry.EntrySector := NextSectorAvailable;
      end
      else
      begin
        Entry.EntrySize := 0;
        Entry.EntrySector := 0;
      end;
    end
    else
    begin
      Entry.EntrySize := ListEntry.Size;
      Entry.EntrySector := NextSectorAvailable;
    end;
    NextSectorAvailable := NextSectorAvailable + OffsetToSector(Entry.EntrySize);

    Entry.LongFileName := Length(ExtractFileName(ListEntry.FileName));
    Entry.Attributes := ListEntry.Attributes;

    // Copiamos el nombre del fichero en la estructura de entrada de la xISO.
    // Copy the name into the xISO Entry Filename.
    FillChar(Entry.FileName,SizeOf(Entry.FileName),$FF);
    s := ExtractFileName(ListEntry.FileName);
    for j := 1 to Length(s) do
       Entry.FileName[j-1] := s[j];            

    // Si es un directorio lo procesamos como tal, sino lo escribimos en la ISO.   
    // If the Entry is a Directory, we are going to process it.
    // else copy the file into the ISO.
    if DirectoryList.IsDirectory(ListEntry) then
    begin
      // Mensaje de progreso.
      // Process messages.
      if Assigned(OnProcess) then
        OnProcess('Processing directory ' + ListEntry.FileName + '.');
      NMakeISO(ISOStream,ListEntry.DirectoryPointer,NextSectorAvailable);
    end
    else
    begin
      try
        F := TFilestream.Create(ListEntry.FileName,fmOpenRead or fmShareDenyNone);
      except
        F := nil;
      end;
      // Si se ha conseguido abrir el fichero lo archivamos en la ISO.
      // sino metemos un dummy.
      // If we have the file opened, we are going to write to the ISO,
      // if not, we must do a dummy file.
      if F <> nil then
      begin
        while (F.Position < F.Size) do
        begin
          ReadIt := F.Read(Buffer,SizeOf(Buffer));
          ISOStream.Write(Buffer,ReadIt);
        end;
        F.Free;
      end
      else
      begin
        // Mostramos el mensaje de error si no podemos abrir el fichero.
        // We show an error text if we cannot open the file.
        if Assigned(OnProcess) then
          OnProcess('A file cannot be opened. Doing dummy.');

        // Escribimos el fichero falso.
        // Write the dummy file.  
        j := 0;
        FillChar(Buffer,SizeOf(Buffer),0);
        while (j < (ListEntry.Size div SizeOf(Buffer))) do
        begin
          ISOStream.Write(Buffer,SizeOf(Buffer));
          j := j+1;
        end;
        
        if j <> ListEntry.Size then
          ISOStream.Write(Buffer,ListEntry.Size-j);
      end;
      
      // Comprobamos que el Entry entra en el sector actual, sino saltamos al siguiente.
      // We check if this entry can be wrote in this sector, if not, we skip to the next sector.
      if (ISOStream.Position mod 2048) <> 0 then
      begin
          ISOStream.Seek(NextSector(ISOStream.Position),soBeginning);
          SetEndOfFile(ISOStream.Handle);
      end;
    end;

    // Comprobamos que el Entry entra en el sector actual, sino saltamos al siguiente.
    // We check if this entry can be wrote in this sector, if not, we skip to the next sector.
    if (2048 - (ISOStream.Position mod 2048)) < ListEntry.RecordSize then
    begin
      ISOStream.Seek(NextSector(ISOStream.Position),soBeginning);
      SetEndOfFile(ISOStream.Handle);
    end;

    // Volvemos al punto donde dejamos la tabla de ficheros y copiamos la entrada
    // actual.
    // We return to the file table and write the current entry.
    ISOStream.Seek(OffsetDirEntry,soBeginning);
    if (2048 - (ISOStream.Position mod 2048)) < ListEntry.RecordSize then
      ISOStream.Seek(NextSector(ISOStream.Position),soBeginning);
    ISOStream.Write(Entry,ListEntry.RecordSize);
    OffsetDirEntry := ISOStream.Position;
    ISOStream.Seek(0,soEnd);
    // Despues de esto hemos vuelto al final de la ISO.
    // In this point we are at the end offset of the ISO.
  end;
  // Comprobamos que la entrada entra en el sector actual, sino saltamos al siguiente.
  // We check if this entry can be wrote in this sector, if not, we skip to the next sector.
  if (ISOStream.Position mod 2048) <> 0 then
  begin
    ISOStream.Seek(NextSector(ISOStream.Position),soBeginning);
    SetEndOfFile(ISOStream.Handle);
  end;
end;

procedure TXBOX_FILESYSTEM.MakeISO(ISOName: String);
var
   F: TFilestream;
   XBOX_VD: TXBOX_FS_VOLUME_DESCRIPTOR;
   Sector: integer;
   ActualDate: FILETIME;
begin
  if not Root.Empty then
  begin
    // Comprobamos que el fichero de destino (ISO) no est en uso.
    // We check if the destination filename is not in using by other process.
    try
      F := TFilestream.Create(ISOName,fmCreate);
    except
      if Assigned(OnProcess) then
        OnProcess('Cannot be created the ISO file. The file is in using by other process.');
      Exit;
    end;
    F.Seek(65536,soBeginning);
    SetEndOfFile(F.Handle);

    if RootSector < 33 then
      RootSector := 33;
      
    GetSystemTimeAsFileTime(ActualDate);

    // Escribimos el Descriptor de Volumen (Identificador) del estandar XDFS.
    // We write the Volume Descriptor (ID) of the XDFS estandar.
    FillChar(XBOX_VD,SizeOf(XBOX_VD),0);
    XBOX_VD.IDIn := XBOX_MEDIA_ID;
    XBOX_VD.RootSector := RootSector;
    XBOX_VD.RootSize := SizeDirEntry(Root);
    XBOX_VD.FileTime := ActualDate;
    XBOX_VD.IDOut := XBOX_MEDIA_ID;
    F.Write(XBOX_VD,SizeOf(XBOX_VD));

    // Establecemos el sector del raiz segun la variable RootSector.
    // Root sector.
    F.Seek(RootSector*2048,soBeginning);
    SetEndOfFile(F.Handle);

    Sector := RootSector;
    if Assigned(OnProcess) then
      OnProcess('Processing directory root directory.');
    NMakeISO(F,Root,Sector);
    F.Free;
  end;
end;

function TXBOX_FILESYSTEM.Make(Directory: string; ISOName: string): Boolean;
begin
  Result := False;
  MakeFileList(Directory,Root);
  MakeISO(ISOName);
  Result := True;
end;


end.
