#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <malloc.h>
#include <gmp.h>


/* Adjust these for your CPU and compiler */
typedef unsigned int BITS32;
typedef unsigned char BITS8;


#define SHF_DIGESTSIZE 20
/* Put out a digest of this size in bytes */
#define SHF_DIGESTWORDSIZE 5
/* and words */

#define SHF_BLOCKSIZE  64
#define SHF_BLOCKWORDSIZE 16
/* Process messages in this block size */

/* Structure for storing SHF info */

typedef BITS32 shadigest[SHF_DIGESTWORDSIZE];
typedef BITS8  sha_digest[SHF_DIGESTSIZE];

typedef struct {
  BITS32    data [SHF_BLOCKWORDSIZE];     /* SHS data buffer */
  BITS32    countLo;       /* Count (64 bits in              */
  BITS32    countHi;       /* 2 32 bit words)                */
  shadigest digest;        /* Message digest                 */
  int       kind;          /* Which type of SHA?             */
  int       thisword;      /* Which word are we processing?  */
  int       thisbyte;      /* Which byte in that word?       */
} sha_context, *sha_context_ptr;

/* Private part of this unit */
static void shapad        (sha_context_ptr context);
static void shaTransform  (sha_context_ptr context);
static void shaTransformCanned                     (void);


#define SHF_LENGTH_PAD  8
#define SHF_PAD_WORDS 2
/* Part of the padding needs to be message length (2  4 byte words) */
#define SHF_PADDING (SHF_BLOCKSIZE - SHF_LENGTH_PAD)
/* Remainder of block to be padded */

/* We have to exercise special care on machines with a natural
 * wordlength >32 characters that we do masking properly
 */
#define MASK32 & 0xFFFFFFFF
#define MASK8 & 0xFF


// Public functions
void shaInit (sha_context_ptr context, int version);
void shaUpdate (sha_context_ptr context, BITS8 *buffer, int count);
void shaFinal (sha_context_ptr context, sha_digest adigest);





/* Implementation */

void shaInit (sha_context_ptr context, int version) {
 int loopindex;

 /* Set initial values */
 context->digest[0] = 0x67452301L;
 context->digest[1] = 0xEFCDAB89L;
 context->digest[2] = 0x98BADCFEL;
 context->digest[3] = 0x10325476L;
 context->digest[4] = 0xC3D2E1F0L;

 /* Initialize bit count */
 context->countLo = 0L;
 context->countHi = 0L;
 
 /* Initialize buffer to empty. */
 context->thisword = 0;
 context->thisbyte = 0;
 
 /* Zero out data */
 for (loopindex=0;loopindex<SHF_BLOCKWORDSIZE;loopindex++)
     context->data[loopindex] = 0;

 /* What sort of SHA are we doing? */
 context->kind = version;
 
}

/* 
**************************************************************** 
*/

/* Rotate Left n bits (32 bit) */
#define S(n,X) (((X)<<(n)) | ((X)>>(32-(n))))

static void shaTransform (sha_context_ptr context) {

 BITS32 W[80], temp;  int i;

/* A buffer of 5 32-bit words */
BITS32 A, B, C, D, E;

#define DG   context->digest

/* Get digest */
 A = DG[0]; B = DG[1]; C = DG[2]; D = DG[3]; E = DG[4];

 /* Copy the data buffer into the work buffer */
 for (i=0; i<SHF_BLOCKWORDSIZE; i++)
     {W[i] = context->data[i];context->data[i] = 0;}
 
/* Expand the 16 words into the SHF_BLOCKSIZE temporary data words */
 for (i=16; i<80; i++) {
     W[i] = W[i-3]^W[i-8]^W[i-14]^W[i-16];
     if (context->kind) W[i] = S(1,W[i]);
 }
 
/* Guts (four sub-rounds) */
#define PRE  temp = W[i] + S(5,A) + E + 
#define POST ; E = D; D = C; C = S(30,B); B = A; A = temp;

#define CONST1 0x5A827999L
#define CONST2 0x6ED9EBA1L
#define CONST3 0x8F1BBCDCL
#define CONST4 0xCA62C1D6L

 for (i= 0; i<20; i++) {PRE CONST1 + (( B & C ) | (~B & D))  POST}
 for (i=20; i<40; i++) {PRE CONST2 + (B^C^D)                 POST}
 for (i=40; i<60; i++) {PRE CONST3 + ((B&C) | (B&D) | (C&D)) POST}
 for (i=60; i<80; i++) {PRE CONST4 + (B^C^D)                 POST}

/* Update digest */
 DG[0] += A; DG[1] += B; DG[2] += C; DG[3] += D; DG[4] += E;

 DG[0] &= 0xFFFFFFFF;
 DG[1] &= 0xFFFFFFFF;
 DG[2] &= 0xFFFFFFFF;
 DG[3] &= 0xFFFFFFFF;
 DG[4] &= 0xFFFFFFFF;

/* Block is now empty */
 context->thisword = 0;
 context->thisbyte = 0;
}


/* ******************************* */

void shaUpdate (sha_context_ptr context, BITS8 *buffer, int count) {

 BITS32 thebits; /* current bit pattern being processed */
 int theword; /* Which word in the buffer we are dealing with. */
 
 /* Add a potentially 32 bit count to the two word count */
 context->countHi += count >> 29;           /* handle count > 2**29 */
 context->countLo += count & 0x1FFFFFFF;    /* Handle count <=2**29 */
 context->countHi += context->countLo >> 29;/* Handle carry */
 context->countLo &= 0x1FFFFFFF;            /* Clear carry  */

 theword = context->thisword;
 thebits = context->data[theword];

 while (count--) {
   thebits = (thebits << 8) | *buffer++;
   if (++context->thisbyte >= 4) {
      context->data[theword++] = thebits; thebits = 0;
      if (theword >= SHF_BLOCKWORDSIZE) { 
         shaTransform (context);
         theword = 0;
      }
      context->thisbyte = 0;
   }
 }
 context->data[theword] = thebits;
 context->thisword = theword;

 }

/* ************************************ */
/* Pad out a block. */

static void shapad(sha_context_ptr context) {
  int loopindex;

  context->data[context->thisword]<<=8;
  context->data[context->thisword] |= 0x80;
  /* pad out the rest of this word */
  switch (context->thisbyte) {
    case 3:                                      ;break;
    case 2: context->data[context->thisword]<<= 8;break;
    case 1: context->data[context->thisword]<<=16;break;
    case 0: context->data[context->thisword]<<=24;break;
  }

  /* and then the rest of the words in the block */
  for (loopindex=context->thisword + 1; loopindex < SHF_BLOCKWORDSIZE;loopindex++)
     context->data[loopindex] = 0;

  /* And note it is now empty */
  context->thisword = 0;  context->thisbyte = 0;
}
/* ************************************ */
/* Convert a word digest to bytes, in a byte order independent manner */

void shaBytes (sha_context_ptr context, sha_digest adigest) {
int loopindex;

 for (loopindex=0;loopindex<SHF_DIGESTWORDSIZE;loopindex++) {
     *adigest++ = (BITS8) (context->digest[loopindex] >> 24   MASK8);
     *adigest++ = (BITS8) (context->digest[loopindex] >> 16  & (BITS32) 0xFF);
     *adigest++ = (BITS8) (context->digest[loopindex] >>  8  & (BITS32) 0xFF);
     *adigest++ = (BITS8) (context->digest[loopindex]        & (BITS32) 0xFF);
 }

}

void shaFinal (sha_context_ptr context, sha_digest adigest) {
 int loopindex; int bytesused;

 /* bytes used in the buffer */
 bytesused = context->thisword * 4 + context->thisbyte + 2;
   /* +1 for un zerobasing, +1 for the pad character we're about to do */
 
 /* Pad, but with the convention that the 0 pad starts with a single */
 /* one bit. */
 shapad(context);

 /* if we don't have room for the message size, then start a new block */
 if (bytesused > (SHF_BLOCKSIZE - SHF_LENGTH_PAD)) {
    shaTransform(context);
    for (loopindex=0; loopindex<(SHF_BLOCKWORDSIZE-SHF_PAD_WORDS); loopindex++) 
        context->data[loopindex] = 0;
    context->thisword = SHF_BLOCKWORDSIZE;
    context->thisbyte = 0;
 }

 /* Append length in bits, and transform */
 context->data[14] = context->countHi;
 context->data[15] = context->countLo<<3; /* We keep it as a byte count */
 shaTransform (context);

 /* get the digest out to the user. */
 shaBytes (context, adigest);

} /* end routine shaFinal */


/* ************************************************************ */

void shaHash (int version, BITS8 *buffer, int count, sha_digest adigest) {
 sha_context example;

 shaInit   (&example, version);
 shaUpdate (&example, buffer, count);
 shaFinal  (&example, adigest);

}

/* end sha.c **************************************************** */


/* Reads a 32 bit value from the specified file address.  The value read is
   in Xbox (Intel) byte order and is converted to host byte order. */
BITS32 read_file_dword(FILE *input, long address)
{
	BITS8 data[4];

	fseek(input, address, SEEK_SET);
	fread(data, sizeof(data), 1, input);

	return ((BITS32) data[3] << 24) | ((BITS32) data[2] << 16) | ((BITS32)
		data[1] << 8) | data[0];
}


/* Table of PKCS paddings that the Xbox considers legal (from xbedump) */
static const BITS8 RSApkcs1paddingtable[][16] = {
	{0x0F, 0x14,0x04,0x00,0x05,0x1A,0x02,0x03,0x0E,0x2B,0x05,0x06,0x09,0x30,0x21,0x30},
	{0x0D, 0x14,0x04,0x1A,0x02,0x03,0x0E,0x2B,0x05,0x06,0x07,0x30,0x1F,0x30,0x00,0x00},
	{0x00, 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00},
	{0xFF}
};



int main(int argc, char **argv)
{
	FILE *handle;
	long address;
	BITS32 public_exponent;
	BITS8 modulus[2048 / 8];
	BITS8 temp[4];
	BITS8 *xlive = NULL;
	long length;
	mpz_t value, result, gmp_modulus;
	size_t count;
	sha_context context;
	sha_digest digest;

	if (!argv[1] || !argv[2] || argv[3])
	{
		printf("Syntax: sigcheck xboxkrnl.exe xboxlive.bin\n"
			"xboxkrnl.exe is a retail unhacked kernel image\n"
			"xboxlive.bin is a dump of the Xbox Live security code\n");
		return 1;
	}

	/* Open the kernel image.  We need the kernel image to get its public
	   key. */
	if (!(handle = fopen(argv[1], "rb")))
	{
		fprintf(stderr, "Error opening %s\n", argv[1]);
		return 1;
	}

	/* Get file length */
	fseek(handle, 0, SEEK_END);
	length = ftell(handle);

	/* Check for a sane length.  Failing for > 2 gig is okay. */
	if (length < 0x40)
		goto kernel_corrupt;

	/* Check for MZ */
	if ((read_file_dword(handle, 0) & 0xFFFF) != 0x5A4D)
		goto kernel_corrupt;

	/* Get PE header address */
	/* IMAGE_DOS_HEADER::e_lfanew */
	if ((address = read_file_dword(handle, 0x3C)) + 0x7C > length)
		goto kernel_corrupt;

	/* Get address of export directory */
	/* IMAGE_NT_HEADERS::OptionalHeader.DataDirectory[IMAGE_DIRECTORY_ENTRY_EXPORT] */
	if ((address = read_file_dword(handle, address + 0x78)) + 0x20 > length)
		goto kernel_corrupt;

	/* Get address of actual export table */
	/* IMAGE_EXPORT_DIRECTORY::AddressOfFunctions */
	if ((address = read_file_dword(handle, address + 0x1C)) + (355 * 4) > length)
		goto kernel_corrupt;

	/* Get address of public key (export #355, XePublicKeyData) */
	if ((address = read_file_dword(handle, address + ((355 - 1) * 4))) + 0x11C > length)
		goto kernel_corrupt;

	/* Check first 4 bytes for being RSA1 */
	if (read_file_dword(handle, address) != 0x31415352)
		goto kernel_corrupt;

	/* Read the public key's exponent - this should always be 65537 */
	public_exponent = read_file_dword(handle, address + 0x10);

	/* Read the modulus from the public key */
	fseek(handle, address + 0x14, SEEK_SET);
	fread(modulus, sizeof(modulus), 1, handle);

	/* Done with kernel image */
	fclose(handle);

	/* Try to open the Xbox Live security code file */
	if (!(handle = fopen(argv[2], "rb")))
	{
		fprintf(stderr, "Error opening %s\n", argv[2]);
		return 1;
	}

	/* Get the length */
	fseek(handle, 0, SEEK_END);
	length = ftell(handle);

	/* Try to allocate memory for the whole thing */
	if (!(xlive = malloc((size_t) length)))
	{
		fprintf(stderr, "Not enough memory to load %s\n", argv[2]);
		goto error;
	}

	/* Read file to memory */
	fseek(handle, 0, SEEK_SET);
	fread(xlive, length, 1, handle);

	/* Check whether second word contains the length of the signed portion */
	if ((read_file_dword(handle, 0) >> 16) != (BITS32) length)
	{
		fprintf(stderr, "Xbox Live image is corrupt\n");
		goto error;
	}

	/* Allocate memory for 2048 bits */
	mpz_init2(value, 2048);
	mpz_init2(result, 2048);
	mpz_init2(gmp_modulus, 2048);

	/* Import the modulus from the public key data */
	mpz_import(gmp_modulus, sizeof(modulus), -1, 1, -1, 0, modulus);

	/* Import the security file's signature */
	mpz_import(value, sizeof(modulus), -1, 1, -1, 0, &xlive[0x1C]);

	/* Decrypt the signature */
	mpz_powm_ui(result, value, public_exponent, gmp_modulus);

	/* Export the decrypted signature as bytes.  This will be the 20 byte
	   SHA in reverse byte order, followed by an optional PKCS-1 header,
	   then a bunch of FF's until there are 254/256 bytes filled, then 01
	   then 00. */
	memset(modulus, 0, sizeof(modulus));
	mpz_export(modulus, &count, -1, 1, -1, 0, result);

	/* Hash the file with SHA-1.  Microsoft's algorithm is to hash the size
	   (as an Intel-order DWORD) then the data. */
	length -= 0x11C;
	temp[0] = (BITS8) length;
	temp[1] = (BITS8) (length >> 8);
	temp[2] = (BITS8) (length >> 16);
	temp[3] = (BITS8) (length >> 24);
	shaInit(&context, 1);
	shaUpdate(&context, temp, sizeof(temp));
	shaUpdate(&context, &xlive[0x11C], length);
	shaFinal(&context, digest);

	/* Compare the SHA hash with the decrypted signature.  Microsoft's
	   signatures have the SHA in backwards order from normal. */
	for (length = 0; length < 20; length++)
		if (digest[length] != modulus[20 - length - 1])
			goto bad_signature;

	/* Check whether the part after the hash is a valid PKCS1 signature */
	for (length = 0; RSApkcs1paddingtable[length][0] != 0xFF; length++)
	{
		if (!RSApkcs1paddingtable[length][0] ||
			!memcmp(&digest[20], &RSApkcs1paddingtable[length][1],
			(size_t) RSApkcs1paddingtable[length][0]))
		{
			length = (long) (unsigned long) RSApkcs1paddingtable[length][0];
			break;
		}
	}
	if (RSApkcs1paddingtable[length][0] == 0xFF)
		goto bad_signature;

	/* The byte after the signature should be 00 */
	length += 20;
	if (modulus[length++] != 0x00)
		goto bad_signature;

	/* Check for a sequence of FF's up until the last 2 bytes */
	for (; length < sizeof(modulus) - 2; length++)
		if (modulus[length] != 0xFF)
			goto bad_signature;

	/* Check for a 01 then a 00 */
	if ((modulus[length] != 0x01) || (modulus[length + 1] != 0x00))
		goto bad_signature;

	/* Signature is valid!! */
	printf("Signature check PASSED.\n");
	goto error;

bad_signature:
	printf("Signature check FAILED.\n");
	goto error;

kernel_corrupt:
	fprintf(stderr, "Kernel image is corrupt or not a kernel\n");
	goto error;

error:
	if (handle)
		fclose(handle);
	if (xlive)
		free(xlive);
	return 0;
}
