/*
FUNCTION
<<tmpfile64>>---create a large temporary file

INDEX
	tmpfile64
INDEX
	_tmpfile64_r

ANSI_SYNOPSIS
	#include <stdio.h>
	FILE *tmpfile64(void);

	FILE *_tmpfile64_r(void *<[reent]>);

TRAD_SYNOPSIS
	#include <stdio.h>
	FILE *tmpfile64();

	FILE *_tmpfile64_r(<[reent]>)
	char *<[reent]>;

DESCRIPTION
Create a large temporary file (a file which will be deleted automatically),
using a name generated by <<tmpnam>>.  The temporary file is opened with
the mode <<"wb+">>, permitting you to read and write anywhere in it
as a binary file (without any data transformations the host system may
perform for text files).  The file may be larger than 2GB.

The alternate function <<_tmpfile64_r>> is a reentrant version.  The
argument <[reent]> is a pointer to a reentrancy structure.

Both <<tmpfile64>> and <<_tmpfile64_r>> are only defined if __LARGE64_FILES
is defined.

RETURNS
<<tmpfile64>> normally returns a pointer to the temporary file.  If no
temporary file could be created, the result is NULL, and <<errno>>
records the reason for failure.

PORTABILITY
<<tmpfile64>> is a glibc extension.

Supporting OS subroutines required: <<close>>, <<fstat>>, <<getpid>>,
<<isatty>>, <<lseek64>>, <<open64>>, <<read>>, <<sbrk>>, <<write>>.

<<tmpfile64>> also requires the global pointer <<environ>>.
*/

#include <stdio.h>
#include <errno.h>

#ifdef __LARGE64_FILES

FILE *
_DEFUN (_tmpfile64_r, (ptr),
	struct _reent *ptr)
{
  FILE *fp;
  int e;
  char *f;
  char buf[L_tmpnam];

  if ((f = _tmpnam_r (ptr, buf)) == NULL)
    return NULL;
  fp = _fopen64_r (ptr, (const char *)f, "wb+");
  e = ptr->_errno;
  _CAST_VOID _remove_r (ptr, f);
  ptr->_errno = e;
  return fp;
}

#ifndef _REENT_ONLY

FILE *
_DEFUN_VOID (tmpfile64)
{
  return _tmpfile64_r (_REENT);
}

#endif

#endif /* __LARGE64_FILES */

