/*
 * MP3 Tag library. It includes an implementation of the ID3 tags and Lyrics3
 * tags as they are defined at www.id3.org
 *
 * Copyright (C) Eric Farng 2003
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.farng.mp3.lyrics3;

import java.io.IOException;
import java.io.RandomAccessFile;

import org.farng.mp3.AbstractMP3Fragment;
import org.farng.mp3.InvalidTagException;
import org.farng.mp3.TagOptionSingleton;
import org.farng.mp3.TagUtilities;


/**
 * Title:       Lyrics3v2Field Description: This class is an individual field
 * within a Lyrics3 tag Copyright:   Copyright (c) 2002 Company:
 *
 * @author Eric Farng
 * @version $Revision: 1.1 $
 */
public class Lyrics3v2Field
    extends AbstractMP3Fragment {
    /**
     * Creates a new Lyrics3v2Field object.
     */
    public Lyrics3v2Field() {}

    /**
     * Creates a new Lyrics3v2Field object.
     *
     * @param body DOCUMENT ME!
     */
    public Lyrics3v2Field(AbstractLyrics3v2FieldBody body) {
        this.body = body;
    }

    /**
     * Creates a new Lyrics3v2Field object.
     *
     * @param file DOCUMENT ME!
     *
     * @throws InvalidTagException DOCUMENT ME!
     * @throws IOException DOCUMENT ME!
     */
    public Lyrics3v2Field(RandomAccessFile file)
                   throws InvalidTagException, IOException {
        this.read(file);
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public String getIdentifier() {
        if (body == null) {
            return "";
        }

        return body.getIdentifier();
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public int getSize() {
        return body.getSize() + 5 + getIdentifier()
                                    .length();
    }

    /**
     * DOCUMENT ME!
     *
     * @param file DOCUMENT ME!
     *
     * @throws InvalidTagException DOCUMENT ME!
     * @throws IOException DOCUMENT ME!
     */
    public void read(RandomAccessFile file)
              throws InvalidTagException, IOException {
        byte[] buffer = new byte[6];

        // lets scan for a non-zero byte;
        long filePointer;
        byte b;

        do {
            filePointer = file.getFilePointer();
            b           = file.readByte();
        } while (b == 0);

        file.seek(filePointer);

        // read the 3 character ID
        file.read(buffer, 0, 3);

        String identifier = new String(buffer, 0, 3);

        // is this a valid identifier?
        if (TagUtilities.isLyrics3v2FieldIdentifier(identifier) == false) {
            throw new InvalidTagException(identifier +
                                          " is not a valid ID3v2.4 frame");
        }

        body = readBody(identifier, file);
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public String toString() {
        if (body == null) {
            return "";
        }

        return body.toString();
    }

    /**
     * DOCUMENT ME!
     *
     * @param file DOCUMENT ME!
     *
     * @throws IOException DOCUMENT ME!
     */
    public void write(RandomAccessFile file)
               throws IOException {
        if ((((AbstractLyrics3v2FieldBody) body).getSize() > 0) ||
                TagOptionSingleton.getInstance()
                    .isLyrics3SaveEmptyField()) {
            byte[] buffer = new byte[3];

            String str = getIdentifier();

            for (int i = 0; i < str.length(); i++) {
                buffer[i] = (byte) str.charAt(i);
            }

            file.write(buffer,
                       0,
                       str.length());

            body.write(file);
        }
    }

    /**
     * DOCUMENT ME!
     *
     * @param identifier DOCUMENT ME!
     * @param file DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     *
     * @throws InvalidTagException DOCUMENT ME!
     * @throws IOException DOCUMENT ME!
     */
    private AbstractLyrics3v2FieldBody readBody(String identifier,
                                                RandomAccessFile file)
                                         throws InvalidTagException, 
                                                IOException {
        AbstractLyrics3v2FieldBody body = null;

        if (identifier.equals("AUT")) {
            body = new FieldBodyAUT(file);
        } else if (identifier.equals("EAL")) {
            body = new FieldBodyEAL(file);
        } else if (identifier.equals("EAR")) {
            body = new FieldBodyEAR(file);
        } else if (identifier.equals("ETT")) {
            body = new FieldBodyETT(file);
        } else if (identifier.equals("IMG")) {
            body = new FieldBodyIMG(file);
        } else if (identifier.equals("IND")) {
            body = new FieldBodyIND(file);
        } else if (identifier.equals("INF")) {
            body = new FieldBodyINF(file);
        } else if (identifier.equals("LYR")) {
            body = new FieldBodyLYR(file);
        } else {
            body = new FieldBodyUnsupported(file);
        }

        return body;
    }
}