/*
 * MP3 Tag library. It includes an implementation of the ID3 tags and Lyrics3
 * tags as they are defined at www.id3.org
 *
 * Copyright (C) Eric Farng 2003
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.farng.mp3.filename;

import java.util.Iterator;

import org.farng.mp3.MP3File;
import org.farng.mp3.TagException;
import org.farng.mp3.TagOptionSingleton;
import org.farng.mp3.TagUtilities;
import org.farng.mp3.id3.AbstractFrameBodyTextInformation;
import org.farng.mp3.id3.AbstractFrameBodyUrlLink;
import org.farng.mp3.id3.AbstractID3v2FrameBody;
import org.farng.mp3.id3.ID3v2_4;
import org.farng.mp3.id3.ID3v2_4Frame;


/**
 * <p>
 * Title:
 * </p>
 * 
 * <p>
 * Description:
 * </p>
 * 
 * <p>
 * Copyright: Copyright (c) 2003
 * </p>
 * 
 * <p>
 * Company:
 * </p>
 *
 * @author $author$
 * @version $Revision: 1.1 $
 */
public class FilenameTagBuilder {
    /**
     * DOCUMENT ME!
     *
     * @param token DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     *
     * @throws TagException DOCUMENT ME!
     */
    static public FilenameComposite createCompositeFromToken(String token)
                                                      throws TagException {
        String[]          splitToken      = null;
        FilenameComposite composite       = null;
        FilenameComposite beforeComposite;
        FilenameComposite middleComposite;
        FilenameComposite afterComposite;

        splitToken = parseParenthesis(token);

        if (splitToken != null) {
            composite = new FilenameParenthesis();
            ((FilenameParenthesis) composite).setOpenDelimiter(splitToken[0]);

            //((FilenameParenthesis) composite).setCloseDelimiter(splitToken[1]);
            beforeComposite = createCompositeFromToken(splitToken[2]);
            ((FilenameParenthesis) composite).setBeforeComposite(beforeComposite);

            middleComposite = createCompositeFromToken(splitToken[3]);
            ((FilenameParenthesis) composite).setMiddleComposite(middleComposite);

            afterComposite = createCompositeFromToken(splitToken[4]);
            ((FilenameParenthesis) composite).setAfterComposite(afterComposite);

            composite.setOriginalToken(token);

            return composite;
        }

        splitToken = parseDelimiter(token);

        if (splitToken != null) {
            composite = new FilenameDelimiter();
            ((FilenameDelimiter) composite).setDelimiter(splitToken[0]);
            beforeComposite = createCompositeFromToken(splitToken[1]);
            ((FilenameDelimiter) composite).setBeforeComposite(beforeComposite);
            afterComposite = createCompositeFromToken(splitToken[2]);
            ((FilenameDelimiter) composite).setAfterComposite(afterComposite);
            composite.setOriginalToken(token);

            return composite;
        }

        splitToken = parseStartWordDelimiter(token);

        if (splitToken != null) {
            composite = new FilenameStartWordDelimiter();
            ((FilenameDelimiter) composite).setDelimiter(splitToken[0]);

            beforeComposite = createCompositeFromToken(splitToken[1]);
            ((FilenameStartWordDelimiter) composite).setBeforeComposite(beforeComposite);

            afterComposite = createCompositeFromToken(splitToken[2]);
            ((FilenameStartWordDelimiter) composite).setAfterComposite(afterComposite);
            composite.setOriginalToken(token);

            return composite;
        }

        splitToken = parseEndWordDelimiter(token);

        if (splitToken != null) {
            composite = new FilenameEndWordDelimiter();

            ((FilenameDelimiter) composite).setDelimiter(splitToken[0]);

            beforeComposite = createCompositeFromToken(splitToken[1]);
            ((FilenameEndWordDelimiter) composite).setBeforeComposite(beforeComposite);
            afterComposite = createCompositeFromToken(splitToken[2]);
            ((FilenameEndWordDelimiter) composite).setAfterComposite(afterComposite);
            composite.setOriginalToken(token);

            return composite;
        }

        if ((token != null) && (token.trim()
                                    .length() > 0)) {
            composite = new FilenameToken();
            ((FilenameToken) composite).setToken(token.trim());
            composite.setOriginalToken(token);

            return composite;
        }

        return null;
    }

    /**
     * DOCUMENT ME!
     *
     * @param mp3File DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     *
     * @throws Exception DOCUMENT ME!
     */
    static public FilenameTag createFilenameTagFromMP3File(MP3File mp3File)
                                                    throws Exception {
        FilenameTag       filenameTag = new FilenameTag();
        FilenameComposite composite;
        ID3v2_4           id3tag;

        // create composite
        composite = createCompositeFromToken(mp3File.getMp3file().getName());
        updateCompositeFromAllTag(composite, mp3File);
        updateCompositeFromAllOption(composite);

        // create tag
        id3tag = createId3TagFromComposite(composite);

        // assign values;
        filenameTag.setMp3file(mp3File);
        filenameTag.setComposite(composite);
        filenameTag.setId3tag(id3tag);

        return filenameTag;
    }

    /**
     * DOCUMENT ME!
     *
     * @param composite DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     *
     * @throws Exception DOCUMENT ME!
     */
    static public ID3v2_4 createId3TagFromComposite(FilenameComposite composite)
                                             throws Exception {
        Class         classname;
        FilenameToken token;
        ID3v2_4       tag      = new ID3v2_4();
        Iterator      iterator = composite.iterator();

        while (iterator.hasNext()) {
            token     = (FilenameToken) iterator.next();
            classname = token.getId3v2FrameBodyClass();

            if (classname != null) {
                AbstractID3v2FrameBody body = (AbstractID3v2FrameBody) classname.newInstance();

                /**
                 * @todo-javadoc need to add support for more frame bodies
                 *       here
                 */
                if (body instanceof AbstractFrameBodyTextInformation) {
                    ((AbstractFrameBodyTextInformation) body).setText(token.getToken());
                    ((AbstractFrameBodyTextInformation) body).setTextEncoding((byte) 0);
                } else if (body instanceof AbstractFrameBodyUrlLink) {
                    ((AbstractFrameBodyUrlLink) body).setUrlLink(token.getToken());
                }

                ID3v2_4Frame frame = new ID3v2_4Frame();
                frame.setBody(body);
                tag.setFrame(frame);
            }
        }

        return tag;
    }

    /**
     * DOCUMENT ME!
     *
     * @param composite DOCUMENT ME!
     */
    static public void updateCompositeFromAllOption(FilenameComposite composite) {
        Iterator iterator = TagOptionSingleton.getInstance()
                            .getKeywordIterator();

        while (iterator.hasNext()) {
            composite.matchAgainstKeyword((Class) iterator.next());
        }
    }

    /**
     * DOCUMENT ME!
     *
     * @param composite DOCUMENT ME!
     * @param mp3File DOCUMENT ME!
     */
    static public void updateCompositeFromAllTag(FilenameComposite composite,
                                                 MP3File mp3File) {
        composite.matchAgainstTag(mp3File.getID3v1Tag());
        composite.matchAgainstTag(mp3File.getID3v2Tag());
        composite.matchAgainstTag(mp3File.getLyrics3Tag());
    }

    /**
     * DOCUMENT ME!
     *
     * @param token DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    static private String[] parseDelimiter(String token) {
        String[] tokenArray = null;

        if ((token != null) && (token.length() > 0)) {
            Iterator iterator = TagOptionSingleton.getInstance()
                                .getFilenameDelimiterIterator();
            int      index;
            String   delimiter;

            while (iterator.hasNext()) {
                delimiter = (String) iterator.next();
                index     = token.indexOf(delimiter);

                if (index >= 0) {
                    tokenArray    = new String[3];
                    tokenArray[0] = delimiter;
                    tokenArray[1] = token.substring(0, index);
                    tokenArray[2] = token.substring(index + delimiter.length());
                }
            }
        }

        return tokenArray;
    }

    /**
     * DOCUMENT ME!
     *
     * @param token DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    static private String[] parseEndWordDelimiter(String token) {
        String[] tokenArray = null;

        if ((token != null) && (token.length() > 0)) {
            Iterator iterator = TagOptionSingleton.getInstance()
                                .getEndWordDelimiterIterator();
            int      index;
            String   delimiter;

            while (iterator.hasNext()) {
                delimiter = (String) iterator.next();

                if (token.endsWith(delimiter)) {
                    index = token.substring(0,
                                            token.length() -
                                            delimiter.length())
                            .indexOf(delimiter);
                } else {
                    index = token.indexOf(delimiter);
                }

                if (index > 0) {
                    tokenArray    = new String[3];
                    tokenArray[0] = delimiter;
                    tokenArray[1] = token.substring(0, index);
                    tokenArray[2] = token.substring(index);
                }
            }
        }

        return tokenArray;
    }

    /**
     * DOCUMENT ME!
     *
     * @param token DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     *
     * @throws TagException DOCUMENT ME!
     */
    static private String[] parseParenthesis(String token)
                                      throws TagException {
        String[] tokenArray = null;

        if ((token != null) && (token.length() > 0)) {
            TagOptionSingleton option     = TagOptionSingleton.getInstance();
            String             tempOpen   = "";
            String             open       = "";
            String             close      = "";
            int                openIndex  = token.length();
            int                tempIndex;
            int                closeIndex = -1;
            Iterator           iterator   = option.getOpenParenthesisIterator();

            // find first parenthesis
            while (iterator.hasNext()) {
                tempOpen  = (String) iterator.next();
                tempIndex = token.indexOf(tempOpen);

                if ((tempIndex >= 0) && (tempIndex < openIndex)) {
                    openIndex = tempIndex;
                    open      = tempOpen;
                }
            }

            // we have a parenthesis
            if ((openIndex >= 0) && (openIndex < token.length())) {
                close      = option.getCloseParenthesis(open);
                closeIndex = TagUtilities.findMatchingParenthesis(token,
                                                                  openIndex);

                if (closeIndex < 0) {
                    throw new TagException("Unmatched parenthesis in \"" +
                                           token + "\" at position : " +
                                           openIndex);
                }

                tokenArray    = new String[5];
                tokenArray[0] = open;
                tokenArray[1] = close;
                tokenArray[2] = token.substring(0, openIndex);
                tokenArray[3] = token.substring(openIndex + open.length(),
                                                closeIndex);
                tokenArray[4] = token.substring(closeIndex + close.length());
            }
        }

        return tokenArray;
    }

    /**
     * DOCUMENT ME!
     *
     * @param token DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    static private String[] parseStartWordDelimiter(String token) {
        String[] tokenArray = null;

        if ((token != null) && (token.length() > 0)) {
            Iterator iterator = TagOptionSingleton.getInstance()
                                .getStartWordDelimiterIterator();
            int      index;
            String   delimiter;

            while (iterator.hasNext()) {
                delimiter = (String) iterator.next();

                if (token.startsWith(delimiter)) {
                    index = token.indexOf(delimiter,
                                          delimiter.length());
                } else {
                    index = token.indexOf(delimiter);
                }

                if (index > 0) {
                    tokenArray    = new String[3];
                    tokenArray[0] = delimiter;
                    tokenArray[1] = token.substring(0, index);
                    tokenArray[2] = token.substring(index);
                }
            }
        }

        return tokenArray;
    }
}