package com.revasoft.jmp3tagextractor;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.image.BufferedImage;
import java.awt.image.ConvolveOp;
import java.awt.image.Kernel;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.util.Date;

import javax.swing.ImageIcon;

import org.farng.mp3.AbstractMP3FragmentBody;
import org.farng.mp3.MP3File;
import org.farng.mp3.filename.FilenameTag;
import org.farng.mp3.id3.AbstractID3v2;
import org.farng.mp3.id3.AbstractID3v2Frame;

import com.sun.image.codec.jpeg.JPEGCodec;
import com.sun.image.codec.jpeg.JPEGImageEncoder;

/*
 * Utility to extract Album art from MP3 files into individual thumbnail files.
 * Can handle individual files as well as complete directories of files organized as Artist/Album.
 * Can also automatically resize the target file to any arbitrary pixel size.
 * 
 * Requires the Java Music Tag library from: http://javamusictag.sourceforge.net
 * 
 *  
 * Run the program without any parameters to get usage instructions
 * 	java -jar mp3tagextractor.jar 
 * 
 * Created on Jul 30, 2003
 *
 * This code is released under the GPL License 
 * @author Sanjay Madhavan
 * 
 * Version History:
 * 
 * Version 		Date			Who				Comments
 * ----------------------------------------------------------------				
 * V1.0 		30.07.2003      Sanjay          First release
 */
public class Mp3ImageTagExtractor {
	private static String VERSION = "V1.0";
	private String PICTURE_TAG = "APIC\0";
	private String PICTURE_DATA_TAG = "Picture Data";
	private String MIME_TAG = "MIME Type";
	//
	private Mp3ImageTagExtractorOptions options = null;
	//
	private long directoriesProcessed = 0;
	private long filesProcessed = 0;
	private long errors = 0;
	/**
	 * @return
	 */
	public Mp3ImageTagExtractorOptions getOptions() {
		return options;
	}

	/**
	 * @param options
	 */
	public void setOptions(Mp3ImageTagExtractorOptions options) {
		this.options = options;
	}

	//
	/**
	 * Prints the valid usage of this tool.
	 *
	 */
	public static void printUsage() {
		System.out.println("\nMP3ImageTagExtractor Valid usage is: \n");
		System.out.println(
			"\tjava -jar MP3ImageTagExtractor.jar \n\t\t-srcDir d:/src \n\t\t[-targetDir d:/thumbnails] \n\t\t[-fileMask *.mp3] \n\t\t[-targetFileExtension .tbn] \n\t\t[-overwriteExisting] \n\t\t[-skipAlbumThumbnail] \n\t\t[-skipArtistThumbnail] \n\t\t[-skipTrackThumbnail] \n\t\t[-resize 64x64]\n");
		System.out.println("\n\tOR\n");
		System.out.println(
			"\tjava -jar MP3ImageTagExtractor.jar \n\t\t-srcFile d:/src/test.mp3 \n\t\t[-targetFile fileName] \n\t\t[-targetDir d:/thumbnails]  \n\t\t[-targetFileExtension .tbn]  \n\t\t[-overwriteExisting]  \n\t\t[-skipAlbumThumbnail] \n\t\t[-skipArtistThumbnail]  \n\t\t[-resize 64x64]\n");

	}

	/**
	* Launch this tool with the supplied parameter. See the class comment for valid parameters.
	*  
	* @param args
	*/

	public static void main(String[] args) {

		Mp3ImageTagExtractor mp3ImageTagExtractor = new Mp3ImageTagExtractor();
		Mp3ImageTagExtractorOptions mp3ImageTagExtractorOptions =
			new Mp3ImageTagExtractorOptions();
		//
		String logString = "";
		Date startTime = new Date();

		for (int i = 0; i < args.length; i++) {
			String arg = args[i].toLowerCase();
			if (arg.equals("-srcdir")) {
				i++;
				mp3ImageTagExtractorOptions.setSourceDir(args[i]);
			}
			if (arg.equals("-targetdir")) {
				i++;
				mp3ImageTagExtractorOptions.setTargetDirectory(args[i]);
			}
			if (arg.equals("-srcfile")) {
				i++;
				mp3ImageTagExtractorOptions.setSourceFile(args[i]);
			}
			if (arg.equals("-targetfile")) {
				i++;
				mp3ImageTagExtractorOptions.setTargetFile(args[i]);
			}
			if (arg.equals("-filemask")) {
				i++;
				mp3ImageTagExtractorOptions.setFileMask(args[i].toLowerCase());
			}
			if (arg.equals("-targetfileextension")) {
				i++;
				mp3ImageTagExtractorOptions.setTargetFileExtension(
					args[i].toLowerCase());
			}

			if (arg.equals("-overwriteexisting")) {
				i++;
				mp3ImageTagExtractorOptions.setOverwriteExisting(true);
			}
			if (arg.equals("-skipalbumthumbnail")) {
				i++;
				mp3ImageTagExtractorOptions.setSkipCreateAlbumThumbNail(true);
			}
			if (arg.equals("-skiptrackthumbnail")) {
				i++;
				mp3ImageTagExtractorOptions.setSkipCreateTrackThumbNail(true);
			}
			if (arg.equals("-skipartistthumbnail")) {
				i++;
				mp3ImageTagExtractorOptions.setSkipCreateArtistThumbNail(true);
			}
			if (arg.equals("-resize")) {
				i++;
				mp3ImageTagExtractorOptions.setResize(true);
				mp3ImageTagExtractorOptions.parseImageSize(
					args[i].toLowerCase());
			}
		}
		//
		mp3ImageTagExtractor.deleteLog(mp3ImageTagExtractorOptions);
		//
		if (mp3ImageTagExtractorOptions.getSourceFile() != null) {
			logString =
				"MP3ImageTagExtractor: "
					+ VERSION
					+ "\n\t\t -srcFile ["
					+ mp3ImageTagExtractorOptions.getSourceFile()
					+ "] -targetFile ["
					+ (mp3ImageTagExtractorOptions.getTargetFile() == null
						? ""
						: mp3ImageTagExtractorOptions.getTargetFile())
					+ (mp3ImageTagExtractorOptions.getTargetDirectory() == null
						? ""
						: "] -targetDir ["
							+ mp3ImageTagExtractorOptions.getTargetDirectory())
					+ (mp3ImageTagExtractorOptions.getTargetFileExtension()
						== null
						? ""
						: "] -targetFileExtension ["
							+ mp3ImageTagExtractorOptions
								.getTargetFileExtension())
					+ "]\n\t\t -overwriteExisting ["
					+ (mp3ImageTagExtractorOptions.isOverwriteExisting()
						? "true"
						: "false")
					+ "] -skipalbumthumbnail ["
					+ (mp3ImageTagExtractorOptions.isSkipCreateAlbumThumbNail()
						? "true"
						: "false")
					+ "] -skipartistthumbnail ["
					+ (mp3ImageTagExtractorOptions.isSkipCreateArtistThumbNail()
						? "true"
						: "false")
					+ (mp3ImageTagExtractorOptions.isResize()
						? "]\n\t\t -resize [width="
							+ mp3ImageTagExtractorOptions.getResizeWidth()
							+ " height="
							+ mp3ImageTagExtractorOptions.getResizeHeight()
						: "")
					+ "]\n";

			System.out.println(logString);
			mp3ImageTagExtractor.logAppend(
					mp3ImageTagExtractorOptions,
					logString);
			try {
				mp3ImageTagExtractor.extractImageFromFile(
					mp3ImageTagExtractorOptions,
					mp3ImageTagExtractorOptions.getSourceFile(),
					mp3ImageTagExtractorOptions.getTargetFile());
			} catch (Exception e) {
				//Errors are already logged so ignore here
			}
		} else if (mp3ImageTagExtractorOptions.getSourceDir() != null) {
			logString =
				"MP3ImageTagExtractor: "
					+ VERSION
					+ "\n\t\t -srcDir ["
					+ mp3ImageTagExtractorOptions.getSourceDir()
					+ (mp3ImageTagExtractorOptions.getTargetDirectory() == null
						? ""
						: "] -targetDir ["
							+ mp3ImageTagExtractorOptions.getTargetDirectory())
					+ "] -fileMask ["
					+ mp3ImageTagExtractorOptions.getFileMask()
					+ (mp3ImageTagExtractorOptions.getTargetFileExtension()
						== null
						? ""
						: "] -targetFileExtension ["
							+ mp3ImageTagExtractorOptions
								.getTargetFileExtension())
					+ "]\n\t\t -overwriteExisting ["
					+ (mp3ImageTagExtractorOptions.isOverwriteExisting()
						? "true"
						: "false")
					+ "] -skipAlbumThumbnail ["
					+ (mp3ImageTagExtractorOptions.isSkipCreateAlbumThumbNail()
						? "true"
						: "false")
					+ "] -skipArtistThumbnail ["
					+ (mp3ImageTagExtractorOptions.isSkipCreateArtistThumbNail()
						? "true"
						: "false")
					+ "] -skipTrackThumbnail ["
					+ (mp3ImageTagExtractorOptions.isSkipCreateTrackThumbNail()
						? "true"
						: "false")
					+ (mp3ImageTagExtractorOptions.isResize()
						? "]\n\t\t -resize [width="
							+ mp3ImageTagExtractorOptions.getResizeWidth()
							+ " height="
							+ mp3ImageTagExtractorOptions.getResizeHeight()
						: "")
					+ "]\n";
			System.out.println(logString);

			mp3ImageTagExtractor.processFilesInDirectory(
				mp3ImageTagExtractorOptions,
				mp3ImageTagExtractorOptions.getSourceDir());
		} else
			printUsage();

		Date endTime = new Date();
		logString =
			"\nFinished processing "
				+ mp3ImageTagExtractor.directoriesProcessed
				+ " Directories, "
				+ mp3ImageTagExtractor.filesProcessed
				+ " Files, "
				+ mp3ImageTagExtractor.errors
				+ " Errors, "
				+ " Elapsed time = "
				+ getElapsedTime(startTime.getTime(), endTime.getTime());
		System.out.println(logString);

		mp3ImageTagExtractor.logAppend(
			mp3ImageTagExtractorOptions,
			logString);
	}

	/**
	 * Return the elapsed time as a string in hh:mm:ss.ttt format
	 * 
	 * @param startTime
	 * @param startTime
	 * @return
	 */
	private static String getElapsedTime(long startTime, long endTime) {
		long elapsedTime = endTime - startTime;
		long ms = (elapsedTime % 1000);
		long timeInSecs = elapsedTime / 1000;
		long secs = timeInSecs % 60;
		long mins = (timeInSecs / 60) % 60;
		long hours = timeInSecs / 3600;

		return (hours < 10 ? "0" : "")
			+ String.valueOf(hours)
			+ ":"
			+ (mins < 10 ? "0" : "")
			+ String.valueOf(mins)
			+ ":"
			+ (secs < 10 ? "0" : "")
			+ String.valueOf(secs)
			+ "."
			+ (ms < 10 ? "00" : (ms < 100 ? "0" : ""))
			+ String.valueOf(ms);
	}

	/**
	 * Extract the Album art from the specified MP3 file and write the album art to the target file.
	 * The targetFile is suffixed with the mime type extension of the picture data (e.g. .jpg, .gif etc).
	 * If the targetFile is null then derive the targetFileName from the srcFile name
	 * 
	 * @param srcFile
	 * @param targetFile
	 */
	private void extractImageFromFile(
		Mp3ImageTagExtractorOptions mp3ImageTagExtractorOptions,
		String srcFile,
		String targetFileOrNull)
		throws Exception {

		try {
			File sourceFile = new File(srcFile);
			MP3File mp3file = new MP3File(sourceFile);
			FilenameTag fileNameTag = mp3file.getFilenameTag();
			AbstractID3v2 id3v2 = mp3file.getID3v2Tag();
			if (id3v2 != null) {
				AbstractID3v2Frame apic = id3v2.getFrame(PICTURE_TAG);
				if (apic != null) {
					AbstractMP3FragmentBody apicBody = apic.getBody();
					String mimeType = (String) apicBody.getObject(MIME_TAG);
					String fileExtension =
						getFileExtensionFromMimeType(mimeType);
					if (mp3ImageTagExtractorOptions.getTargetFileExtension()
						!= null)
						fileExtension =
							mp3ImageTagExtractorOptions
								.getTargetFileExtension();
					if (fileExtension.charAt(0) == '.')
						fileExtension = fileExtension.substring(1);
					Object bytes = apicBody.getObject(PICTURE_DATA_TAG);
					if (bytes != null) {
						byte[] pictureBytes = (byte[]) bytes;
						String outputFileName = targetFileOrNull;
						if (outputFileName == null) {
							if (mp3ImageTagExtractorOptions
								.getTargetDirectory()
								!= null) {
								outputFileName =
									mp3ImageTagExtractorOptions
										.getTargetDirectory()
										+ File.separator
										+ getUnqualifiedFileName(
											stripFileExtension(srcFile));
							} else
								outputFileName = stripFileExtension(srcFile);
						}
						String fullyQualifiedFileName =
							outputFileName + "." + fileExtension;
						writeFile(
							fullyQualifiedFileName,
							pictureBytes,
							mp3ImageTagExtractorOptions.isOverwriteExisting());
						if (mp3ImageTagExtractorOptions.isResize())
							resizeImage(
								mp3ImageTagExtractorOptions,
								fullyQualifiedFileName,
								fullyQualifiedFileName,
								mp3ImageTagExtractorOptions.getResizeWidth(),
								mp3ImageTagExtractorOptions.getResizeHeight());
					}

				}
			}

		} catch (Exception e) {
			errors++;
			logAppend(
				mp3ImageTagExtractorOptions,
				"Error Processing file: " + srcFile + "\n");
			logStackTrace(mp3ImageTagExtractorOptions, e);
			logAppend(mp3ImageTagExtractorOptions, "\n*\n");
			throw e;

		}

	}

	/**
	 * Dump the exception stack trace to the log file
	 * 
	 * @param e
	 */
	private void logStackTrace(
		Mp3ImageTagExtractorOptions mp3ImageTagExtractorOptions,
		Exception e) {
		PrintWriter pw = getLogPrintWriter(mp3ImageTagExtractorOptions);
		e.printStackTrace(pw);
		pw.flush();
	}

	/**
	 * Return the current log file wrapped with a PrintWriter
	 * 
	 * @return
	 */
	private PrintWriter getLogPrintWriter(Mp3ImageTagExtractorOptions mp3ImageTagExtractorOptions) {
		return new PrintWriter(
			getLogFileOutputStream(mp3ImageTagExtractorOptions));
	}

	/**
	 * Return the current logFile wrapped as an OutputStream
	 * 
	 * @return
	 */
	private OutputStream getLogFileOutputStream(Mp3ImageTagExtractorOptions mp3ImageTagExtractorOptions) {
		try {
			return new FileOutputStream(
				mp3ImageTagExtractorOptions.getLogFile(),
				true);
		} catch (FileNotFoundException e) {
			//e.printStackTrace();
			return null;
		}
	}

	/**
	 * Return just the file name without its parent directory path.
	 * 
	 * @param string
	 * @return
	 */
	private String getUnqualifiedFileName(String fullyQualifiedPath) {
		int ndx1 = fullyQualifiedPath.lastIndexOf('/');
		int ndx2 = fullyQualifiedPath.lastIndexOf('\\');
		int index = ndx2 > ndx1 ? ndx2 : ndx1;

		if (index == -1) {
			return fullyQualifiedPath;
		}

		String name =
			fullyQualifiedPath.substring(
				index + 1,
				fullyQualifiedPath.length());
		return name;
	}

	/**
	 * Dump the specified bytes to the specified file name.
	 * Overwrite the existing file only if overwriteExisiting is true
	 * 
	 * @param string
	 * @param pictureBytes
	 */
	private static void writeFile(
		String targetFileName,
		byte[] pictureBytes,
		boolean overwriteExisiting)
		throws Exception {
		File outFile = new File(targetFileName);
		if (!outFile.exists() || overwriteExisiting) {
			System.out.println("Writing: [" + targetFileName + "]");
			FileOutputStream fos = new FileOutputStream(outFile);
			fos.write(pictureBytes);
			fos.close();

		} else
			System.out.println(
				"File: [" + targetFileName + "] exists. Skipping.");

	}

	/**
	 * Return the file extension from the mime type (e.g. image/jpg will return jpg)
	 * 
	 * @param mimeType
	 * @return
	 */
	private String getFileExtensionFromMimeType(String mimeType) {
		String name = "unknown";
		if (mimeType != null) {
			int idx = mimeType.lastIndexOf('/');

			if (idx > 0)
				name = mimeType.substring(idx + 1, mimeType.length());
		}
		return name;
	}

	/**
	 * Return the filename without the file extension 
	 * 
	 * @param fileName
	 * @return
	 */
	private static String stripFileExtension(String fileName) {
		int idx = fileName.lastIndexOf('.');
		String name = fileName;
		if (idx > 0)
			name = fileName.substring(0, idx);
		return name;
	}

	/**
	 * Extracts the images from the files in specified source directory and subdirectories 
	 * 
	 * @param sourceDirectory
	 * @param targetDirectory
	 */
	public void processFilesInDirectory(
		Mp3ImageTagExtractorOptions mp3ImageTagExtractorOptions,
		String sourceDirectory) {

		File srcDir = new File(sourceDirectory);
		File[] files = srcDir.listFiles();
		for (int i = 0; i < files.length; i++) {
			File file = files[i];
			if (file.isDirectory()) {
				directoriesProcessed++;
				processFilesInDirectory(
					mp3ImageTagExtractorOptions,
					file.getAbsolutePath());
			} else {
				filesProcessed++;
				if (matches(mp3ImageTagExtractorOptions.getFileMask(),
					file.getAbsolutePath())) {
					try {
						if (!mp3ImageTagExtractorOptions
							.isSkipCreateTrackThumbNail())
							extractImageFromFile(
								mp3ImageTagExtractorOptions,
								file.getAbsolutePath(),
								null);
						if (!mp3ImageTagExtractorOptions
							.isSkipCreateAlbumThumbNail()) {
							String albumThumbNail =
								getDirectoryPath(file.getAbsolutePath());
							extractImageFromFile(
								mp3ImageTagExtractorOptions,
								file.getAbsolutePath(),
								albumThumbNail);
							if (!mp3ImageTagExtractorOptions
								.isSkipCreateArtistThumbNail()) {
								String artistThumbNail =
									getDirectoryPath(albumThumbNail);
								extractImageFromFile(
									mp3ImageTagExtractorOptions,
									file.getAbsolutePath(),
									artistThumbNail);
							}

						}
					} catch (Exception e) {
					}
				}
			}
		}

	}

	/**
	 * Return true if the mask matches the target
	 *  
	 * e.g. valid mask is *.mp3, *.mp3pro , etc...
	 */
	public static boolean matches(String mask, String target) {
		return target.toLowerCase().endsWith(mask.substring(1));
	}

	/**
	 * Return parent directory path for the specified file (or directory)
	 *  
	 * @param targetFileName
	 */
	public static String getDirectoryPath(String targetFileName) {
		String dir = targetFileName;

		int ndx1 = targetFileName.lastIndexOf('/');
		int ndx2 = targetFileName.lastIndexOf('\\');
		int index = ndx2 > ndx1 ? ndx2 : ndx1;

		if (index != -1)
			dir = targetFileName.substring(0, index);
		return dir;
	}

	/**
	 * Process the specified sourceFile into the resizedFile target after 
	 * resizing it to the specified dimensions.
	 * 
	 * @param sourceFile
	 * @param resizedFile
	 * @param newWidth
	 * @param newHeight
	 * @throws Exception
	 */
	public void resizeImage(
		Mp3ImageTagExtractorOptions mp3ImageTagExtractorOptions,
		String sourceFile,
		String resizedFile,
		int newWidth,
		int newHeight)
		throws Exception {

		float factor = 1.0f;
		try {
			// get original image
			Image originalImage = new ImageIcon(sourceFile).getImage();
			Image resizedImage = null;

			int imageWidth = originalImage.getWidth(null);
			int imageHeight = originalImage.getHeight(null);

			// optimization skip resize if targetsize is required size
			if (imageHeight == newHeight && imageWidth == newWidth) {
				if (sourceFile.equals(resizedFile))
					return; // nothing to do
				else {
					copyFile(
						sourceFile,
						resizedFile,
						mp3ImageTagExtractorOptions.isOverwriteExisting());
					// just copy input to output
					return;
				}

			}
			if (imageWidth > newWidth) {
				factor = (float) newWidth / (float) imageWidth;
				factor = 0.0f - factor;
			} else {
				factor = (float) imageWidth / (float) newWidth;
			}

			resizedImage =
				originalImage.getScaledInstance(
					newWidth,
					newHeight,
					Image.SCALE_SMOOTH);

			originalImage.flush();

			// this code ensures that all the pixels in the image are loaded
			Image temp = new ImageIcon(resizedImage).getImage();

			// create the buffered image
			BufferedImage bufferedImage =
				new BufferedImage(
					temp.getWidth(null),
					temp.getHeight(null),
					BufferedImage.TYPE_INT_RGB);

			// copy image to buffered image
			Graphics g = bufferedImage.createGraphics();

			// clear background and paint the image
			g.setColor(Color.white);
			g.fillRect(0, 0, temp.getWidth(null), temp.getHeight(null));
			g.drawImage(temp, 0, 0, null);
			g.dispose();

			// sharpen/soften (depending on factor (-ve for sharpening, +ve	for softening)
			float[] array =
				{
					0,
					factor,
					0,
					factor,
					1 - (factor * 4),
					factor,
					0,
					factor,
					0 };
			Kernel kernel = new Kernel(3, 3, array);
			ConvolveOp cOp =
				new ConvolveOp(kernel, ConvolveOp.EDGE_NO_OP, null);
			bufferedImage = cOp.filter(bufferedImage, null);

			// write the jpeg to a file
			OutputStream out = new FileOutputStream(resizedFile);

			// encodes image as a JPEG data stream
			JPEGImageEncoder encoder = JPEGCodec.createJPEGEncoder(out);
			com.sun.image.codec.jpeg.JPEGEncodeParam param =
				encoder.getDefaultJPEGEncodeParam(bufferedImage);
			param.setQuality(0.7f, true);
			encoder.setJPEGEncodeParam(param);
			encoder.encode(bufferedImage);

			out.close();
		} catch (Exception e) {
			throw e;
		}
	}

	/**
	 * Read the contents from the specified file name.
	 * 
	 * @param file
	 * @return byte[]
	 * @throws IOException
	 */
	public static byte[] readFile(File file) throws IOException {
		InputStream is = new FileInputStream(file);
		// Get the size of the file
		long length = file.length();

		if (length > Integer.MAX_VALUE) {
			// File is too large
		}

		// Create the byte array to hold the data
		byte[] bytes = new byte[(int) length];

		// Read in the bytes
		int offset = 0;
		int numRead = 0;
		while (offset < bytes.length
			&& (numRead = is.read(bytes, offset, bytes.length - offset)) >= 0) {
			offset += numRead;
		}

		// Ensure all the bytes have been read in
		if (offset < bytes.length) {
			throw new IOException(
				"Could not completely read file " + file.getName());
		}

		// Close the input stream and return bytes
		is.close();
		return bytes;
	}

	/**
	 * Copy the contents of the sourceFile to the targetFile
	 * 
	 * @param sourceFile
	 * @param resizedFile
	 */
	private static void copyFile(
		String sourceFile,
		String resizedFile,
		boolean overWriteExisting)
		throws IOException, Exception {
		writeFile(
			resizedFile,
			readFile(new File(sourceFile)),
			overWriteExisting);

	}

	/**
	 * Remove the current logFile
	 *
	 */
	private void deleteLog(Mp3ImageTagExtractorOptions mp3ImageTagExtractorOptions) {
		if (mp3ImageTagExtractorOptions.getLogFile().exists()) {
			mp3ImageTagExtractorOptions.getLogFile().delete();
		}
	}
	/**
	 * Append the specified string to the current logFile.
	 * Any exceptions while writing to the file are ignored.
	 * 
	 * @param logString
	 */
	private void logAppend(
		Mp3ImageTagExtractorOptions mp3ImageTagExtractorOptions,
		String logString) {
		try {
			if (!mp3ImageTagExtractorOptions.getLogFile().exists()) {
				mp3ImageTagExtractorOptions.getLogFile().createNewFile();
			}
			OutputStream fos =
				getLogFileOutputStream(mp3ImageTagExtractorOptions);
			fos.write(logString.getBytes());
			fos.close();
		} catch (Exception e) {
			//e.printStackTrace();
		}
	}
}
