/*
 * Copyright (C) 2003 Sanjay Madhavan.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package com.revasoft.jmp3tagextractor;

import java.io.File;

/**
 * Utility class to capture all options that can be used while invoking the Mp3ImageTagExtractor
 * 
 * @author Sanjay Madhavan
 *
 */
public class Mp3ImageTagExtractorOptions {
	private String fileMask = "*.mp3";
	private String targetFileExtension = null;
	private String targetDirectory = null;
	//
	private boolean overwriteExisting = false;
	//
	private boolean skipCreateAlbumThumbNail = false;
	private boolean skipCreateArtistThumbNail = false;
	private boolean skipCreateTrackThumbNail = false;
	//	
	private boolean resize = false;
	private int resizeWidth = 64;
	private int resizeHeight = 64;
	//
	private String logFileName = "Mp3ImageTagExtractor.log";
	private File logFile = null;

	private String sourceDir = null;
	private String sourceFile = null;
	private String targetFile = null;

	private Mp3ImageTagExtractorOptions() {
	}

	/**
	 * Parse the startup parameters and return an Mp3ImageTagExtractorOptions instance that
	 * contains all valid parameters.
	 *  
	 * @param args
	 * @return
	 */
	public Mp3ImageTagExtractorOptions(String[] args) {

		for (int i = 0; i < args.length; i++) {
			String arg = args[i].toLowerCase();
			if (arg.equalsIgnoreCase("-srcDir")) {
				i++;
				setSourceDir(args[i]);
			} else if (arg.equalsIgnoreCase("-targetDir")) {
				i++;
				setTargetDirectory(args[i]);
			} else if (arg.equalsIgnoreCase("-srcFile")) {
				i++;
				setSourceFile(args[i]);
			} else if (arg.equalsIgnoreCase("-targetFile")) {
				i++;
				setTargetFile(args[i]);
			} else if (arg.equalsIgnoreCase("-fileMask")) {
				i++;
				setFileMask(args[i].toLowerCase());
			} else if (arg.equalsIgnoreCase("-targetFileExtension")) {
				i++;
				setTargetFileExtension(args[i].toLowerCase());
			} else if (arg.equalsIgnoreCase("-overwriteExisting")) {
				i++;
				setOverwriteExisting(true);
			} else if (arg.equalsIgnoreCase("-skipAlbumThumbnail")) {
				i++;
				setSkipCreateAlbumThumbNail(true);
			} else if (arg.equalsIgnoreCase("-skipTrackThumbnail")) {
				i++;
				setSkipCreateTrackThumbNail(true);
			} else if (arg.equalsIgnoreCase("-skipArtistThumbnail")) {
				i++;
				setSkipCreateArtistThumbNail(true);
			} else if (arg.equalsIgnoreCase("-resize")) {
				i++;
				setResize(true);
				parseImageSize(args[i].toLowerCase());
			} else {
				System.out.println(
					"WARNING: Ignoring Unknown parameter: " + args[i]);
			}
		}

	}
	/**
	 * Return a string containing all configured options and their values.
	 * 
	 * @return
	 */
	public String getLogString() {
		String logString = "";
		if (getSourceFile() != null) {
			logString =
				"MP3ImageTagExtractor: "
					+ Mp3ImageTagExtractor.VERSION
					+ "\n\t\t -srcFile ["
					+ getSourceFile()
					+ "] -targetFile ["
					+ (getTargetFile() == null ? "" : getTargetFile())
					+ (getTargetDirectory() == null
						? ""
						: "] -targetDir [" + getTargetDirectory())
					+ (getTargetFileExtension() == null
						? ""
						: "] -targetFileExtension [" + getTargetFileExtension())
					+ "]\n\t\t -overwriteExisting ["
					+ (isOverwriteExisting() ? "true" : "false")
					+ "] -skipAlbumThumbnail ["
					+ (isSkipCreateAlbumThumbNail() ? "true" : "false")
					+ "] -skipArtistThumbnail ["
					+ (isSkipCreateArtistThumbNail() ? "true" : "false")
					+ (isResize()
						? "]\n\t\t -resize [width="
							+ getResizeWidth()
							+ " height="
							+ getResizeHeight()
						: "")
					+ "]\n";
		} else {

			logString =
				"MP3ImageTagExtractor: "
					+ Mp3ImageTagExtractor.VERSION
					+ "\n\t\t -srcDir ["
					+ getSourceDir()
					+ (getTargetDirectory() == null
						? ""
						: "] -targetDir [" + getTargetDirectory())
					+ "] -fileMask ["
					+ getFileMask()
					+ (getTargetFileExtension() == null
						? ""
						: "] -targetFileExtension [" + getTargetFileExtension())
					+ "]\n\t\t -overwriteExisting ["
					+ (isOverwriteExisting() ? "true" : "false")
					+ "] -skipAlbumThumbnail ["
					+ (isSkipCreateAlbumThumbNail() ? "true" : "false")
					+ "] -skipArtistThumbnail ["
					+ (isSkipCreateArtistThumbNail() ? "true" : "false")
					+ "] -skipTrackThumbnail ["
					+ (isSkipCreateTrackThumbNail() ? "true" : "false")
					+ (isResize()
						? "]\n\t\t -resize [width="
							+ getResizeWidth()
							+ " height="
							+ getResizeHeight()
						: "")
					+ "]\n";

		}

		return logString;

	}

	/**
	 * Expects an sizeString in format WidthxHeight (e.g. 64x75 = width=64, height=75)
	 * 
	 * @param sizeString
	 */
	public void parseImageSize(String sizeString) {
		int idx = sizeString.indexOf('x');
		if (idx > 0 && idx < sizeString.length()) {
			setResizeWidth(Integer.parseInt(sizeString.substring(0, idx)));
			setResizeHeight(
				Integer.parseInt(
					sizeString.substring(idx + 1, sizeString.length())));

		}

	}
	/**
	 * @return
	 */
	public String getSourceDir() {
		return sourceDir;
	}

	/**
	 * @param sourceDir
	 */
	public void setSourceDir(String sourceDir) {
		this.sourceDir = sourceDir;
	}

	/**
	 * @return
	 */
	public String getSourceFile() {
		return sourceFile;
	}

	/**
	 * @param sourceFile
	 */
	public void setSourceFile(String sourceFile) {
		this.sourceFile = sourceFile;
	}

	/**
	 * @return
	 */
	public String getTargetFile() {
		return targetFile;
	}

	/**
	 * @param targetFile
	 */
	public void setTargetFile(String targetFile) {
		this.targetFile = targetFile;
	}

	/**
	 * @return
	 */
	public String getFileMask() {
		return fileMask;
	}

	/**
	 * @param fileMask
	 */
	public void setFileMask(String fileMask) {
		this.fileMask = fileMask;
	}

	/**
	 * @return
	 */
	public File getLogFile() {
		if (logFile == null)
			logFile = new File(getLogFileName());
		return logFile;
	}

	/**
	 * @param logFile
	 */
	public void setLogFile(File logFile) {
		this.logFile = logFile;
	}

	/**
	 * @return
	 */
	public String getLogFileName() {
		return logFileName;
	}

	/**
	 * @param logFileName
	 */
	public void setLogFileName(String logFileName) {
		this.logFileName = logFileName;
	}

	/**
	 * @return
	 */
	public boolean isOverwriteExisting() {
		return overwriteExisting;
	}

	/**
	 * @param overwriteExisting
	 */
	public void setOverwriteExisting(boolean overwriteExisting) {
		this.overwriteExisting = overwriteExisting;
	}

	/**
	 * @return
	 */
	public boolean isResize() {
		return resize;
	}

	/**
	 * @param resize
	 */
	public void setResize(boolean resize) {
		this.resize = resize;
	}

	/**
	 * @return
	 */
	public int getResizeHeight() {
		return resizeHeight;
	}

	/**
	 * @param resizeHeight
	 */
	public void setResizeHeight(int resizeHeight) {
		this.resizeHeight = resizeHeight;
	}

	/**
	 * @return
	 */
	public int getResizeWidth() {
		return resizeWidth;
	}

	/**
	 * @param resizeWidth
	 */
	public void setResizeWidth(int resizeWidth) {
		this.resizeWidth = resizeWidth;
	}

	/**
	 * @return
	 */
	public boolean isSkipCreateAlbumThumbNail() {
		return skipCreateAlbumThumbNail;
	}

	/**
	 * @param skipCreateAlbumThumbNail
	 */
	public void setSkipCreateAlbumThumbNail(boolean skipCreateAlbumThumbNail) {
		this.skipCreateAlbumThumbNail = skipCreateAlbumThumbNail;
	}

	/**
	 * @return
	 */
	public boolean isSkipCreateArtistThumbNail() {
		return skipCreateArtistThumbNail;
	}

	/**
	 * @param skipCreateArtistThumbNail
	 */
	public void setSkipCreateArtistThumbNail(boolean skipCreateArtistThumbNail) {
		this.skipCreateArtistThumbNail = skipCreateArtistThumbNail;
	}

	/**
	 * @return
	 */
	public boolean isSkipCreateTrackThumbNail() {
		return skipCreateTrackThumbNail;
	}

	/**
	 * @param skipCreateTrackThumbNail
	 */
	public void setSkipCreateTrackThumbNail(boolean skipCreateTrackThumbNail) {
		this.skipCreateTrackThumbNail = skipCreateTrackThumbNail;
	}

	/**
	 * @return
	 */
	public String getTargetDirectory() {
		return targetDirectory;
	}

	/**
	 * @param targetDirectory
	 */
	public void setTargetDirectory(String targetDirectory) {
		this.targetDirectory = targetDirectory;
	}

	/**
	 * @return
	 */
	public String getTargetFileExtension() {
		return targetFileExtension;
	}

	/**
	 * @param targetFileExtension
	 */
	public void setTargetFileExtension(String targetFileExtension) {
		this.targetFileExtension = targetFileExtension;
	}

}
