/*
 * Copyright (C) 2003 Sanjay Madhavan.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package com.revasoft.jmp3tagextractor;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.image.BufferedImage;
import java.awt.image.ConvolveOp;
import java.awt.image.Kernel;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.util.Date;

import javax.swing.ImageIcon;

// Imports from the Java Music Tag library see: http://javamusictag.sourceforge.net
import org.farng.mp3.AbstractMP3FragmentBody;
import org.farng.mp3.MP3File;
import org.farng.mp3.filename.FilenameTag;
import org.farng.mp3.id3.AbstractID3v2;
import org.farng.mp3.id3.AbstractID3v2Frame;
//
import com.sun.image.codec.jpeg.JPEGCodec;
import com.sun.image.codec.jpeg.JPEGImageEncoder;

/*
 * Utility to extract Album art from MP3 files into individual thumbnail files.
 * Can handle individual files as well as complete directories of files organized as Artist/Album.
 * Can also automatically resize the target file to any arbitrary pixel size.
 * 
 * Requires the Java Music Tag library from: http://javamusictag.sourceforge.net
 * 
 *  
 * Run the program without any parameters to get usage instructions
 * 	java -jar mp3tagextractor.jar 
 * 
 * Created on Jul 30, 2003
 *
 * This code is released under the GPL License 
 * @author Sanjay Madhavan
 * 
 * Version History:
 * 
 * Version 		Date			Who				Comments
 * ----------------------------------------------------------------				
 * V1.0 		30.07.2003      Sanjay          First release
 */
public class Mp3ImageTagExtractor {
	//
	public static String VERSION = "V1.0";
	//
	private String PICTURE_TAG = "APIC\0";
	private String PICTURE_DATA_TAG = "Picture Data";
	private String MIME_TAG = "MIME Type";
	//
	private Mp3ImageTagExtractorOptions options = null;
	//
	private long directoriesProcessed = 0;
	private long filesProcessed = 0;
	private long errors = 0;

	/**
	 * Return a new instance of an extractor initialzed on the specified options.
	 */
	public Mp3ImageTagExtractor(Mp3ImageTagExtractorOptions mp3ImageTagExtractorOptions) {
		setOptions(mp3ImageTagExtractorOptions);
		deleteLog();
	}

	/**
	 * Prints the valid usage of this tool.
	 *
	 */
	public static void printUsage() {
		System.out.println("\nMP3ImageTagExtractor Valid usage is: \n");
		System.out.println(
			"\tjava -jar MP3ImageTagExtractor.jar \n\t\t-srcDir d:/src \n\t\t[-targetDir d:/thumbnails] \n\t\t[-fileMask *.mp3] \n\t\t[-targetFileExtension .tbn] \n\t\t[-overwriteExisting] \n\t\t[-skipAlbumThumbnail] \n\t\t[-skipArtistThumbnail] \n\t\t[-skipTrackThumbnail] \n\t\t[-resize 64x64]\n");
		System.out.println("\n\tOR\n");
		System.out.println(
			"\tjava -jar MP3ImageTagExtractor.jar \n\t\t-srcFile d:/src/test.mp3 \n\t\t[-targetFile fileName] \n\t\t[-targetDir d:/thumbnails]  \n\t\t[-targetFileExtension .tbn]  \n\t\t[-overwriteExisting]  \n\t\t[-skipAlbumThumbnail] \n\t\t[-skipArtistThumbnail]  \n\t\t[-resize 64x64]\n");

	}

	/**
	* Launch this tool with the supplied parameters.
	*  
	* @param args
	*/

	public static void main(String[] args) {

		Date startTime = new Date();
		//

		Mp3ImageTagExtractorOptions options =
			new Mp3ImageTagExtractorOptions(args);
		Mp3ImageTagExtractor extractor = new Mp3ImageTagExtractor(options);
		//

		//
		String logString = options.getLogString();
		System.out.println(logString);
		extractor.logAppend(logString);

		//
		if (options.getSourceFile() != null) {
			try {
				extractor.extractImageFromFile(
					options.getSourceFile(),
					options.getTargetFile());
			} catch (Exception e) {
				//Errors are already logged so ignore here
			}
		} else if (options.getSourceDir() != null) {
			extractor.processFilesInDirectory(options.getSourceDir());
		} else
			printUsage();
		//
		Date endTime = new Date();
		logString =
			"\nFinished processing "
				+ extractor.directoriesProcessed
				+ " Directories, "
				+ extractor.filesProcessed
				+ " Files, "
				+ extractor.errors
				+ " Errors, "
				+ " Elapsed time = "
				+ getElapsedTime(startTime.getTime(), endTime.getTime());
		System.out.println(logString);
		extractor.logAppend(logString);
	}

	/**
	 * Return the elapsed time as a string in hh:mm:ss.ttt format
	 * 
	 * @param startTime
	 * @param startTime
	 * @return
	 */
	private static String getElapsedTime(long startTime, long endTime) {
		long elapsedTime = endTime - startTime;
		long ms = (elapsedTime % 1000);
		long timeInSecs = elapsedTime / 1000;
		long secs = timeInSecs % 60;
		long mins = (timeInSecs / 60) % 60;
		long hours = timeInSecs / 3600;

		return (hours < 10 ? "0" : "")
			+ String.valueOf(hours)
			+ ":"
			+ (mins < 10 ? "0" : "")
			+ String.valueOf(mins)
			+ ":"
			+ (secs < 10 ? "0" : "")
			+ String.valueOf(secs)
			+ "."
			+ (ms < 10 ? "00" : (ms < 100 ? "0" : ""))
			+ String.valueOf(ms);
	}

	/**
	 * Extract the Album art from the specified MP3 file and write the album art to the target file.
	 * The targetFile is suffixed with the mime type extension of the picture data (e.g. .jpg, .gif etc).
	 * If the targetFile is null then derive the targetFileName from the srcFile name
	 * 
	 * @param srcFile
	 * @param targetFile
	 */
	private void extractImageFromFile(String srcFile, String targetFileOrNull)
		throws Exception {

		try {
			File sourceFile = new File(srcFile);
			MP3File mp3file = new MP3File(sourceFile);
			FilenameTag fileNameTag = mp3file.getFilenameTag();
			AbstractID3v2 id3v2 = mp3file.getID3v2Tag();
			if (id3v2 != null) {
				AbstractID3v2Frame apic = id3v2.getFrame(PICTURE_TAG);
				if (apic != null) {
					AbstractMP3FragmentBody apicBody = apic.getBody();
					String mimeType = (String) apicBody.getObject(MIME_TAG);
					String fileExtension =
						getFileExtensionFromMimeType(mimeType);
					if (getOptions().getTargetFileExtension() != null)
						fileExtension = getOptions().getTargetFileExtension();
					if (fileExtension.charAt(0) == '.')
						fileExtension = fileExtension.substring(1);
					Object bytes = apicBody.getObject(PICTURE_DATA_TAG);
					if (bytes != null) {
						byte[] pictureBytes = (byte[]) bytes;
						String outputFileName = targetFileOrNull;
						if (outputFileName == null) {
							if (getOptions().getTargetDirectory() != null) {
								outputFileName =
									getOptions().getTargetDirectory()
										+ File.separator
										+ getUnqualifiedFileName(
											stripFileExtension(srcFile));
							} else
								outputFileName = stripFileExtension(srcFile);
						}
						String fullyQualifiedFileName =
							outputFileName + "." + fileExtension;
						writeFile(
							fullyQualifiedFileName,
							pictureBytes,
							getOptions().isOverwriteExisting());
						if (getOptions().isResize())
							resizeImage(
								getOptions(),
								fullyQualifiedFileName,
								fullyQualifiedFileName,
								getOptions().getResizeWidth(),
								getOptions().getResizeHeight());
					}

				}
			}

		} catch (Exception e) {
			errors++;
			logAppend("Error Processing file: " + srcFile + "\n");
			logStackTrace(e);
			logAppend("\n*\n");
			throw e;

		}

	}

	/**
	 * Dump the exception stack trace to the log file
	 * 
	 * @param e
	 */
	private void logStackTrace(Exception e) {
		PrintWriter pw = getLogPrintWriter();
		e.printStackTrace(pw);
		pw.flush();
	}

	/**
	 * Return the current log file wrapped with a PrintWriter
	 * 
	 * @return
	 */
	private PrintWriter getLogPrintWriter() {
		return new PrintWriter(getLogFileOutputStream());
	}

	/**
	 * Return the current logFile wrapped as an OutputStream
	 * 
	 * @return
	 */
	private OutputStream getLogFileOutputStream() {
		try {
			return new FileOutputStream(getOptions().getLogFile(), true);
		} catch (FileNotFoundException e) {
			//e.printStackTrace();
			return null;
		}
	}

	/**
	 * Return just the file name without its parent directory path.
	 * 
	 * @param string
	 * @return
	 */
	private String getUnqualifiedFileName(String fullyQualifiedPath) {
		int ndx1 = fullyQualifiedPath.lastIndexOf('/');
		int ndx2 = fullyQualifiedPath.lastIndexOf('\\');
		int index = ndx2 > ndx1 ? ndx2 : ndx1;

		if (index == -1) {
			return fullyQualifiedPath;
		}

		String name =
			fullyQualifiedPath.substring(
				index + 1,
				fullyQualifiedPath.length());
		return name;
	}

	/**
	 * Dump the specified bytes to the specified file name.
	 * Overwrite the existing file only if overwriteExisiting is true
	 * 
	 * @param string
	 * @param pictureBytes
	 */
	private static void writeFile(
		String targetFileName,
		byte[] pictureBytes,
		boolean overwriteExisiting)
		throws Exception {
		File outFile = new File(targetFileName);
		if (!outFile.exists() || overwriteExisiting) {
			System.out.println("Writing: [" + targetFileName + "]");
			FileOutputStream fos = new FileOutputStream(outFile);
			fos.write(pictureBytes);
			fos.close();

		} else
			System.out.println(
				"File: [" + targetFileName + "] exists. Skipping.");

	}

	/**
	 * Return the file extension from the mime type (e.g. image/jpg will return jpg)
	 * 
	 * @param mimeType
	 * @return
	 */
	private String getFileExtensionFromMimeType(String mimeType) {
		String name = "unknown";
		if (mimeType != null) {
			int idx = mimeType.lastIndexOf('/');

			if (idx > 0)
				name = mimeType.substring(idx + 1, mimeType.length());
		}
		return name;
	}

	/**
	 * Return the filename without the file extension 
	 * 
	 * @param fileName
	 * @return
	 */
	private static String stripFileExtension(String fileName) {
		int idx = fileName.lastIndexOf('.');
		String name = fileName;
		if (idx > 0)
			name = fileName.substring(0, idx);
		return name;
	}

	/**
	 * Extracts the images from the files in specified source directory and subdirectories 
	 * 
	 * @param sourceDirectory
	 * @param targetDirectory
	 */
	private void processFilesInDirectory(String sourceDirectory) {

		File srcDir = new File(sourceDirectory);
		File[] files = srcDir.listFiles();
		for (int i = 0; i < files.length; i++) {
			File file = files[i];
			if (file.isDirectory()) {
				directoriesProcessed++;
				processFilesInDirectory(file.getAbsolutePath());
			} else {
				filesProcessed++;
				if (matches(getOptions().getFileMask(),
					file.getAbsolutePath())) {
					try {
						if (!getOptions().isSkipCreateTrackThumbNail())
							extractImageFromFile(file.getAbsolutePath(), null);
						if (!getOptions().isSkipCreateAlbumThumbNail()) {
							String albumThumbNail =
								getDirectoryPath(file.getAbsolutePath());
							extractImageFromFile(
								file.getAbsolutePath(),
								albumThumbNail);
							if (!getOptions().isSkipCreateArtistThumbNail()) {
								String artistThumbNail =
									getDirectoryPath(albumThumbNail);
								extractImageFromFile(
									file.getAbsolutePath(),
									artistThumbNail);
							}

						}
					} catch (Exception e) {
					}
				}
			}
		}

	}

	/**
	 * Return true if the mask matches the target
	 *  
	 * e.g. valid mask is *.mp3, *.mp3pro , etc...
	 */
	private static boolean matches(String mask, String target) {
		return target.toLowerCase().endsWith(mask.substring(1));
	}

	/**
	 * Return parent directory path for the specified file (or directory)
	 *  
	 * @param targetFileName
	 */
	private static String getDirectoryPath(String targetFileName) {
		String dir = targetFileName;

		int ndx1 = targetFileName.lastIndexOf('/');
		int ndx2 = targetFileName.lastIndexOf('\\');
		int index = ndx2 > ndx1 ? ndx2 : ndx1;

		if (index != -1)
			dir = targetFileName.substring(0, index);
		return dir;
	}

	/**
	 * Process the specified sourceFile into the resizedFile target after 
	 * resizing it to the specified dimensions.
	 * 
	 * @param sourceFile
	 * @param resizedFile
	 * @param newWidth
	 * @param newHeight
	 * @throws Exception
	 */
	private void resizeImage(
		Mp3ImageTagExtractorOptions mp3ImageTagExtractorOptions,
		String sourceFile,
		String resizedFile,
		int newWidth,
		int newHeight)
		throws Exception {

		float factor = 1.0f;
		try {
			// get original image
			Image originalImage = new ImageIcon(sourceFile).getImage();
			Image resizedImage = null;

			int imageWidth = originalImage.getWidth(null);
			int imageHeight = originalImage.getHeight(null);

			// optimization skip resize if targetsize is required size
			if (imageHeight == newHeight && imageWidth == newWidth) {
				if (sourceFile.equals(resizedFile))
					return; // nothing to do
				else {
					copyFile(
						sourceFile,
						resizedFile,
						mp3ImageTagExtractorOptions.isOverwriteExisting());
					// just copy input to output
					return;
				}

			}
			if (imageWidth > newWidth) {
				factor = (float) newWidth / (float) imageWidth;
				factor = 0.0f - factor;
			} else {
				factor = (float) imageWidth / (float) newWidth;
			}

			resizedImage =
				originalImage.getScaledInstance(
					newWidth,
					newHeight,
					Image.SCALE_SMOOTH);

			originalImage.flush();

			// this code ensures that all the pixels in the image are loaded
			Image temp = new ImageIcon(resizedImage).getImage();

			// create the buffered image
			BufferedImage bufferedImage =
				new BufferedImage(
					temp.getWidth(null),
					temp.getHeight(null),
					BufferedImage.TYPE_INT_RGB);

			// copy image to buffered image
			Graphics g = bufferedImage.createGraphics();

			// clear background and paint the image
			g.setColor(Color.white);
			g.fillRect(0, 0, temp.getWidth(null), temp.getHeight(null));
			g.drawImage(temp, 0, 0, null);
			g.dispose();

			// sharpen/soften (depending on factor (-ve for sharpening, +ve	for softening)
			float[] array =
				{
					0,
					factor,
					0,
					factor,
					1 - (factor * 4),
					factor,
					0,
					factor,
					0 };
			Kernel kernel = new Kernel(3, 3, array);
			ConvolveOp cOp =
				new ConvolveOp(kernel, ConvolveOp.EDGE_NO_OP, null);
			bufferedImage = cOp.filter(bufferedImage, null);

			// write the jpeg to a file
			OutputStream out = new FileOutputStream(resizedFile);

			// encodes image as a JPEG data stream
			JPEGImageEncoder encoder = JPEGCodec.createJPEGEncoder(out);
			com.sun.image.codec.jpeg.JPEGEncodeParam param =
				encoder.getDefaultJPEGEncodeParam(bufferedImage);
			param.setQuality(0.7f, true);
			encoder.setJPEGEncodeParam(param);
			encoder.encode(bufferedImage);

			out.close();
		} catch (Exception e) {
			throw e;
		}
	}

	/**
	 * Read the contents from the specified file name.
	 * 
	 * @param file
	 * @return byte[]
	 * @throws IOException
	 */
	private static byte[] readFile(File file) throws IOException {
		InputStream is = new FileInputStream(file);
		// Get the size of the file
		long length = file.length();

		if (length > Integer.MAX_VALUE) {
			// File is too large
		}

		// Create the byte array to hold the data
		byte[] bytes = new byte[(int) length];

		// Read in the bytes
		int offset = 0;
		int numRead = 0;
		while (offset < bytes.length
			&& (numRead = is.read(bytes, offset, bytes.length - offset)) >= 0) {
			offset += numRead;
		}

		// Ensure all the bytes have been read in
		if (offset < bytes.length) {
			throw new IOException(
				"Could not completely read file " + file.getName());
		}

		// Close the input stream and return bytes
		is.close();
		return bytes;
	}

	/**
	 * Copy the contents of the sourceFile to the targetFile
	 * 
	 * @param sourceFile
	 * @param resizedFile
	 */
	private static void copyFile(
		String sourceFile,
		String resizedFile,
		boolean overWriteExisting)
		throws IOException, Exception {
		writeFile(
			resizedFile,
			readFile(new File(sourceFile)),
			overWriteExisting);

	}

	/**
	 * Remove the current logFile
	 *
	 */
	private void deleteLog() {
		if (getOptions().getLogFile().exists()) {
			getOptions().getLogFile().delete();
		}
	}

	/**
	 * Append the specified string to the current logFile.
	 * Any exceptions while writing to the file are ignored.
	 * 
	 * @param logString
	 */
	private void logAppend(String logString) {
		try {
			if (!getOptions().getLogFile().exists()) {
				getOptions().getLogFile().createNewFile();
			}
			OutputStream fos = getLogFileOutputStream();
			fos.write(logString.getBytes());
			fos.close();
		} catch (Exception e) {
			//e.printStackTrace();
		}
	}

	/**
	 * Return the options object that contains all parameters used during exeution
	 * 
	 * @return
	 */
	private Mp3ImageTagExtractorOptions getOptions() {
		return options;
	}

	/**
	 * Set the options object that contains all parameters used during exeution
	 * 
	 * @param options
	 */
	private void setOptions(Mp3ImageTagExtractorOptions options) {
		this.options = options;
	}
}
