/*
 * MP3 Tag library. It includes an implementation of the ID3 tags and Lyrics3
 * tags as they are defined at www.id3.org
 *
 * Copyright (C) Eric Farng 2003
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.farng.mp3.lyrics3;

import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.HashMap;
import java.util.Iterator;

import org.farng.mp3.AbstractMP3Tag;
import org.farng.mp3.InvalidTagException;
import org.farng.mp3.TagException;
import org.farng.mp3.TagNotFoundException;
import org.farng.mp3.TagOptionSingleton;
import org.farng.mp3.id3.AbstractFrameBodyTextInformation;
import org.farng.mp3.id3.AbstractID3;
import org.farng.mp3.id3.AbstractID3v2Frame;
import org.farng.mp3.id3.FrameBodyCOMM;
import org.farng.mp3.id3.FrameBodySYLT;
import org.farng.mp3.id3.FrameBodyUSLT;
import org.farng.mp3.id3.ID3v1;
import org.farng.mp3.id3.ID3v2_4;


/**
 * Title:       Lyrics3v2 Description: This class is a Lyrics3 2.00 tag
 * Copyright:   Copyright (c) 2002 Company:
 *
 * @author Eric Farng
 * @version $Revision: 1.1 $
 */
public class Lyrics3v2
    extends AbstractLyrics3 {
    /** DOCUMENT ME! */
    private HashMap fieldMap = new HashMap();

    /**
     * Creates a new Lyrics3v2 object.
     */
    public Lyrics3v2() {}

    /**
     * Creates a new Lyrics3v2 object.
     *
     * @param mp3tag DOCUMENT ME!
     */
    public Lyrics3v2(AbstractMP3Tag mp3tag) {
        if (mp3tag != null) {
            if (mp3tag instanceof AbstractLyrics3) {
                // upgrade the tag to lyrics3v2
                if (mp3tag instanceof Lyrics3v2) {
                    Lyrics3v2 lyricOld = (Lyrics3v2) mp3tag;
                    this.fieldMap = lyricOld.fieldMap;
                } else if (mp3tag instanceof Lyrics3v1) {
                    Lyrics3v1      lyricOld = (Lyrics3v1) mp3tag;
                    Lyrics3v2Field newField;
                    newField = new Lyrics3v2Field(new FieldBodyLYR(lyricOld.getLyric()));
                    fieldMap.put(newField.getIdentifier(),
                                 newField);
                }
            } else if (mp3tag instanceof AbstractID3) {
                ID3v2_4                          id3tag;
                AbstractFrameBodyTextInformation textFrame;
                Lyrics3v2Field                   newField;
                String                           text;

                // if the tag is id3v1 or id3v2, id3v2_4 constructor
                // will convert it all to id32_4
                id3tag = new ID3v2_4(mp3tag);

                // not sure why i didn't put this conversion in the field constructor
                // similar to how i put them in the frame constructor for id3v2
                if (id3tag.hasFrameOfType("USLT") ||
                        id3tag.hasFrameOfType("SYLT")) {
                    FieldBodyLYR       body          = new FieldBodyLYR("");
                    AbstractID3v2Frame frame;
                    newField = new Lyrics3v2Field(body);

                    Iterator      frameIterator = id3tag.getFrameOfType("USLT");
                    FrameBodyUSLT frameUSLT;

                    while (frameIterator.hasNext()) {
                        frame = (AbstractID3v2Frame) frameIterator.next();
                        body.addLyric((FrameBodyUSLT) frame.getBody());
                    }

                    frameIterator = id3tag.getFrameOfType("SYLT");

                    FrameBodySYLT frameSYLT;

                    while (frameIterator.hasNext()) {
                        frame = (AbstractID3v2Frame) frameIterator.next();
                        body.addLyric((FrameBodySYLT) frame.getBody());
                    }

                    fieldMap.put(newField.getIdentifier(),
                                 newField);
                }

                if (id3tag.hasFrameOfType("COMM")) {
                    Iterator           iterator = id3tag.getFrameOfType("COMM");
                    AbstractID3v2Frame frame;
                    text = "";

                    while (iterator.hasNext()) {
                        frame = (AbstractID3v2Frame) iterator.next();
                        text += (((FrameBodyCOMM) frame.getBody()).getText() +
                        " ");
                    }

                    newField = new Lyrics3v2Field(new FieldBodyINF(text));
                    fieldMap.put(newField.getIdentifier(),
                                 newField);
                }

                if (id3tag.hasFrame("TCOM")) {
                    textFrame = (AbstractFrameBodyTextInformation) id3tag.getFrame("TCOM")
                                .getBody();

                    if ((textFrame != null) &&
                            (((String) ((String) textFrame.getText())).length() > 0)) {
                        newField = new Lyrics3v2Field(new FieldBodyAUT(((String) textFrame.getText())));
                        fieldMap.put(newField.getIdentifier(),
                                     newField);
                    }
                }

                if (id3tag.hasFrame("TALB")) {
                    textFrame = (AbstractFrameBodyTextInformation) id3tag.getFrame("TALB")
                                .getBody();

                    if ((textFrame != null) &&
                            (((String) textFrame.getText()).length() > 0)) {
                        newField = new Lyrics3v2Field(new FieldBodyEAL(((String) textFrame.getText())));
                        fieldMap.put(newField.getIdentifier(),
                                     newField);
                    }
                }

                if (id3tag.hasFrame("TPE1")) {
                    textFrame = (AbstractFrameBodyTextInformation) id3tag.getFrame("TPE1")
                                .getBody();

                    if ((textFrame != null) &&
                            (((String) textFrame.getText()).length() > 0)) {
                        newField = new Lyrics3v2Field(new FieldBodyEAR(((String) textFrame.getText())));
                        fieldMap.put(newField.getIdentifier(),
                                     newField);
                    }
                }

                if (id3tag.hasFrame("TIT2")) {
                    textFrame = (AbstractFrameBodyTextInformation) id3tag.getFrame("TIT2")
                                .getBody();

                    if ((textFrame != null) &&
                            (((String) textFrame.getText()).length() > 0)) {
                        newField = new Lyrics3v2Field(new FieldBodyETT(((String) textFrame.getText())));
                        fieldMap.put(newField.getIdentifier(),
                                     newField);
                    }
                }
            }
        }
    }

    /**
     * Creates a new Lyrics3v2 object.
     *
     * @param file DOCUMENT ME!
     *
     * @throws TagNotFoundException DOCUMENT ME!
     * @throws IOException DOCUMENT ME!
     */
    public Lyrics3v2(RandomAccessFile file)
              throws TagNotFoundException, IOException {
        this.read(file);
    }

    /**
     * DOCUMENT ME!
     *
     * @param field DOCUMENT ME!
     */
    public void setField(Lyrics3v2Field field) {
        fieldMap.put(field.getIdentifier(),
                     field);
    }

    /**
     * Gets the value of the frame identified by identifier
     *
     * @param identifier The three letter code
     *
     * @return The value associated with the identifier
     */
    public Lyrics3v2Field getField(String identifier) {
        return (Lyrics3v2Field) fieldMap.get(identifier);
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public String getIdentifier() {
        return "Lyrics3v2.00";
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public int getSize() {
        int            size     = 0;
        Iterator       iterator = fieldMap.values()
                                  .iterator();
        Lyrics3v2Field field;

        while (iterator.hasNext()) {
            field = (Lyrics3v2Field) iterator.next();
            size += field.getSize();
        }

        // include LYRICSBEGIN, but not 6 char size or LYRICSEND
        return 11 + size;
    }

    /**
     * DOCUMENT ME!
     *
     * @param tag DOCUMENT ME!
     */
    public void append(AbstractMP3Tag tag) {
        Lyrics3v2 oldTag = this;
        Lyrics3v2 newTag = null;

        if (tag != null) {
            if (tag instanceof Lyrics3v2) {
                newTag = (Lyrics3v2) tag;
            } else {
                newTag = new Lyrics3v2(tag);
            }

            Iterator                   iterator = newTag.fieldMap.values()
                                                  .iterator();
            Lyrics3v2Field             field;
            AbstractLyrics3v2FieldBody body;

            while (iterator.hasNext()) {
                field = (Lyrics3v2Field) iterator.next();

                if (oldTag.hasField(field.getIdentifier()) == false) {
                    oldTag.setField(field);
                } else {
                    body = (AbstractLyrics3v2FieldBody) oldTag.getField(field.getIdentifier())
                           .getBody();

                    boolean save = TagOptionSingleton.getInstance()
                                   .getLyrics3SaveField(field.getIdentifier());

                    if ((body.getSize() == 0) && save) {
                        oldTag.setField(field);
                    }
                }
            }

            // reset tag options to save all current fields.
            iterator = oldTag.fieldMap.keySet()
                       .iterator();

            String id;

            while (iterator.hasNext()) {
                id = (String) iterator.next();
                TagOptionSingleton.getInstance()
                .setLyrics3SaveField(id, true);
            }
        }

        super.append(newTag);
    }

    /**
     * DOCUMENT ME!
     *
     * @param file DOCUMENT ME!
     *
     * @throws IOException DOCUMENT ME!
     * @throws TagException DOCUMENT ME!
     */
    public void append(RandomAccessFile file)
                throws IOException, TagException {
        Lyrics3v2 oldTag;

        try {
            oldTag = new Lyrics3v2(file);
            oldTag.append(this);
            oldTag.write(file);
        } catch (TagNotFoundException ex) {
            oldTag = null;
        }
    }

    /**
     * DOCUMENT ME!
     *
     * @param obj DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public boolean equals(Object obj) {
        if ((obj instanceof Lyrics3v2) == false) {
            return false;
        }

        Lyrics3v2 object = (Lyrics3v2) obj;

        if (this.fieldMap.equals(object.fieldMap) == false) {
            return false;
        }

        return super.equals(obj);
    }

    /**
     * DOCUMENT ME!
     *
     * @param identifier DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public boolean hasField(String identifier) {
        return fieldMap.containsKey(identifier);
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public Iterator iterator() {
        return fieldMap.values()
               .iterator();
    }

    /**
     * DOCUMENT ME!
     *
     * @param tag DOCUMENT ME!
     */
    public void overwrite(AbstractMP3Tag tag) {
        Lyrics3v2 oldTag = this;
        Lyrics3v2 newTag = null;

        if (tag != null) {
            if (tag instanceof Lyrics3v2) {
                newTag = (Lyrics3v2) tag;
            } else {
                newTag = new Lyrics3v2(tag);
            }

            Iterator       iterator = newTag.fieldMap.values()
                                      .iterator();
            Lyrics3v2Field field;

            while (iterator.hasNext()) {
                field = (Lyrics3v2Field) iterator.next();

                if (TagOptionSingleton.getInstance()
                        .getLyrics3SaveField(field.getIdentifier())) {
                    oldTag.setField(field);
                }
            }

            // reset tag options to save all current fields.
            iterator = oldTag.fieldMap.keySet()
                       .iterator();

            String id;

            while (iterator.hasNext()) {
                id = (String) iterator.next();
                TagOptionSingleton.getInstance()
                .setLyrics3SaveField(id, true);
            }
        }

        super.overwrite(newTag);
    }

    /**
     * DOCUMENT ME!
     *
     * @param file DOCUMENT ME!
     *
     * @throws IOException DOCUMENT ME!
     * @throws TagException DOCUMENT ME!
     */
    public void overwrite(RandomAccessFile file)
                   throws IOException, TagException {
        Lyrics3v2 oldTag;

        try {
            oldTag = new Lyrics3v2(file);
            oldTag.overwrite(this);
            oldTag.write(file);
        } catch (TagNotFoundException ex) {
            oldTag = null;
        }
    }

    /**
     * DOCUMENT ME!
     *
     * @param file DOCUMENT ME!
     *
     * @throws TagNotFoundException DOCUMENT ME!
     * @throws IOException DOCUMENT ME!
     */
    public void read(RandomAccessFile file)
              throws TagNotFoundException, IOException {
        long filePointer;
        int  lyricSize;

        if (seek(file)) {
            lyricSize = seekSize(file);
        } else {
            throw new TagNotFoundException("Lyrics3v2.00 Tag Not Found");
        }

        // reset file pointer to the beginning of the tag;
        seek(file);
        filePointer = file.getFilePointer();

        fieldMap = new HashMap();

        Lyrics3v2Field lyric;

        // read each of the fields
        while ((file.getFilePointer() - filePointer) < (lyricSize - 11)) {
            try {
                lyric = new Lyrics3v2Field(file);
                setField(lyric);
            } catch (InvalidTagException ex) {
                // if we get an invalid tag, we should still
                // keep going until we hit the end of the size
            }

            ;
        }
    }

    /**
     * DOCUMENT ME!
     *
     * @param identifier DOCUMENT ME!
     */
    public void removeField(String identifier) {
        fieldMap.remove(identifier);
    }

    /**
     * DOCUMENT ME!
     *
     * @param file DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     *
     * @throws IOException DOCUMENT ME!
     */
    public boolean seek(RandomAccessFile file)
                 throws IOException {
        byte[] buffer      = new byte[11];
        String lyricEnd    = "";
        String lyricStart  = "";
        long   filePointer = 0;
        long   lyricSize   = 0;

        // check right before the ID3 1.0 tag for the lyrics tag
        file.seek(file.length() - 128 - 9);
        file.read(buffer, 0, 9);
        lyricEnd = new String(buffer, 0, 9);

        if (lyricEnd.equals("LYRICS200")) {
            filePointer = file.getFilePointer();
        } else {
            // check the end of the file for a lyrics tag incase an ID3
            // tag wasn't placed after it.
            file.seek(file.length() - 9);
            file.read(buffer, 0, 9);
            lyricEnd = new String(buffer, 0, 9);

            if (lyricEnd.equals("LYRICS200")) {
                filePointer = file.getFilePointer();
            } else {
                return false;
            }
        }

        // read the 6 bytes for the length of the tag
        filePointer -= (9 + 6);
        file.seek(filePointer);
        file.read(buffer, 0, 6);

        String debug = new String(buffer, 0, 6);
        lyricSize = Integer.parseInt(new String(buffer, 0, 6));

        // read the lyrics begin tag if it exists.
        file.seek(filePointer - lyricSize);
        file.read(buffer, 0, 11);
        lyricStart = new String(buffer, 0, 11);

        return lyricStart.equals("LYRICSBEGIN") == true;
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public String toString() {
        Iterator       iterator = fieldMap.values()
                                  .iterator();
        Lyrics3v2Field field;
        String         str = getIdentifier() + " " + this.getSize() + "\n";

        while (iterator.hasNext()) {
            field = (Lyrics3v2Field) iterator.next();
            str += (field.toString() + "\n");
        }

        return str;
    }

    /**
     * DOCUMENT ME!
     *
     * @param identifier DOCUMENT ME!
     */
    public void updateField(String identifier) {
        Lyrics3v2Field lyrField;

        if (identifier.equals("IND")) {
            boolean lyricsPresent    = fieldMap.containsKey("LYR");
            boolean timeStampPresent = false;

            if (lyricsPresent) {
                lyrField = (Lyrics3v2Field) fieldMap.get("LYR");

                FieldBodyLYR lyrBody = (FieldBodyLYR) lyrField.getBody();
                timeStampPresent = lyrBody.hasTimeStamp();
            }

            lyrField = new Lyrics3v2Field(new FieldBodyIND(lyricsPresent,
                                                           timeStampPresent));
            setField(lyrField);
        }
    }

    /**
     * DOCUMENT ME!
     *
     * @param tag DOCUMENT ME!
     */
    public void write(AbstractMP3Tag tag) {
        Lyrics3v2 oldTag = this;
        Lyrics3v2 newTag = null;

        if (tag != null) {
            if (tag instanceof Lyrics3v2) {
                newTag = (Lyrics3v2) tag;
            } else {
                newTag = new Lyrics3v2(tag);
            }

            Iterator                   iterator = newTag.fieldMap.values()
                                                  .iterator();
            Lyrics3v2Field             field;
            AbstractLyrics3v2FieldBody body;
            oldTag.fieldMap.clear();

            while (iterator.hasNext()) {
                field = (Lyrics3v2Field) iterator.next();
                oldTag.setField(field);
            }
        }

        super.write(newTag);
    }

    /**
     * DOCUMENT ME!
     *
     * @param file DOCUMENT ME!
     *
     * @throws IOException DOCUMENT ME!
     */
    public void write(RandomAccessFile file)
               throws IOException {
        int offset = 0;
        ;

        long           size;
        long           filePointer;
        byte[]         buffer = new byte[6 + 9];

        String         str;
        Lyrics3v2Field field;
        Iterator       iterator;
        ID3v1          id3v1tag = new ID3v1();

        id3v1tag = id3v1tag.getID3tag(file);

        delete(file);
        file.seek(file.length());

        filePointer = file.getFilePointer();

        str = "LYRICSBEGIN";

        for (int i = 0; i < str.length(); i++) {
            buffer[i] = (byte) str.charAt(i);
        }

        file.write(buffer,
                   0,
                   str.length());

        // IND needs to go first. lets create/update it and write it first.
        updateField("IND");
        field = (Lyrics3v2Field) fieldMap.get("IND");
        field.write(file);

        iterator = fieldMap.values()
                   .iterator();

        while (iterator.hasNext()) {
            field = (Lyrics3v2Field) iterator.next();

            String  id   = field.getIdentifier();
            boolean save = TagOptionSingleton.getInstance()
                           .getLyrics3SaveField(id);

            if ((id.equals("IND") == false) && save) {
                field.write(file);
            }
        }

        size = file.getFilePointer() - filePointer;

        if (this.getSize() != size) {
            System.out.println("Lyrics3v2 size didn't match up while writing.");
            System.out.println("this.getsize()     = " + this.getSize());
            System.out.println("size (filePointer) = " + size);
        }

        str = Long.toString(size);

        for (int i = 0; i < (6 - str.length()); i++) {
            buffer[i] = (byte) '0';
        }

        offset += (6 - str.length());

        for (int i = 0; i < str.length(); i++) {
            buffer[i + offset] = (byte) str.charAt(i);
        }

        offset += str.length();

        str = "LYRICS200";

        for (int i = 0; i < str.length(); i++) {
            buffer[i + offset] = (byte) str.charAt(i);
        }

        offset += str.length();

        file.write(buffer, 0, offset);

        if (id3v1tag != null) {
            id3v1tag.write(file);
        }
    }

    /**
     * DOCUMENT ME!
     *
     * @param file DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     *
     * @throws IOException DOCUMENT ME!
     */
    private int seekSize(RandomAccessFile file)
                  throws IOException {
        byte[] buffer      = new byte[11];
        String lyricEnd    = "";
        String lyricStart  = "";
        long   filePointer = 0;
        long   lyricSize   = 0;

        // check right before the ID3 1.0 tag for the lyrics tag
        file.seek(file.length() - 128 - 9);
        file.read(buffer, 0, 9);
        lyricEnd = new String(buffer, 0, 9);

        if (lyricEnd.equals("LYRICS200")) {
            filePointer = file.getFilePointer();
        } else {
            // check the end of the file for a lyrics tag incase an ID3
            // tag wasn't placed after it.
            file.seek(file.length() - 9);
            file.read(buffer, 0, 9);
            lyricEnd = new String(buffer, 0, 9);

            if (lyricEnd.equals("LYRICS200")) {
                filePointer = file.getFilePointer();
            } else {
                return -1;
            }
        }

        // read the 6 bytes for the length of the tag
        filePointer -= (9 + 6);
        file.seek(filePointer);
        file.read(buffer, 0, 6);

        return Integer.parseInt(new String(buffer, 0, 6));
    }
}