/*
 * MP3 Tag library. It includes an implementation of the ID3 tags and Lyrics3
 * tags as they are defined at www.id3.org
 *
 * Copyright (C) Eric Farng 2003
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.farng.mp3;

import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.ArrayList;
import java.util.Iterator;

import org.farng.mp3.object.AbstractMP3Object;

/**
 * This class is contained in the <code>AbstractMP3Fragment</code> and
 * represents the actual data of tags. It contains default get/set methods for
 * all data objects. The data is represented as an ArrayList of
 * <code>MP3Object</code>. ID3v2 tags have frames. Lyrics3 tags have fields.
 * ID3v1 tags do not have fragments.
 *
 * @author Eric Farng
 * @version $Revision: 1.1 $
 */
public abstract class AbstractMP3FragmentBody extends AbstractMP3FileItem {
	/** list of <code>MP3Object</code> */
	protected ArrayList objectList = new ArrayList();

	/**
	 * Creates a new MP3FragmentBody object.
	 */
	public AbstractMP3FragmentBody() {
		setupObjectList();
	}

	/**
	 * This method calls <code>toString</code> for all it's objects and appends
	 * them without any newline characters.
	 *
	 * @return brief description string
	 */
	public String getBriefDescription() {
		String str = "";
		AbstractMP3Object object;
		Iterator iterator = objectList.listIterator();

		while (iterator.hasNext()) {
			object = (AbstractMP3Object) iterator.next();

			if ((object.toString() != null)
				&& (object.toString().length() > 0)) {
				str
					+= (object.getIdentifier()
						+ "=\""
						+ object.toString()
						+ "\"; ");
			}
		}

		return str;
	}

	/**
	 * This method calls <code>toString</code> for all it's objects and appends
	 * them. It contains new line characters and is more suited for display
	 * purposes
	 *
	 * @return formatted description string
	 */
	public String getDescription() {
		String str = "";
		AbstractMP3Object object;
		Iterator iterator = objectList.listIterator();

		while (iterator.hasNext()) {
			object = (AbstractMP3Object) iterator.next();
			str += (object.getIdentifier() + " = " + object.toString() + "\n");
		}

		return str.trim();
	}

	/**
	 * Set the all objects of identifier type to <code>obj</code> argument.
	 *
	 * @param identifier <code>MP3Object</code> identifier
	 * @param obj new object value
	 */
	public void setObject(String identifier, Object obj) {
		AbstractMP3Object object;
		Iterator iterator = objectList.listIterator();

		while (iterator.hasNext()) {
			object = (AbstractMP3Object) iterator.next();

			if (object.getIdentifier().equals(identifier)) {
				object.setValue(obj);
			}
		}
	}

	/**
	 * Returns the object of the <code>MP3Object</code> with the specified
	 * <code>identifier</code>
	 *
	 * @param identifier <code>MP3Object</code> identifier
	 *
	 * @return the object of the <code>MP3Object</code> with the specified
	 *         <code>identifier</code>
	 */
	public Object getObject(String identifier) {
		AbstractMP3Object object;
		Object obj = null;
		Iterator iterator = objectList.listIterator();

		while (iterator.hasNext()) {
			object = (AbstractMP3Object) iterator.next();

			if (object.getIdentifier().equals(identifier)) {
				obj = object.getValue();
			}
		}

		return obj;
	}

	/**
	 * Returns the estimated size in bytes of this object if it was to be
	 * written to file. This is not guaranteed to be accurate 100% of the
	 * time.
	 *
	 * @return estimated size in bytes of this object
	 *
	 * @todo-javadoc get this working 100% of the time
	 */
	public int getSize() {
		int size = 0;
		AbstractMP3Object object;
		Iterator iterator = objectList.listIterator();

		while (iterator.hasNext()) {
			object = (AbstractMP3Object) iterator.next();
			size += object.getSize();
		}

		return size;
	}

	/**
	 * Returns true if this instance and its entire <code>MP3Object</code>
	 * array list is a subset of the argument. This class is a subset if it is
	 * the same class as the argument.
	 *
	 * @param obj object to determine subset of
	 *
	 * @return true if this instance and its entire object array list is a
	 *         subset of the argument.
	 */
	public boolean isSubsetOf(Object obj) {
		if ((obj instanceof AbstractMP3FragmentBody) == false) {
			return false;
		}

		ArrayList superset = ((AbstractMP3FragmentBody) obj).objectList;

		for (int i = 0; i < objectList.size(); i++) {
			if (((AbstractMP3Object) objectList.get(i)).getValue() != null) {
				if (superset.contains(objectList.get(i)) == false) {
					return false;
				}
			}
		}

		return true;
	}

	/**
	 * Returns true if this object and its entire <code>MP3Object</code> array
	 * list equals the argument. This object is equal to the argument if they
	 * are the same class.
	 *
	 * @param obj object to determine equality of
	 *
	 * @return true if this object and its entire <code>MP3Object</code> array
	 *         list equals the argument.
	 */
	public boolean equals(Object obj) {
		if ((obj instanceof AbstractMP3FragmentBody) == false) {
			return false;
		}

		AbstractMP3FragmentBody object = (AbstractMP3FragmentBody) obj;

		if (this.objectList.equals(object.objectList) == false) {
			return false;
		}

		return super.equals(obj);
	}

	/**
	 * Returns an iterator of the <code>MP3Object</code> object list.
	 *
	 * @return iterator of the <code>MP3Object</code> object list.
	 */
	public Iterator iterator() {
		return objectList.iterator();
	}

	/**
	 * Read the data from the given file into this object. The file needs to
	 * have its file pointer in the correct location.
	 *
	 * @param file file to read from
	 *
	 * @throws IOException on any I/O error
	 * @throws InvalidTagException if there is any error in the data format.
	 */
	public void read(RandomAccessFile file)
		throws IOException, InvalidTagException {
		int size = readHeader(file);
		byte[] buffer = new byte[size];
		int offset = 0;

		file.read(buffer);

		AbstractMP3Object object;
		Iterator iterator = objectList.listIterator();

		while (iterator.hasNext()) {
			// Sanjay start bug fix
			if (offset > size - 1) {
				throw new InvalidTagException("Invalid size for Frame Body");
			}
			// Sanjay end bug fix
			object = (AbstractMP3Object) iterator.next();
			object.readByteArray(buffer, offset);
			offset += object.getSize();
		}
	}

	/**
	 * Calls <code>toString</code> for all <code>MP3Object</code> objects and
	 * creates a string with a new line character.
	 *
	 * @return description string
	 */
	public String toString() {
		String str = getIdentifier() + "\n";
		AbstractMP3Object object;
		Iterator iterator = objectList.listIterator();

		while (iterator.hasNext()) {
			object = (AbstractMP3Object) iterator.next();
			str += (object.getIdentifier() + " = " + object.toString() + "\n");
		}

		return str;
	}

	/**
	 * Write the contents of this object to the file at the position it is
	 * currently at.
	 *
	 * @param file destination file
	 *
	 * @throws IOException on any I/O error
	 */
	public void write(RandomAccessFile file) throws IOException {
		int debug = this.getSize();
		writeHeader(file, this.getSize());

		byte[] buffer;
		AbstractMP3Object object;
		Iterator iterator = objectList.listIterator();

		while (iterator.hasNext()) {
			object = (AbstractMP3Object) iterator.next();
			buffer = object.writeByteArray();
			file.write(buffer);
		}
	}

	/**
	 * Reads the header for the fragment body. The header contains things such
	 * as the length.
	 *
	 * @param file file to read the header from.
	 *
	 * @return size of body in bytes as stated in the header.
	 *
	 * @throws IOException on any I/O error
	 * @throws InvalidTagException if there is any error in the data format.
	 */
	abstract protected int readHeader(RandomAccessFile file)
		throws IOException, InvalidTagException;

	/**
	 * Create the order of <code>MP3Object</code> objects that this body
	 * expects. This method needs to be overwritten.
	 *
	 * @todo-javadoc Make this abstract. Can't do that yet because not all
	 *       implementations of the object have been finished.
	 */
	protected void setupObjectList() {
	}

	/**
	 * Write the body header to the file at the current file position.
	 *
	 * @param file file to write to
	 * @param size number of bytes the body contains.
	 *
	 * @throws IOException on any I/O error
	 */
	abstract protected void writeHeader(RandomAccessFile file, int size)
		throws IOException;
}