unit uMain;

{*******************************************************************************
  ShoutScraper

  Information:
    Shoutcast Scraper for use with Xbox Media Player
    See http://www.xboxmediaplayer.de for details

  Copyright:
    Released under the GPL (GNU Public License)
    Copyright 2003 kon at digital-bit.com (Kon)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  Author Contact:
    Email: kon at digital-bit dot com
    Yahoo & XBMP Forums: konfoo
    ICQ UIN: 429446

  Build Environment + Requirements:
    See Readme.txt
  Changelog:
    See Readme.txt

*******************************************************************************}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, SvcMgr, Dialogs,
  xmldom, XMLIntf, ExtCtrls, msxmldom, XMLDoc, IdAntiFreezeBase,
  IdAntiFreeze, IdBaseComponent, IdComponent, IdTCPConnection, IdTCPClient,
  IdHTTP, jsHTMLParser, strutils, activex, faststrings, faststringfuncs;

type
  TScraperThread = class(TThread)
    public
      procedure Execute; override;
    end;
  TShoutScraper = class(TService)
    vclParser: TjsHTMLParser;
    vclHTTP: TIdHTTP;
    IdAntiFreeze1: TIdAntiFreeze;
    xXML: TXMLDocument;
    procedure ServiceExecute(Sender: TService);
    procedure ServicePause(Sender: TService; var Paused: Boolean);
    procedure ServiceContinue(Sender: TService; var Continued: Boolean);
    procedure ServiceStop(Sender: TService; var Stopped: Boolean);
    procedure ServiceStart(Sender: TService; var Started: Boolean);
  private
    { Private declarations }
    procedure DoUpdate;
    procedure LoadSettings;
    procedure ParseXML;
    procedure UpdateShoutcastTree;
    procedure GenerateFile;
    procedure PurgePath;
    function GetFile(input:string):string;
    function GetTitle(input:string):string;
  public
    function GetServiceController: TServiceController; override;
    { Public declarations }
  end;

var
  ShoutScraper: TShoutScraper;
  ScraperThread: TScraperThread;

  iTimeout:integer;
  xNode:IXMLNode;
  sProxy:string;
  sRootFolder:string;
  iTrunc:short;
  iIterationInterval:integer;
  iMinBitrate:integer;
  iRetryInterval:integer;
  txtFile:TextFile;

  scFile:string;
  scTitle:string;

  inputRoot:string;
  inputTitle:string;
  inputResource:string;
  inputURL:string;
  inputCategory:string;

  sTempTitle:string;

  sURLText:string;
  sPLSText:string;
  sBitrate:string;

  tInterval:integer;

implementation

{$R *.DFM}

{*******************************************************************************
  Service Specific
*******************************************************************************}
procedure ServiceController(CtrlCode: DWord); stdcall;
begin
  ShoutScraper.Controller(CtrlCode);
end;

function TShoutScraper.GetServiceController: TServiceController;
begin
  Result := ServiceController;
end;

procedure TShoutScraper.ServicePause(Sender: TService;
  var Paused: Boolean);
begin
  ScraperThread.Suspend;
  Paused:=true;
end;

procedure TShoutScraper.ServiceContinue(Sender: TService;
  var Continued: Boolean);
begin
  ScraperThread.Resume;
  Continued:=true;
end;

procedure TShoutScraper.ServiceStop(Sender: TService;
  var Stopped: Boolean);
begin
  ScraperThread.Terminate;
  Stopped:=true;
end;

{*******************************************************************************
  Main Scraping Thread
*******************************************************************************}
procedure TScraperThread.Execute;
begin
  CoInitialize(nil);
  while not terminated do
    begin
      try
        try
          ShoutScraper.vclHTTP.Connect(iTimeout);
        except
          sleep(iRetryInterval);
          try
            ShoutScraper.vclHTTP.Connect(iTimeout);
          except
            sleep(iRetryInterval);
            ShoutScraper.vclHTTP.Connect(iTimeout);
          end;
        end;
        ShoutScraper.vclHTTP.Disconnect;
        ShoutScraper.vclHTTP.DisconnectSocket;
        ShoutScraper.doUpdate;
        sleep(tInterval);
      except
        ShoutScraper.LogMessage('Error Connecting to ShoutCast server http://'
                    +ShoutScraper.vclHTTP.Host+' on port '+
                    inttostr(ShoutScraper.vclHTTP.Port)+'. Timed out after '+
                    inttostr(round(iTimeout/1000)*3*round(iRetryInterval/1000))+
                    ' seconds',EVENTLOG_ERROR_TYPE);
        sleep(tInterval);
      end;
    end;
  CoUnInitialize;
end;

{*******************************************************************************
  Do Update on Period
*******************************************************************************}
procedure TShoutScraper.DoUpdate;
begin
  try
    ParseXML;
  except
    //
  end;
end;

{*******************************************************************************
  Update the Local Shoutcast Tree
*******************************************************************************}
procedure TShoutScraper.UpdateShoutcastTree;
var iCount:integer;
    iNumber:integer;
    iBStart:integer;
    iBEnd:integer;
begin
  PurgePath;
  iNumber:=0;
  sURLText:=vclhttp.Get(inputURL);
  vclParser.HTML.Text:=sURLText;
  vclParser.Execute;
  for iCount:=0 to vclParser.Parsed.Hyperlinks.Count-1 do
    begin
      if midstr(vclParser.Parsed.Hyperlinks[iCount],1,28)=
        '/sbin/shoutcast-playlist.pls' then
        begin
          inc(iNumber);
          ibStart:=smartpos(vclParser.Parsed.Hyperlinks[iCount],sURLText);
          ibStart:=smartpos('target="_scurl"',sURLText,false,ibStart+1,true);
          ibStart:=smartpos('</font></td>',sURLText,false,ibStart+1,true);
          ibStart:=smartpos('</font></td>',sURLText,false,ibStart+1,true);
          ibStart:=smartpos('color="#FFFFFF"',sURLText,false,ibStart+1,true);
          ibEnd:=smartpos('<',surltext,false,ibStart,true);
          sBitrate:=midstr(surltext,ibstart+16,ibend-ibstart-16);
          if sbitrate='' then
            sbitrate:='???';
          sPLSText:=vclHTTP.Get('http://'+vclHTTP.Host+'/'
                    +vclParser.Parsed.Hyperlinks[iCount]);
          scFile:=GetFile(splstext);
          scTitle:=format('%.2d',[iNumber]);
          scTitle:=scTitle+' '+GetTitle(splstext);
          try
          if strtoint(sbitrate) >= iMinBitrate then
            GenerateFile;
          except
            GenerateFile;
          end;
        end;
    end;
  vclParser.HTML.Clear;
end;

{*******************************************************************************
  Purge before repopulating
*******************************************************************************}
Procedure TShoutScraper.PurgePath;
var
  APath: string;
  MySearch: TSearchRec;
begin
  APath:=inputroot+'\'+inputcategory;
  FindFirst(APath+'\*.sc', faAnyFile, MySearch);
  try
  DeleteFile(APath+'\'+MySearch.Name);
  except
    //
  end;
  while FindNext(MySearch)=0 do
  begin
    try
      DeleteFile(APath+'\'+MySearch.Name);
    except
      //
    end;
  end;
  FindClose(MySearch);
end;

{*******************************************************************************
  Generate the XBMP shoutcast file
*******************************************************************************}
procedure TShoutScraper.GenerateFile;
begin
  if not DirectoryExists(inputroot) then
    CreateDir(inputroot);
  if not DirectoryExists(inputroot+'\'+inputcategory) then
    CreateDir(inputroot+'\'+inputcategory);
  AssignFile(txtFile,inputroot+'\'+inputcategory+'\'+sctitle+' ('+
              sBitrate+'k).sc');
  Rewrite(txtFile);
  Writeln(txtFile,scfile);
  CloseFile(txtFile);
end;

{*******************************************************************************
  Strip the Shoutcast File from the PLS
*******************************************************************************}
function TShoutScraper.GetFile(input:string):string;
var iStart:integer;
    iEnd:integer;
begin
  try
    for iStart:=1 to length(input)-1 do
      if midstr(input,iStart,6)='File1=' then break;
    for iEnd:=iStart to length(input)-1 do
      if midstr(input,iEnd,1)=#10 then break;
    Result:='shoutcast'+midstr(input,iStart+10,iEnd-iStart-10);
  except
    result:='';
  end;
end;

{*******************************************************************************
  Strip the Shoutcast Title from the PLS
*******************************************************************************}
function TShoutScraper.GetTitle(input:string):string;
var iStart:integer;
    iEnd:integer;
begin
  try
    for iStart:=1 to length(input)-1 do
      if midstr(input,iStart,7)='Title1=' then break;
    for iStart:=iStart to length(input)-1 do
      if midstr(input,iStart,2)=') ' then break;
    for iEnd:=iStart to length(input)-1 do
      if midstr(input,iEnd,1)=#10 then break;
    sTempTitle:=midstr(input,iStart+2,iEnd-iStart-2);
    for iEnd:=1 to length(sTempTitle) do
      if (midstr(sTempTitle,iEnd,2)=' -') or
         (midstr(sTempTitle,iEnd,3)=' : ') or
         (midstr(sTempTitle,iEnd,2)=', ') then break;
    sTempTitle:=midstr(sTempTitle,1,iEnd);
    sTempTitle:=AnsiReplaceStr(sTempTitle,'.com',' dotcom');
    sTempTitle:=AnsiReplaceStr(sTempTitle,'.COM',' dotcom');
    sTempTitle:=AnsiReplaceStr(sTempTitle,'.net',' dotnet');
    sTempTitle:=AnsiReplaceStr(sTempTitle,'.NET',' dotnet');
    sTempTitle:=AnsiReplaceStr(sTempTitle,'.org',' dotorg');
    sTempTitle:=AnsiReplaceStr(sTempTitle,'.ORG',' dotorg');
    sTempTitle:=AnsiReplaceStr(sTempTitle,'.',' ');
    sTempTitle:=AnsiReplaceStr(sTempTitle,' @ ','at');
    sTempTitle:=AnsiReplaceStr(sTempTitle,'@',' at ');
    sTempTitle:=AnsiReplaceStr(sTempTitle,':',' ');
    sTempTitle:=AnsiReplaceStr(sTempTitle,'`','');
    sTempTitle:=AnsiReplaceStr(sTempTitle,'\','');
    sTempTitle:=AnsiReplaceStr(sTempTitle,'/','');
    sTempTitle:=AnsiReplaceStr(sTempTitle,'?','');
    sTempTitle:=AnsiReplaceStr(sTempTitle,'<','');
    sTempTitle:=AnsiReplaceStr(sTempTitle,'>','');
    sTempTitle:=AnsiReplaceStr(sTempTitle,'|','');
    sTempTitle:=AnsiReplaceStr(sTempTitle,'-',' ');
    sTempTitle:=AnsiReplaceStr(sTempTitle,'=',' ');
    sTempTitle:=AnsiReplaceStr(sTempTitle,'~',' ');
    sTempTitle:=AnsiReplaceStr(sTempTitle,'"','');
    sTempTitle:=AnsiReplaceStr(sTempTitle,'*','');
    sTempTitle:=AnsiReplaceStr(sTempTitle,',','');
    sTempTitle:=AnsiReplaceStr(sTempTitle,'[','');
    sTempTitle:=AnsiReplaceStr(sTempTitle,']','');
    sTempTitle:=AnsiReplaceStr(sTempTitle,')','');
    sTempTitle:=AnsiReplaceStr(sTempTitle,'(','');
    sTempTitle:=AnsiReplaceStr(sTempTitle,'http','');
    sTempTitle:=AnsiReplaceStr(sTempTitle,'HTTP','');
    sTempTitle:=AnsiReplaceStr(sTempTitle,' www','');
    sTempTitle:=AnsiReplaceStr(sTempTitle,'www ','');
    sTempTitle:=AnsiReplaceStr(sTempTitle,' WWW','');
    sTempTitle:=AnsiReplaceStr(sTempTitle,'WWW ','');
    sTempTitle:=AnsiReplaceStr(sTempTitle,'#','');
    sTempTitle:=AnsiReplaceStr(sTempTitle,'^','');
    sTempTitle:=AnsiReplaceStr(sTempTitle,'_',' ');
    sTempTitle:=AnsiReplaceStr(sTempTitle,'}','');
    sTempTitle:=AnsiReplaceStr(sTempTitle,'{','');
    sTempTitle:=TrimLeft(sTempTitle);
    sTempTitle:=TrimRight(sTempTitle);
    if length(stemptitle) > iTrunc then
      sTempTitle:=MidStr(sTempTitle,1,iTrunc);
    iStart:=0;
    iEnd:=length(sTempTitle)-1;
    while iStart <= iEnd do
      begin
        if midstr(sTempTitle,iStart,2)='  ' then
          begin
            sTempTitle:=StuffString(sTempTitle,iStart+1,1,'');
            dec(iEnd);
          end
        else
          inc(iStart);
      end;
    iEnd:=SmartPos(' ',sTempTitle,false,length(sTempTitle)-1,false);
    if iEnd>1 then
      sTempTitle:=MidStr(sTempTitle,1,iEnd-1);
    if sTempTitle<>'' then
      Result:=sTempTitle
    else
      Result:='ShoutScrape''d Station';
  except
    Result:='ShoutScrape''d Station';
  end;
end;

{*******************************************************************************
  Iterate the XML file's shoutcast serverlist
*******************************************************************************}
procedure TShoutScraper.ParseXML;
var iCountsc:integer;
begin
  xXML.loadfromfile(ExtractFilePath(paramstr(0))+'shoutscraper.xml');
  xNode:=xXML.DocumentElement.ChildNodes['shoutcast'];
  for iCountsc:=0 to xNode.ChildNodes.Count-1 do
    begin
      inputRoot:=sRootFolder;
      inputCategory:=xNode.ChildNodes[iCountsc].ChildNodes['name'].Text;
      inputURL:='http://'+vclHTTP.Host+'/directory/?sgenre='+
                xNode.ChildNodes[iCountsc].ChildNodes['url'].Text+'&'+
                xNode.ChildNodes[iCountsc].ChildNodes['tags'].Text;
      inputURL:=StringReplace(inputURL,'|','&',[rfReplaceAll]);
      inputResource:=xNode.ChildNodes[iCountsc].ChildNodes['resource'].Text;
      UpdateShoutcastTree;
      CopyFile(pchar(ExtractFilePath(paramstr(0))+'Images\'+
               inputresource),
               pchar(inputroot+'\'+inputcategory+'.tbn'),false);
      sleep(iIterationInterval);
    end;
end;

{*******************************************************************************
  Load Settings
*******************************************************************************}
procedure TShoutScraper.LoadSettings;
begin
  xXML.loadfromfile(ExtractFilePath(paramstr(0))+'shoutscraper.xml');
  xNode:=xXML.DocumentElement.ChildNodes['main'];
  sProxy:=xNode.ChildNodes['useproxy'].text;
  if sProxy='true' then
    begin
      vclHTTP.ProxyParams.ProxyServer:=xNode.ChildNodes['proxyhost'].text;
      vclHTTP.ProxyParams.ProxyPort:=
                                  strtoint(xNode.ChildNodes['proxyport'].text);
    end;
  vclHTTP.HandleRedirects:=false;
  vclHTTP.RedirectMaximum:=1;
  vclhttp.Host:=xNode.ChildNodes['basehost'].text;
  vclHTTP.Port:=strtoint(xNode.ChildNodes['baseport'].text);
  vclHTTP.Request.UserAgent:=xNode.ChildNodes['useragent'].text;
  iTimeout:=strtoint(xNode.ChildNodes['timeout'].text)*1000;
  iTrunc:=strtoint(xNode.ChildNodes['trunc'].text);
  iIterationInterval:=strtoint(xNode.ChildNodes['iterationinterval'].text)*1000;
  sRootFolder:=xNode.ChildNodes['rootfolder'].text;
  iRetryInterval:=strtoint(xNode.ChildNodes['retryinterval'].text)*1000;
  iMinBitrate:=strtoint(xNode.ChildNodes['minbitrate'].text);
  tInterval:=strtoint(xNode.ChildNodes['updateinterval'].text)
                      *60*60*1000;
end;

{*******************************************************************************
  Service Thread
*******************************************************************************}
procedure TShoutScraper.ServiceExecute(Sender: TService);
begin
  while not Terminated do begin
    ServiceThread.ProcessRequests(True);
  end;
  CoUnInitialize;
end;

{*******************************************************************************
  Main. Duh.
*******************************************************************************}
procedure TShoutScraper.ServiceStart(Sender: TService;
  var Started: Boolean);
begin
  try
    CoInitialize(nil);
    LoadSettings;
    if not DirectoryExists(sRootFolder) then
      begin
        LogMessage('Error Starting Service. Root folder '+srootfolder
                   +' does not exist.',EVENTLOG_ERROR_TYPE);
        Started:=false;
        exit;
      end;
    if not DirectoryExists(sRootFolder+'\ShoutScraper') then
      MkDir(sRootFolder+'\ShoutScraper');
    CopyFile(pchar(ExtractFilePath(paramstr(0))+'ShoutScraper.jpg'),
               pchar(sRootFolder+'\ShoutScraper.tbn'),false);
    sRootFolder:=sRootFolder+'\ShoutScraper';
    ScraperThread:=TScraperThread.Create(false);
    Started:=true;
  except
    LogMessage('Error Starting Service. Check XML Files & Settings.',
                EVENTLOG_ERROR_TYPE);
    Started:=false;
    CoUnInitialize;
  end;
end;

end.
