// PIXIT XIP Archive Manipulator
// Copyright (c) Voltaic 2003. All rights reserved.
//
// Loosely based on what dexip.exe pumped out - by Hartec
//
// pixit.cpp : Defines the entry point for the console application.
//

// -create=mah.csv mah.xip

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <direct.h>
#include <ctype.h> 
#include <list>

#define PIXIT_MAJOR_VER	0
#define PIXIT_MINOR_VER	5

typedef struct _tagXIPHDR
{
	char cbMagic[4];
	unsigned long nDataOffset;
	unsigned short nFiles;
	unsigned short nNames;
	unsigned long nDataSize;
} XIPHDR, *LPXIPHDR;

typedef struct _tagFILEDATA
{
	unsigned long nOffset;
	unsigned long nSize;
	unsigned long nType;
	unsigned long nTimestamp;
} FILEDATA, *LPFILEDATA;

typedef struct _tagFILENAME
{
	unsigned short nDataIndex;
	unsigned short nNameOffset;
} FILENAME, *LPFILENAME;

typedef struct _tagENTRY
{
	char szSrcFilename[_MAX_PATH + 1];
	char szDstFilename[_MAX_PATH + 1];
	FILEDATA fd;
	FILENAME fn;
} ENTRY, *LPENTRY;

void Usage(const char* pszError)
{
	if(pszError != NULL)
	{
		printf("  ***  Error: '%s'  ***  \n", pszError);
		printf("\n");
		printf("\n");
	}
	printf("Usage: \n pixit.exe <-create[=default.csv]|-display[=csv]|-extract[=path]> [default.xip]\n");
	printf("\n");
	printf("\t-create   Create a new XIP archive file (def. default.csv).\n");
	printf("\t-display  Display information about an XIP archive file.\n");
	printf("\t-extract  Extract files from an XIP archive file to specified folder.\n");
	printf("\n");
	printf("Note: The list file (default.txt) must contain only one file per line.\n");
	printf("      All lines that begin with # are considered comments and skipped.\n");
	exit(-1);
}

bool GetCmdLineArg(const char* pszName, int argc, char* argv[])
{
	char* pszArg = NULL, szTemp[256] = "";
	int n, m, l = strlen(pszName);
	for(n = 0; n < argc; n++)
	{
		pszArg = argv[n];
		for(m = 0; ((szTemp[m] = pszArg[m]) != '=') && (szTemp[m] != '\0'); m++);
		szTemp[m] = '\0';
		if(((*pszArg == '-') || (*pszArg == '/')) && (stricmp(szTemp + 1, pszName) == 0))
			return true;
	}
	return false;
}

char* GetCmdLineArgValue(const char* pszName, int argc, char* argv[])
{
	char* pszArg = NULL;
	int n, m, l = strlen(pszName);
	for(n = 0; n < argc; n++)
	{
		pszArg = argv[n];
		if((*pszArg == '-') || (*pszArg == '/'))
		{
			for(m = 0; m < l; m++)
			{
				if(tolower(pszArg[1 + m]) != tolower(pszName[m]))
					break;
			}

			if((m == l) && (pszArg[l + 1] == '='))
				return pszArg + l + 2;
		}
	}
	return NULL;
}

void PrintXipHeader(LPXIPHDR pXipHeader)
{
	printf("Header information:\n\n");
	printf("File ID....:  %.4s\n", pXipHeader->cbMagic);
	printf("Data Offset:  0x%.8X\n", pXipHeader->nDataOffset, pXipHeader->nDataOffset);
	printf("Files......:  %lu\n", pXipHeader->nFiles);
	printf("Data Size..:  0x%.8X (%lu bytes)\n", pXipHeader->nDataSize, pXipHeader->nDataSize);
}

const char* GetFilenameAt(const char* pszFilenameBlock, int nIndex)
{
	const char* pszFilename = pszFilenameBlock;
	for(int n = 0; n < nIndex; n++)
	{
		if(n == nIndex)
			break;
		pszFilename += strlen(pszFilename) + 1;
		if(pszFilename == NULL)
			return NULL;
	}
	return pszFilename;
}

int CreateArchive(const char* pszListFile, const char* pszArchiveFile)
{
	FILE* fpIn = fopen(pszListFile, "rt");
	if(fpIn == NULL)
	{
		printf("Error opening file '%s'. (errno: %d)\n", pszListFile, errno);
		return -2;
	}

	printf("Creating '%s' form '%s'\n\n", pszArchiveFile, pszListFile);

	std::list<ENTRY> lstEntries;

	while(feof(fpIn) == 0)
	{
		char szLine[1024];
		if(fgets(szLine, 1024, fpIn) != NULL)
		{
			char* pszLine = szLine;
			while(isspace(*pszLine))
				pszLine++;

			if(*pszLine == NULL || *pszLine == '#')
				continue;

			ENTRY en;
			memset(&en, 0, sizeof(en));

			if(sscanf(szLine, "%[^,],0x%X,0x%X,0x%X,0x%X,0x%X,0x%X\n", 
				en.szSrcFilename, 
				&en.fd.nSize, &en.fd.nOffset, &en.fd.nType, &en.fd.nTimestamp, 
				&en.fn.nDataIndex, &en.fn.nNameOffset) != 7)
			{
				printf("Skipping malformed line...\n");
				continue;
			}

			if(en.fd.nType != 4)
			{
				FILE* fpIn = fopen(en.szSrcFilename, "rb");
				if(fpIn == NULL)
				{
					printf("Error opening file '%s'. (errno: %d)\n", en.szSrcFilename, errno);
					return -2;
				}
				fseek(fpIn, 0, SEEK_END);
				en.fd.nSize = ftell(fpIn);
				fclose(fpIn);
			}

			char* pszPart = strrchr(en.szSrcFilename, '\\');
			if(pszPart != NULL)
				strcpy(en.szDstFilename, ++pszPart);
			else
				strcpy(en.szDstFilename, en.szSrcFilename);

			lstEntries.push_back(en);
		}
	}

	fclose(fpIn);

	std::list<ENTRY>::iterator it;

	XIPHDR xh;
	memset(&xh, 0, sizeof(xh));
	xh.cbMagic[0] = 'X';
	xh.cbMagic[1] = 'I';
	xh.cbMagic[2] = 'P';
	xh.cbMagic[3] = '0';
	xh.nDataOffset = sizeof(xh);
	for(it = lstEntries.begin(); it != lstEntries.end(); it++)
		xh.nDataOffset += sizeof(it->fd) + sizeof(it->fn) + strlen(it->szDstFilename) + 1;
	xh.nFiles = lstEntries.size();
	xh.nNames = lstEntries.size();
	xh.nDataSize = 0L; 

	for(it = lstEntries.begin(); it != lstEntries.end(); it++)
	{
		LPFILEDATA pFileData = &(it->fd);

		if(pFileData->nType == 4)
			continue;

		FILE* fpIn = fopen(it->szSrcFilename, "rb");
		if(fpIn == NULL)
		{
			printf("Error opening file '%s'. (errno: %d)\n", it->szSrcFilename, errno);
			return -2;
		}

		fseek(fpIn, 0, SEEK_END);
		pFileData->nOffset = xh.nDataSize;
		pFileData->nSize = ftell(fpIn);
		xh.nDataSize += pFileData->nSize;

		fclose(fpIn);
	}

	FILE* fpOut = fopen(pszArchiveFile, "wb");
	if(fpOut == NULL)
	{
		printf("Error creating file '%s'. (errno: %d)\n", pszArchiveFile, errno);
		return -2;
	}

	fwrite(&xh, 1, sizeof(xh), fpOut);
	for(it = lstEntries.begin(); it != lstEntries.end(); it++)
		fwrite(&(it->fd), 1, sizeof(FILEDATA), fpOut);
	for(it = lstEntries.begin(); it != lstEntries.end(); it++)
		fwrite(&(it->fn), 1, sizeof(FILENAME), fpOut);
	for(it = lstEntries.begin(); it != lstEntries.end(); it++)
		fwrite(&(it->szDstFilename), 1, strlen(it->szDstFilename) + 1, fpOut);

	unsigned long nTotalBytesWritten = 0L;
	for(it = lstEntries.begin(); it != lstEntries.end(); it++)
	{
		if(it->fd.nType == 4)
			continue;

		FILE* fpIn = fopen(it->szSrcFilename, "rb");
		if(fpIn == NULL)
		{
			printf("Error opening file '%s'. (errno: %d)\n", it->szSrcFilename, errno);
			fclose(fpOut);
			return -2;
		}


		fseek(fpIn, 0, SEEK_END);
		long nSize = ftell(fpIn);
		fseek(fpIn, 0, SEEK_SET);

		unsigned char cbBuffer[4096];

		long nBlocks = nSize / sizeof(cbBuffer);
		for(int m = 0; m < nBlocks; m++)
		{
			int nRead = fread(cbBuffer, 1, sizeof(cbBuffer), fpIn);
			if(nRead != sizeof(cbBuffer))
			{
				fclose(fpOut);
				fclose(fpIn);
				printf("Error reading file. (errno: %d)\n", errno);
				return -2;
			}

			int nWritten = fwrite(cbBuffer, 1, sizeof(cbBuffer), fpOut);
			if(nWritten != sizeof(cbBuffer))
			{
				fclose(fpOut);
				fclose(fpIn);
				printf("Error writing file. (errno: %d)\n", errno);
				return -2;
			}

			nTotalBytesWritten += nWritten;
		}
		long nRemainder = nSize % sizeof(cbBuffer);
		if(nRemainder > 0)
		{
			int nRead = fread(cbBuffer, 1, nRemainder, fpIn);
			if(nRead != nRemainder)
			{
				fclose(fpOut);
				fclose(fpIn);
				printf("Error reading file. (errno: %d)\n", errno);
				return -2;
			}

			int nWritten = fwrite(cbBuffer, 1, nRemainder, fpOut);
			if(nWritten != nRemainder)
			{
				fclose(fpOut);
				fclose(fpIn);
				printf("Error writing file. (errno: %d)\n", errno);
				return -2;
			}

			nTotalBytesWritten += nWritten;
		}
	}
	if(nTotalBytesWritten != xh.nDataSize)
		printf("Data lenght written to archive file is inconsistent with what was expected (0x%.8X vs. 0x%.8X).\n", nTotalBytesWritten, xh.nDataSize);

	fclose(fpOut);

	return 0;
}

int DisplayArchiveInfo(bool bCSV, const char* pszArchiveFile)
{
	FILE* fpIn = fopen(pszArchiveFile, "rb");
	if(fpIn == NULL)
	{
		printf("Error opening file '%s'. (errno: %d)\n", pszArchiveFile, errno);
		return -2;
	}

	printf("Displaying content of '%s'.\n\n\n", pszArchiveFile);

	XIPHDR xh;
	memset(&xh, 0, sizeof(xh));

	int nRead = fread(&xh, 1, sizeof(xh), fpIn);
	if(nRead != sizeof(xh))
	{
		fclose(fpIn);
		printf("Error reading file. (errno: %d)\n", errno);
		return -2;
	}

	if(strncmp(xh.cbMagic, "XIP0", 4) != 0)
	{
		fclose(fpIn);
		printf("Invalid file format.\n");
		return -2;
	}

	PrintXipHeader(&xh);

	printf("\n\n");

	LPFILEDATA pFileData = new FILEDATA[xh.nFiles];
	if(pFileData == NULL)
	{
		fclose(fpIn);
		printf("Out of memory.\n");
		return -2;
	}

	nRead = fread(pFileData, 1, xh.nFiles * sizeof(FILEDATA), fpIn);
	if(nRead != (int)(xh.nFiles * sizeof(FILEDATA)))
	{
		delete [] pFileData;
		fclose(fpIn);
		printf("Error reading file. (errno: %d)\n", errno);
		return -2;
	}

	LPFILENAME pFileName = new FILENAME[xh.nNames];
	if(pFileName == NULL)
	{
		fclose(fpIn);
		printf("Out of memory.\n");
		return -2;
	}

	nRead = fread(pFileName, 1, xh.nNames * sizeof(FILENAME), fpIn);
	if(nRead != (int)(xh.nFiles * sizeof(FILENAME)))
	{
		delete [] pFileName;
		delete [] pFileData;
		fclose(fpIn);
		printf("Error reading file. (errno: %d)\n", errno);
		return -2;
	}

	int nFilenameBlockOffset = xh.nNames * sizeof(FILENAME) + xh.nFiles * sizeof(FILEDATA) + sizeof(xh);
	int nFilenameBlockSize = xh.nDataOffset - nFilenameBlockOffset;

	char* pszFilenameBlock = new char[nFilenameBlockSize + 1];
	if(pszFilenameBlock == NULL)
	{
		delete [] pFileName;
		delete [] pFileData;
		fclose(fpIn);
		printf("Out of memory\n");
		return -2;
	}

	nRead = fread(pszFilenameBlock, 1, nFilenameBlockSize, fpIn);
	if(nRead != nFilenameBlockSize)
	{
		delete [] pszFilenameBlock;
		delete [] pFileName;
		delete [] pFileData;
		fclose(fpIn);
		printf("Error reading file. (errno: %d)\n", errno);
		return -2;
	}

	pszFilenameBlock[nFilenameBlockSize] = NULL;

	printf("File entries:\n\n");

	if(bCSV == true)
	{
		printf("# *** Anything before this line is garbage when using this CSV file *** \n");
		printf("#File,Size,Offset,Type,DW,Ord,W\n");
	}
	else
	{
		printf("File                       Size   D.Offset       Type  Timestamp  Index Offset\n");
		printf("-------------------- ---------- ---------- ---------- ---------- ------ ------\n");
	}

	for(int n = 0; n < xh.nFiles; n++)
	{
		LPFILEDATA pData = &pFileData[n];
		LPFILENAME pName = &pFileName[n];

		if(bCSV == true)
		{
			printf("%s,0x%.8X,0x%.8X,0x%.8X,0x%.8X,0x%.4X,0x%.4X\n", 
				GetFilenameAt(pszFilenameBlock, n), 
				pData->nSize, 
				pData->nOffset, 
				pData->nType, 
				pData->nTimestamp, 
				pName->nDataIndex, 
				pName->nNameOffset);
		}
		else
		{
			printf("%-20s 0x%.8X 0x%.8X 0x%.8X 0x%.8X 0x%.4X 0x%.4X\n", 
				GetFilenameAt(pszFilenameBlock, n), 
				pData->nSize, 
				pData->nOffset, 
				pData->nType, 
				pData->nTimestamp, 
				pName->nDataIndex, 
				pName->nNameOffset);
		}
	}

	delete [] pFileData;
	delete [] pFileName;
	fclose(fpIn);

	return 0;
}

int ExtractArchive(const char* pszExtractFolder, const char* pszArchiveFile)
{
	char szFolder[_MAX_PATH + 1] = "";

	if(pszExtractFolder == NULL)
	{
		strcpy(szFolder, pszArchiveFile);
		strcat(szFolder, ".d");
	}
	else
	{
		strcpy(szFolder, pszExtractFolder);
	}

	int len = strlen(szFolder);
	if(szFolder[len - 1] == '\\' || szFolder[len - 1] == '/')
		szFolder[len - 1] = '\0';

	mkdir(szFolder);

	FILE* fpIn = fopen(pszArchiveFile, "rb");
	if(fpIn == NULL)
	{
		printf("Error opening file '%s'. (errno: %d)\n", pszArchiveFile, errno);
		return -2;
	}

	printf("Extracting files from '%s' to '%s'.\n\n\n", pszArchiveFile, szFolder);

	XIPHDR xh;
	memset(&xh, 0, sizeof(xh));

	int nRead = fread(&xh, 1, sizeof(xh), fpIn);
	if(nRead != sizeof(xh))
	{
		fclose(fpIn);
		printf("Error reading file. (errno: %d)\n", errno);
		return -2;
	}

	if(strncmp(xh.cbMagic, "XIP0", 4) != 0)
	{
		fclose(fpIn);
		printf("Invalid file format.\n");
		return -2;
	}

	PrintXipHeader(&xh);

	printf("\n\n");

	LPFILEDATA pFileData = new FILEDATA[xh.nFiles];
	if(pFileData == NULL)
	{
		fclose(fpIn);
		printf("Out of memory.\n");
		return -2;
	}

	nRead = fread(pFileData, 1, xh.nFiles * sizeof(FILEDATA), fpIn);
	if(nRead != (int)(xh.nFiles * sizeof(FILEDATA)))
	{
		delete [] pFileData;
		fclose(fpIn);
		printf("Error reading file. (errno: %d)\n", errno);
		return -2;
	}

	LPFILENAME pFileName = new FILENAME[xh.nFiles];
	if(pFileName == NULL)
	{
		fclose(fpIn);
		printf("Out of memory.\n");
		return -2;
	}

	nRead = fread(pFileName, 1, xh.nFiles * sizeof(FILENAME), fpIn);
	if(nRead != (int)(xh.nFiles * sizeof(FILENAME)))
	{
		delete [] pFileName;
		delete [] pFileData;
		fclose(fpIn);
		printf("Error reading file. (errno: %d)\n", errno);
		return -2;
	}

	int nFilenameBlockOffset = xh.nNames * sizeof(FILENAME) + xh.nFiles * sizeof(FILEDATA) + sizeof(xh);
	int nFilenameBlockSize = xh.nDataOffset - nFilenameBlockOffset;

	char* pszFilenameBlock = new char[nFilenameBlockSize + 1];
	if(pszFilenameBlock == NULL)
	{
		delete [] pFileName;
		delete [] pFileData;
		fclose(fpIn);
		printf("Out of memory\n");
		return -2;
	}

	nRead = fread(pszFilenameBlock, 1, nFilenameBlockSize, fpIn);
	if(nRead != nFilenameBlockSize)
	{
		delete [] pszFilenameBlock;
		delete [] pFileName;
		delete [] pFileData;
		fclose(fpIn);
		printf("Error reading file. (errno: %d)\n", errno);
		return -2;
	}

	pszFilenameBlock[nFilenameBlockSize] = NULL;

	printf("Extracting files:\n\n");

	for(int n = 0; n < xh.nFiles; n++)
	{
		LPFILEDATA pData = &pFileData[n];

		const char* pszFilename = GetFilenameAt(pszFilenameBlock, n);

		if(pData->nType == 4)
		{
			printf("Skipping file '%s' ...\n", pszFilename);
			continue;
		}

		printf("Extracting file '%s' ... ", pszFilename);

		char szFilename[_MAX_PATH + 1];
		sprintf(szFilename, "%s\\%s", szFolder, pszFilename);
		FILE* fpOut = fopen(szFilename, "wb");
		if(fpOut == NULL)
		{
			delete [] pszFilenameBlock;
			delete [] pFileName;
			delete [] pFileData;
			fclose(fpIn);
			printf("Error creating output file '%s'. (errno: %d)\n", szFilename, errno);
			return -2;
		}

		fseek(fpIn, xh.nDataOffset + pData->nOffset, SEEK_SET);

		unsigned char cbBuffer[4096];
		unsigned long nTotalBytesWritten = 0L;

		long nBlocks = pData->nSize / sizeof(cbBuffer);
		for(int m = 0; m < nBlocks; m++)
		{
			nRead = fread(cbBuffer, 1, sizeof(cbBuffer), fpIn);
			if(nRead != sizeof(cbBuffer))
			{
				fclose(fpOut);
				delete [] pszFilenameBlock;
				delete [] pFileName;
				delete [] pFileData;
				fclose(fpIn);
				printf("Error reading file. (errno: %d)\n", errno);
				return -2;
			}

			int nWritten = fwrite(cbBuffer, 1, sizeof(cbBuffer), fpOut);
			if(nWritten != sizeof(cbBuffer))
			{
				fclose(fpOut);
				delete [] pszFilenameBlock;
				delete [] pFileName;
				delete [] pFileData;
				fclose(fpIn);
				printf("Error writing file. (errno: %d)\n", errno);
				return -2;
			}

			nTotalBytesWritten += nWritten;
		}
		long nRemainder = pData->nSize % sizeof(cbBuffer);
		if(nRemainder > 0)
		{
			nRead = fread(cbBuffer, 1, nRemainder, fpIn);
			if(nRead != nRemainder)
			{
				fclose(fpOut);
				delete [] pszFilenameBlock;
				delete [] pFileName;
				delete [] pFileData;
				fclose(fpIn);
				printf("Error reading file. (errno: %d)\n", errno);
				return -2;
			}

			int nWritten = fwrite(cbBuffer, 1, nRemainder, fpOut);
			if(nWritten != nRemainder)
			{
				fclose(fpOut);
				delete [] pszFilenameBlock;
				delete [] pFileName;
				delete [] pFileData;
				fclose(fpIn);
				printf("Error writing file. (errno: %d)\n", errno);
				return -2;
			}

			nTotalBytesWritten += nWritten;
		}

		fclose(fpOut);

		printf(" %lu bytes written.\n", nTotalBytesWritten);
	}

	delete [] pFileName;
	delete [] pFileData;
	fclose(fpIn);

	return 0;
}

int main(int argc, char* argv[])
{
	char* pszArchiveFile = "default.xip";

	printf("PIXIT XIP Archive Manipulator Version %d.%d\n", PIXIT_MAJOR_VER, PIXIT_MINOR_VER);
	printf("Copyright (c) Voltaic 2003. All rights reserved.\n");
	printf("\n");

	if(argc < 2)
		Usage(NULL);

	if(argv[argc - 1][0] != '-' && argv[argc - 1][0] != '/')
		pszArchiveFile = argv[argc - 1];

	if(GetCmdLineArg("create", argc, argv) != 0)
	{
		char* pszListFile = GetCmdLineArgValue("create", argc, argv);
		if(pszListFile == NULL)
			pszListFile = "default.csv";
		return CreateArchive(pszListFile, pszArchiveFile);
	}
	else if(GetCmdLineArg("display", argc, argv) != 0)
	{
		bool bCSV = false;
		char* pszOption = GetCmdLineArgValue("display", argc, argv);
		if(pszOption != NULL && stricmp(pszOption, "csv") == 0)
			bCSV = true;
		return DisplayArchiveInfo(bCSV, pszArchiveFile);
	}
	else if(GetCmdLineArg("extract", argc, argv) != 0)
	{
		char* pszExtractFolder = GetCmdLineArgValue("extract", argc, argv);
		return ExtractArchive(pszExtractFolder, pszArchiveFile);
	}

	Usage("Unknown command.");

	return 0;
}
