#define STRICT
#define WIN32_LEAN_AND_MEAN

#include <d3dx8.h>
#include <stdio.h>
#include <commdlg.h>
#include "resource.h"

typedef struct _tagXMHDR
{
	D3DPRIMITIVETYPE dwPrimitiveType;
	DWORD dwFaceCount;
	DWORD dwFVF;
	DWORD dwVertexStride;
	DWORD dwVertexCount;
	DWORD dwIndexCount;
} XMHDR, *LPXMHDR;

LPDIRECT3D8 g_pD3D = NULL;
LPDIRECT3DDEVICE8 g_pd3dDevice = NULL;
LPDIRECT3DVERTEXBUFFER8 g_pVertexBuffer = NULL;
LPDIRECT3DINDEXBUFFER8 g_pIndexBuffer = NULL;
XMHDR g_xh;

FLOAT g_fX = 0.0f, g_fY = 0.0f, g_fZ = 30.0f;

int WINAPI WinMain(HINSTANCE hInst, HINSTANCE hPrevInst, LPSTR lpCmdLine, int nCmdShow);
LRESULT CALLBACK WndProc(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam);
HRESULT Init(HWND hWnd);
HRESULT CreateBuffers(HWND hWnd, LPCTSTR pszXmFilename);
HRESULT Render();
void Shutdown();

int WINAPI WinMain(HINSTANCE hInst, HINSTANCE hPrevInst, LPSTR lpCmdLine, int nCmdShow)
{
	WNDCLASSEX wc;
	memset(&wc, 0, sizeof(wc));
	wc.cbSize = sizeof(WNDCLASSEX);
	wc.lpszClassName = "XMVIEWER_CLASS";
	wc.style = CS_HREDRAW | CS_VREDRAW;
	wc.lpfnWndProc = WndProc;
	wc.hInstance = hInst;
	wc.hIcon = LoadIcon(hInst, (LPCTSTR)IDI_DIRECTX_ICON);
    wc.hIconSm = LoadIcon(hInst, (LPCTSTR)IDI_DIRECTX_ICON);
	wc.hCursor = LoadCursor(NULL, IDC_ARROW);
	wc.hbrBackground = (HBRUSH)GetStockObject(BLACK_BRUSH);
	if(!RegisterClassEx(&wc))
	{
		MessageBox(NULL, "Failed to register window class.", "Error", MB_OK);
		return E_FAIL;
	}

	HWND hWnd = CreateWindowEx(NULL, "XMVIEWER_CLASS", "XM Viewer", 
		WS_OVERLAPPEDWINDOW | WS_VISIBLE, CW_USEDEFAULT, CW_USEDEFAULT, 640, 480, NULL, NULL, hInst, NULL);
	if(hWnd == NULL)
	{
		MessageBox(hWnd, "Failed to create main window.", "Error", MB_OK);
		return E_FAIL;
	}

    ShowWindow(hWnd, nCmdShow);
    UpdateWindow(hWnd);

	CHAR szFilename[MAX_PATH] = "";
	if(lpCmdLine != NULL && lpCmdLine[0] != NULL)
	{
		if(lpCmdLine[0] == '\"' && lpCmdLine[strlen(lpCmdLine) - 1] == '\"')
		{
			lpCmdLine[strlen(lpCmdLine) - 1] = NULL;
			lpCmdLine++;
		}
		strcpy(szFilename, lpCmdLine);
	}
	else
	{
		OPENFILENAME of;
		memset(&of, 0, sizeof(of));
		of.lStructSize = sizeof(of); 
		of.hwndOwner = hWnd; 
		of.hInstance = hInst; 
		of.lpstrFilter = "All supported formas (*.XIP;*.XM)\0*.XIP;*.XM\0Xbox archive (*.XIP)\0*.XIP\0Xbox Mesh (*.XM)\0*.XM\0All files (*.*)\0*.*\0"; 
		of.lpstrFile = szFilename; 
		of.nMaxFile = MAX_PATH; 
		of.Flags = OFN_FILEMUSTEXIST|OFN_READONLY; 
		if(GetOpenFileName(&of) == FALSE)
			return 0;
	}

	if(FAILED(Init(hWnd)))
	{
		MessageBox(hWnd, "Failed to initialize Direct3D.", "Error", MB_OK);
		return E_FAIL;
	}

	MSG msg;
	memset(&msg, 0, sizeof(msg));
	if(SUCCEEDED(CreateBuffers(hWnd, szFilename)))
	{
		while(msg.message != WM_QUIT )
		{
			if(PeekMessage(&msg, NULL, 0, 0, PM_REMOVE))
			{ 
				TranslateMessage(&msg);
				DispatchMessage(&msg);
			}
			else
			{
				try
				{
					Render();
				}
				catch(...)
				{
					MessageBox(hWnd, "Render() threw an exception.", "Error", MB_OK);
					break;
				}
			}
		}
	}

	Shutdown();

    UnregisterClass("XMVIEWER_CLASS", wc.hInstance);

	return msg.wParam;
}

LRESULT CALLBACK WndProc(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	switch(msg)
	{
	case WM_KEYDOWN:
		{
			switch(wParam)
			{
			case VK_ESCAPE:
				PostQuitMessage(0);
				return 0;
			case VK_ADD:
				g_fZ -= 1.0f;
				break;
			case VK_SUBTRACT:
				g_fZ += 1.0f;
				break;
			case VK_LEFT:
				g_fX -= 1.0f;
				break;
			case VK_RIGHT:
				g_fX += 1.0f;
				break;
			case VK_UP:
				g_fY += 1.0f;
				break;
			case VK_DOWN:
				g_fY -= 1.0f;
				break;
			}
		}
		break;
	case WM_CLOSE:
	case WM_DESTROY:
		PostQuitMessage(0);
		return 0;
	}

	return DefWindowProc( hWnd, msg, wParam, lParam );
}

HRESULT Init(HWND hWnd)
{
	g_pD3D = Direct3DCreate8(D3D_SDK_VERSION);
	if(g_pD3D == NULL)
		return E_FAIL;

	D3DDISPLAYMODE d3ddm;
	g_pD3D->GetAdapterDisplayMode(D3DADAPTER_DEFAULT, &d3ddm);

	D3DPRESENT_PARAMETERS d3dpp;
	ZeroMemory(&d3dpp, sizeof(d3dpp));

	d3dpp.Windowed = TRUE;
	d3dpp.SwapEffect = D3DSWAPEFFECT_DISCARD;
	d3dpp.BackBufferFormat = d3ddm.Format;
	d3dpp.EnableAutoDepthStencil = TRUE;
	d3dpp.AutoDepthStencilFormat = D3DFMT_D16;

	g_pD3D->CreateDevice(D3DADAPTER_DEFAULT, D3DDEVTYPE_HAL, hWnd, D3DCREATE_SOFTWARE_VERTEXPROCESSING, &d3dpp, &g_pd3dDevice);

	g_pd3dDevice->SetRenderState(D3DRS_LIGHTING, FALSE);
	g_pd3dDevice->SetRenderState(D3DRS_ZENABLE, TRUE);

	D3DXMATRIX matProj;
	D3DXMatrixPerspectiveFovLH(&matProj, 45.0f, 640.0f / 480.0f, 0.1f, 100.0f);
	g_pd3dDevice->SetTransform(D3DTS_PROJECTION, &matProj);

	return S_OK;
}

HRESULT CreateBuffers(HWND hWnd, LPCTSTR pszFilename)
{
	char szTitle[1024] = "";

	FILE* fp = fopen(pszFilename, "rb");
	if(fp == NULL)
	{
		MessageBox(hWnd, "Error opening file.", "Error", MB_OK);
		return E_FAIL;
	}

	if(fread(&g_xh, 1, sizeof(g_xh), fp) != sizeof(g_xh))
	{
		MessageBox(hWnd, "Error reading file.", "Error", MB_OK);
		fclose(fp);
		return E_FAIL;
	}

	if(g_xh.dwPrimitiveType < 1 || g_xh.dwPrimitiveType > 4)
	{
		MessageBox(hWnd, "Invalid primitive type.", "Error", MB_OK);
		fclose(fp);
		return E_FAIL;
	}

	g_pd3dDevice->CreateVertexBuffer(g_xh.dwVertexCount * g_xh.dwVertexStride, D3DUSAGE_DYNAMIC, g_xh.dwFVF, D3DPOOL_DEFAULT, &g_pVertexBuffer);

	BYTE* pcbVertices = NULL;
	g_pVertexBuffer->Lock(0, 0, &pcbVertices, D3DLOCK_DISCARD);
	int nRead = fread(pcbVertices, 1, g_xh.dwVertexCount * g_xh.dwVertexStride, fp);
	g_pVertexBuffer->Unlock();
	if((DWORD)nRead != g_xh.dwVertexCount * g_xh.dwVertexStride)
	{
		MessageBox(hWnd, "Error reading file.", "Error", MB_OK);
		fclose(fp);
		return E_FAIL;
	}

	g_pd3dDevice->CreateIndexBuffer(g_xh.dwIndexCount * sizeof(WORD), D3DUSAGE_DYNAMIC, D3DFMT_INDEX16, D3DPOOL_DEFAULT, &g_pIndexBuffer);

	BYTE* pcbIndices = NULL;
	g_pIndexBuffer->Lock(0, g_xh.dwIndexCount * sizeof(WORD), &pcbIndices, D3DLOCK_DISCARD);
	nRead = fread(pcbIndices, 1, g_xh.dwIndexCount * sizeof(WORD), fp);
	g_pIndexBuffer->Unlock();
	if((DWORD)nRead != g_xh.dwIndexCount * sizeof(WORD))
	{
		MessageBox(hWnd, "Error reading file.", "Error", MB_OK);
		fclose(fp);
		return E_FAIL;
	}

	sprintf(szTitle, "XM Viewer - '%s' (mesh only)", pszFilename);

	fclose(fp);

	SetWindowText(hWnd, szTitle);

	return S_OK;
}

void Shutdown()
{
	if(g_pIndexBuffer != NULL)
	{
		g_pIndexBuffer->Release();
		g_pIndexBuffer = NULL;
	}
	if(g_pVertexBuffer != NULL)
	{
		g_pVertexBuffer->Release(); 
		g_pVertexBuffer = NULL;
	}

	if(g_pd3dDevice != NULL)
	{
		g_pd3dDevice->Release();
		g_pd3dDevice = NULL;
	}

	if(g_pD3D != NULL)
	{
		g_pD3D->Release();
		g_pD3D = NULL;
	}
}

HRESULT Render()
{
	g_pd3dDevice->Clear(0, NULL, D3DCLEAR_TARGET | D3DCLEAR_ZBUFFER, D3DCOLOR_COLORVALUE(0.0f,0.0f,0.0f,1.0f), 1.0f, 0);

	static float fXrot = 0.0f;
	static float fYrot = 0.0f;
	static float fZrot = 0.0f;

	fXrot += 0.1f;
	fYrot += 0.2f;
	fZrot += 0.3f;

	D3DXMATRIX matWorld;
	D3DXMATRIX matTrans;
	D3DXMATRIX matRot;

	D3DXMatrixTranslation(&matTrans, g_fX, g_fY, g_fZ);

	D3DXMatrixRotationYawPitchRoll(&matRot, D3DXToRadian(fXrot), D3DXToRadian(fYrot), D3DXToRadian(fZrot));

	matWorld = matRot * matTrans;
	g_pd3dDevice->SetTransform(D3DTS_WORLD, &matWorld);

	g_pd3dDevice->BeginScene();
		g_pd3dDevice->SetStreamSource(0, g_pVertexBuffer, g_xh.dwVertexStride);
		g_pd3dDevice->SetVertexShader(g_xh.dwFVF);
		g_pd3dDevice->SetIndices(g_pIndexBuffer, 0);
		g_pd3dDevice->DrawIndexedPrimitive(g_xh.dwPrimitiveType, 0, g_xh.dwVertexCount, 0, g_xh.dwFaceCount);
	g_pd3dDevice->EndScene();

	g_pd3dDevice->Present(NULL, NULL, NULL, NULL);

	Sleep(2);

	return S_OK;
}
