package com.superhac.JXBStreamer.Core;

import java.io.File;
import java.util.ArrayList;
import java.util.logging.Logger;

/**
 * The DirectoryHandlerObject maps directories to handles and provides services
 * for fulfilling client requests on directories. E.g. get a list of files in a
 * directory.
 * <p>
 * <p>
 * More information can be found at <a
 * href="http://www.superhac.com">Superhac.com</a>
 * 
 * @author Steven Scott (superhac007@gmail.com)
 */

public class DirectoryHandlerObject {

	/** The Directory Handle assigned by the XBMSPServerMessageHandler */
	private int handle;

	/** The Directory Path. will be null if it is the Virtual Root */
	private File path;

	/** A list of the files in the directory. */
	private ArrayList<File> files = new ArrayList<File>();

	/**
	 * the last path seqment of all virtual roots. E.g. the last the directory.
	 * No files!
	 */
	private ArrayList<File> virtualRoot = null;

	/** how many files in directory */
	private int totalDirectoryEntries;

	/** The current Directory listing position were on */
	private int directoryListingPosition = 0;

	/** debug logger */
	private static Logger logger;

	/** If this is a directory handle to the virtual root this is set */
	private boolean isVirtualRoot = false;

	/**
	 * All regular non-Virtual root paths will use this constructor.
	 * 
	 * @param handle
	 *            the handle assigned to this path.
	 * @param path
	 *            the acutal directory path.
	 */
	protected DirectoryHandlerObject(int handle, File path) {
		// if (com.superhac.JXBStreamer.Core.Debug.debug)
		logger = com.superhac.JXBStreamer.Core.Debug.getLogger();

		this.handle = handle;
		this.path = path;

		buildDirectoryList();

	}

	/**
	 * Only use this constructor if its the virtual root directory.
	 * 
	 * @param handle
	 *            handle the handle assigned to this path.
	 * @param virtualRoot
	 *            The virtual root File list.
	 */
	protected DirectoryHandlerObject(int handle, ArrayList<File> virtualRoot) {
		logger = com.superhac.JXBStreamer.Core.Debug.getLogger();
		this.handle = handle;
		this.virtualRoot = virtualRoot;
		this.path = null;
		isVirtualRoot = true;

		buildVirtualRootList();
	}

	/**
	 * Builds the list of files(or directories) contained with this directory.
	 * Not to be used for the virtual root.
	 */
	private void buildDirectoryList() {
		// clear out array
		files.removeAll(files);

		// get the listing of the current directory
		File[] children = path.listFiles();

		// add some kind of filters a some point... like to remove . files

		// add each file as FILE to the arraylist
		for (int i = 0; i < children.length; i++)
			files.add(children[i]);

		// set the total number of Directory entries
		totalDirectoryEntries = files.size();

		if (com.superhac.JXBStreamer.Core.Debug.debug)
			logger.info("Build directory list.  Total Entries:"
					+ totalDirectoryEntries);

	}

	/**
	 * Build the directory list for the Virtual Root directory. Only used when
	 * its the virtual root.
	 */
	private void buildVirtualRootList() {
		files.removeAll(files);
		// File[] files = new File[virtualRoot.size()];

		// copy array list

		files  = (ArrayList) virtualRoot.clone();
		// set the total number of Directory entries
		totalDirectoryEntries = files.size();

		

	}

	/**
	 * Get the handle number for this directory.
	 * 
	 * @return The handle number.
	 */
	protected int getHandle() {
		return handle;

		/** String[0] = name, String[1] = XML Data */
	}

	/**
	 * Gets the next directory item(file) requested by the client.
	 * 
	 * @return String[0] is the name of the file, String[1] = contains the XML
	 *         data for client response.
	 */
	protected String[] getNextEntry() {
		String[] payloadStrings = new String[2];

		/**
		 * if there are no more entries reset the directoryListingPosition back
		 * to 0 and send null strings
		 */
		if (directoryListingPosition == totalDirectoryEntries) {
			payloadStrings[0] = null;
			payloadStrings[1] = null;

			// reset position
			directoryListingPosition = 0;

			// refresh the list... if there were new files added they will show
			// up!
			if (isVirtualRoot)
				buildVirtualRootList();
			else
				buildDirectoryList();

			if (com.superhac.JXBStreamer.Core.Debug.debug)
				logger.info("Last directory entry.  Reset List..");

			return payloadStrings;

		}

		// System.out.println("current position in dir: "+
		// directoryListingPosition );
		if (com.superhac.JXBStreamer.Core.Debug.debug)
			logger
					.info("Reading and packing next directory entry at position: "
							+ directoryListingPosition);

		File entry = files.get(directoryListingPosition);

		// build the String Name and the XML Data payload
		payloadStrings[0] = entry.getName();

		// call the XML function for String[1]
		payloadStrings[1] = XBMSPxmlEncoder.encodeEntry(entry);

		// increment the current position
		directoryListingPosition++;
		return payloadStrings;

	}

	/**
	 * Gets the file index number of the named file within the directory.
	 * 
	 * @param name
	 *            The name of the file
	 * @return the index number of the requested filename.
	 */
	protected int getFileIndexNumber(String name) {
		if (com.superhac.JXBStreamer.Core.Debug.debug)
			logger.info("Getting file index number in directory for file <"
					+ name + "?");

		for (int i = 0; i < files.size(); i++)
			if (files.get(i).getName().compareTo(name) == 0) // matched
				return i;

		return -1;

		/** this needs to throw an exception is the file does not exists */
	}

	/**
	 * Gets a specfic directory item specified by index.
	 * 
	 * @param index
	 *            The index number of the requested file. Get from
	 *            getFileIndexNumber(String name).
	 * @return String[0] is the name of the file, String[1] = contains the XML
	 *         data for client response.
	 */
	protected String[] getEntryDataByIndex(int index) {
		if (com.superhac.JXBStreamer.Core.Debug.debug)
			logger.info("Getting entry data(xml) by index...");

		String[] payloads = new String[2];

		payloads[0] = files.get(index).getName();
		payloads[1] = XBMSPxmlEncoder.encodeEntry(files.get(index));

		return payloads;
	}

	/**
	 * This gets the File Object for the index number requested.
	 * 
	 * @param index
	 *            The index number of the requested file. Get from
	 *            getFileIndexNumber(String name).
	 * @return The file object of the request.
	 */
	protected File getFileByIndex(int index) {
		if (com.superhac.JXBStreamer.Core.Debug.debug)
			logger.info("Get file by index....");

		File file = files.get(index);
		return file;
	}

	/**
	 * Gets the path(File) represented by this DirectoryHandlerObject.
	 * 
	 * @return The path.
	 */
	protected File getPath() {
		return path;
	}

}
