/*   -*- c -*-
 * 
 *  ----------------------------------------------------------------------
 *  CcXstream Client Library for XBOX Media Player
 *  ----------------------------------------------------------------------
 *
 *  Copyright (c) 2002-2003 by PuhPuh
 *  
 *  This code is copyrighted property of the author.  It can still
 *  be used for any non-commercial purpose following conditions:
 *  
 *      1) This copyright notice is not removed.
 *      2) Source code follows any distribution of the software
 *         if possible.
 *      3) Copyright notice above is found in the documentation
 *         of the distributed software.
 *  
 *  Any express or implied warranties are disclaimed.  Author is
 *  not liable for any direct or indirect damages caused by the use
 *  of this software.
 *
 *  ----------------------------------------------------------------------
 *
 */

#include "ccincludes.h"
#include "ccbuffer.h"
#include "ccxclient.h"

#define CC_TIMEVAL_CMP(t1, t2)                       \
       ((((t1)->tv_sec == (t2)->tv_sec) &&           \
         ((t1)->tv_usec == (t2)->tv_usec)) ?         \
        0 :                                          \
       ((((t1)->tv_sec < (t2)->tv_sec) ||            \
         (((t1)->tv_sec == (t2)->tv_sec) &&          \
          ((t1)->tv_usec < (t2)->tv_usec))) ?        \
        -1 :                                         \
        1))

CcXstreamClientError cc_xstream_client_connect(const char *host,
					       int port,
					       CcXstreamServerConnection *s)
{
  int sock;
  struct sockaddr_in sa;
  socklen_t sa_len;
  struct hostent *he;

  he = gethostbyname(host);
  if ((he == NULL) || (he->h_addrtype != AF_INET) || (he->h_length != 4))
    return CC_XSTREAM_CLIENT_SERVER_NOT_FOUND;
  sock = socket(AF_INET, SOCK_STREAM, 0);
  if (sock < 0)
    return CC_XSTREAM_CLIENT_FATAL_ERROR;
  memset(&sa, 0, sizeof (sa));
  sa.sin_family = AF_INET;
  memcpy(&(sa.sin_addr), he->h_addr, 4);
  sa.sin_port = htons(port);
  sa_len = sizeof (sa);
  if (connect(sock, (struct sockaddr *)&sa, sa_len) != 0)
    {
      close(sock);
      return CC_XSTREAM_CLIENT_SERVER_CONNECTION_FAILED;
    }
  *s = (CcXstreamServerConnection)sock;
  return CC_XSTREAM_CLIENT_OK;
}

CcXstreamClientError cc_xstream_client_disconnect(CcXstreamServerConnection s)
{
  int sock;

  sock = (int)s;
  close(s);
  return CC_XSTREAM_CLIENT_OK;
}

unsigned char *cc_xstream_client_read_data(CcXstreamServerConnection s, 
					   size_t len, 
					   unsigned long timeout_ms)
{
  unsigned char *buf;
  size_t done;
  int rv, sock;
  struct timeval t0, t1, t2;
  fd_set rfds;

  /* Convert the conenction handle to simple file descriptor. */
  sock = (int)s;

  /* We terminate incoming buffer just to make code safer. 
     Caller should not count on it anyway. */
  buf = cc_xmalloc(len + 1);
  buf[len] = '\0'; 

  /* If we can't get time, disable timeout. */
  if (gettimeofday(&t0, NULL) != 0)
    timeout_ms = 0;
  
  for (done = 0; done < len; /*NOTHING*/)
    {
      if (timeout_ms > 0)
	{
	  if (gettimeofday(&t1, NULL) != 0)
	    {
	      timeout_ms = 0;
	      goto disable_timeout;
	    }
	  /* Check if clock has gone backwards. */
	  if (CC_TIMEVAL_CMP(&t0, &t1) > 0)
	    t0 = t1;
	  t2.tv_sec = t1.tv_sec - t0.tv_sec;
	  if (t0.tv_usec > t1.tv_usec)
	    {
	      t2.tv_sec--;
	      t2.tv_usec = t1.tv_usec + 1000000 - t0.tv_usec;
	    }
	  else
	    {
	      t2.tv_usec = t1.tv_usec - t0.tv_usec;
	    }
	  FD_ZERO(&rfds);
	  FD_SET(sock, &rfds);
	  if (select(sock + 1, &rfds, NULL, NULL, &t2) != 1)
	    {
	      /* Timeout or error, we don't really care. */
	      cc_xfree(buf);
	      return NULL;
	    }
	}
    disable_timeout:
      rv = read(sock, buf + done, len - done);
      if (rv < 1)
	{
	  cc_xfree(buf);
	  return NULL;
	}
      done += rv;
    }
  return buf;
}

int cc_xstream_client_write_data(CcXstreamServerConnection s,
				 unsigned char *buf,
				 size_t len,
				 unsigned long timeout_ms)
{
  size_t done;
  int rv, sock;
  struct timeval t0, t1, t2;
  fd_set wfds;

  /* Convert the conenction handle to simple file descriptor. */
  sock = (int)s;

  /* If we can't get time, disable timeout. */
  if (gettimeofday(&t0, NULL) != 0)
    timeout_ms = 0;

  for (done = 0; done < len; /*NOTHING*/)
    {
      if (timeout_ms > 0)
	{
	  if (gettimeofday(&t1, NULL) != 0)
	    {
	      timeout_ms = 0;
	      goto disable_timeout;
	    }
	  /* Check if clock has gone backwards. */
	  if (CC_TIMEVAL_CMP(&t0, &t1) > 0)
	    t0 = t1;
	  t2.tv_sec = t1.tv_sec - t0.tv_sec;
	  if (t0.tv_usec > t1.tv_usec)
	    {
	      t2.tv_sec--;
	      t2.tv_usec = t1.tv_usec + 1000000 - t0.tv_usec;
	    }
	  else
	    {
	      t2.tv_usec = t1.tv_usec - t0.tv_usec;
	    }
	  FD_ZERO(&wfds);
	  FD_SET(sock, &wfds);
	  if (select(sock + 1, NULL, &wfds, NULL, &t2) != 1)
	    {
	      /* Timeout or error, we don't really care. */
	      return 0;
	    }
	}
    disable_timeout:
      rv = write(sock, buf + done, len - done);
      if (rv < 1)
	return 0;
      done += rv;
    }
  return 1;
}

/* eof (ccxclientconn.c) */
