/************************************************************************
  XBox media server v0.30.6 (http://www.h4x0rz.de)

  xbms is a *nix streaming server for xbox mediaplayer. 
  It provides access to shared directory on a unix box right from the 
  xboxmediaplayer2  (www.xboxmediaplayer.de).

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Lets do the Credits (in order of appearance):
  Ugly code by jeff <jeff@bidou.info> (initial code), BigUglyFool, Runtime, 
               d7o3g4q, Stefan Mansby <stefan@nis.nu>, tric, don,
			   Forsaken, tc

  Currently maintaining is tric (tric@h4x0rz.de)

  TODO:
  - test and enhance player2 support && add iso/bin support
  - cleanup code, find overflow and security issues
  
*************************************************************************/

#include <sys/stat.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <sys/wait.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <unistd.h>
#include <dirent.h>   
#include <time.h>
#include <fcntl.h>
#include <stdlib.h>
#include <pwd.h>
#include <grp.h>
#include <errno.h>
#include <signal.h>
#include <pthread.h>


#define VERSION "0.30.6-dev"

#define CONFIG "/etc/xbms.conf"
#define PIDFILE "/var/run/xbms.pid"
#define C_SERVER_PORT 1400
#define MAX_MSG_LENGTH 4096
#define MAX_CMD_LENGTH 4

/* user and group will only be used if running as root
 * or if you pass it by cmdline						*/
#define USER_NAME "nobody"
#define GROUP_NAME "nogroup"

/* This should be the only thing to configure for the moment */
#define MEDIA_FILES_PATH "/tmp" /* go somewhere reasonably safe as default */

struct main_config_s {
	char 	*log_file;
	char 	*current_path;
	char	*root_dir;
	char	*user;
	char	*group;
	char 	*config_file;
	char	*pid_file;
	unsigned char foreground;
	int 	sd;
	int     debug_lvl;
	uid_t   first_user;
	int		port;
	int		running;
	in_addr_t remote_ip;
	struct in_addr local_ip;
};

struct main_config_s config;
pthread_mutex_t data_mutex = PTHREAD_MUTEX_INITIALIZER;

extern char *ctime(const time_t *time_tzgr);
extern time_t time(time_t *time_tzgr);
extern unsigned long inet_addr(const char *cp);
extern char *inet_ntoa(struct in_addr in);
extern int isspace(int c);

int free_mainconfig(struct main_config_s *main_config) {
    
	if (main_config->user!=NULL) {
        free(main_config->user);
        main_config->user=NULL;
    }
    if (main_config->group!=NULL) {
        free(main_config->group);
        main_config->group=NULL;
    }
    if (main_config->log_file!=NULL) {
        free(main_config->log_file);
        main_config->log_file=NULL;
    }
    if (main_config->current_path!=NULL) {
        free(main_config->current_path);
        main_config->current_path=NULL;
    }
    if (main_config->pid_file!=NULL) {
        free(main_config->pid_file);
        main_config->pid_file=NULL;
    }
    if (main_config->root_dir!=NULL) {
        free(main_config->root_dir);
        main_config->root_dir=NULL;
    }
    if (main_config->config_file!=NULL) {
        free(main_config->config_file);
        main_config->config_file=NULL;
    }
	return 0;
}



int usage(char *prog, struct main_config_s *main_config)
{
  printf("XBox media server %s (http://www.h4x0rz.de)\n",VERSION);
  printf("Usage:");
  printf("%s [options]\n\n", prog);
  printf("\t-i [ip-address]");
  printf("\t-d [media dir]\n");
  printf("\t-p [port]\n");
  printf("\t-P [pidfile]\n");
  printf("\t-u [username to run as]\n");
  printf("\t-g [group to run as]\n");
  printf("\t-D [level] debug mode (lvl 20 VERY verbose)\n");
  printf("\t-c [config file]\n");
  printf("\t-l [logfile]\n");
  printf("\t-r [allow only this remote ip]\n");
  printf("\t-f foreground, do not deamonize\n");
  printf("\t-h help\n");
  
  free_mainconfig(main_config);
  exit(1);
}

/* This function does not make any safety check, beware ! */
int substr(char *buf, char **dest, int s, int l,int e)
{
  if( l > (signed int)(strlen(buf) - s) )
     l = strlen(buf) - s;

  if (e == 1)
    memset(*dest,0x0,l+1);
  memcpy(*dest,buf+s,l);
  return 1;
}

/* Returns the string aligned to the left */
int pad_right( char *src, char *dst, char fill, int totallength )
{
int srclen = strlen(src);

   if( totallength <= srclen ) {
      strncpy( dst, src, totallength );
   } else {
      strcpy( dst, src );
      memset( dst + srclen, fill, totallength - srclen );
   }
   dst[totallength] = 0;
   return 1;
}


void d_log(char *fmt, ...) {
 time_t     timenow;
 FILE       *file;
 va_list    ap;
 int pid;
 pid=getpid();

 va_start(ap, fmt);
 timenow = time(NULL);
 if (config.log_file!=NULL) {
	 if ((file = fopen(config.log_file, "a")) != NULL ) {
	    fprintf(file, "%.24s (%i) - ", ctime(&timenow),pid);
		vfprintf(file, fmt, ap);
		fclose(file);
	 } else {
		printf("%.24s (%i) - ", ctime(&timenow),pid);
		vprintf(fmt, ap);
	 }
 } else {
	printf("%.24s (%i) - ", ctime(&timenow),pid);
	vprintf(fmt, ap);
 }
}

char *trim_whitespace(char *str)
{
char *dud = str;
int i;

   // beginning whitespace first
   while( isspace(*dud) )
      ++dud;
   // now trailing whitespace
   i = strlen(dud) - 1;
   while( isspace(dud[i]) )
      --i;
   dud[i+1] = 0;
   return dud;
}


int read_config(char *config_file, char *parameter, struct main_config_s *main_config)
{
FILE *fp;
char *buffer;
char *token, *value;

  if (config_file==NULL) return -1;

  if( (fp = fopen( config_file, "r" ) ) == NULL ) {
     fprintf( stderr, "Unable to read config file %s. Continuing without it.\n",
           config_file );
     return -1;
  }

  buffer = (char*)malloc( sizeof(char) * 4096 );

  while( fgets( buffer, 4096, fp ) ) {
   value = trim_whitespace( buffer );
//   token = strtok( value, " " ); /*replaced strsep by strtok for tru64 comp reason */
   token = strsep( &value, " " );   
   if( token == NULL )  /* ignore this line if there isn't a token/value pair */
      continue;
   token = trim_whitespace( token );
   token = trim_whitespace( token );

   if( strcmp( token, "USER" ) == 0 ) {
	  if (strchr(parameter,'u')==NULL) { 
	      if( main_config->user != NULL )
		     free( main_config->user );
	      main_config->user = strdup( value );
	  }
   } else if( strcmp( token, "GROUP" ) == 0 ) {
	   if (strchr(parameter,'g')==NULL) {
	      if( main_config->group != NULL )
		     free( main_config->group );
	      main_config->group = strdup( value );
	   }
   } else if( strcmp( token, "PATH" ) == 0 ) {
	   if (strchr(parameter,'d')==NULL) {
	      if( main_config->root_dir != NULL )
		     free( main_config->root_dir );
	      main_config->root_dir = strdup( value );
	   }
   } else if( strcmp( token, "LOGFILE" ) == 0 ) {
	   if (strchr(parameter,'l')==NULL) {
	      if( main_config->log_file != NULL )
		     free( main_config->log_file );
	      main_config->log_file = strdup( value );
	   }
   } else if( strcmp( token, "PIDFILE" ) == 0 ) {
	   if (strchr(parameter,'P')==NULL) {
	      if( main_config->pid_file != NULL )
		     free( main_config->pid_file );
	      main_config->pid_file = strdup( value );
	   }
   } else if( strcmp( token, "PORT" ) == 0 ) {
	   if (strchr(parameter,'p')==NULL) {
	      main_config->port = (unsigned short int)strtol( value, (char**)NULL, 10 );
	   }
   } else if( strcmp( token, "FOREGROUND" ) == 0 ) {
		  main_config->foreground = 1;
   } else if( strcmp( token, "LOCAL_IP" ) == 0 ) {
	   if (strchr(parameter,'i')==NULL) {
	      main_config->local_ip.s_addr = inet_addr( value );
	   }
   } else if( strcmp( token, "REMOTE_IP" ) == 0 ) {
	   if (strchr(parameter,'r')==NULL) {
	      main_config->remote_ip = inet_addr( value );
	   }
   } else if( strcmp( token, "DEBUG" ) == 0 ) {
	   if (strchr(parameter,'D')==NULL) {
	      main_config->debug_lvl = strtol( value, (char**)NULL, 10 );
	   }
   }
  }
  free(buffer);
  fclose(fp);

 return 0;
}

void sig_int(int signr) {
   if(signr) {} /* get rid of compiler warning. will be optimized away by compiler */
	pthread_mutex_lock(&data_mutex);
    config.running=0;
	if (config.sd!=0) close(config.sd);
	pthread_mutex_unlock(&data_mutex);
}

void sig_hup(int signr) {
   if(signr) {} /* get rid of compiler warning. will be optimized away by compiler */
	pthread_mutex_lock(&data_mutex);
	read_config(config.config_file,"", &config);
	pthread_mutex_unlock(&data_mutex);
}


int tbnexists(char *filename, char *newext,int DEBUG)
{
char *tmpbuf,*ext;
char result;

   tmpbuf = (char *)malloc(strlen(filename)+strlen(newext)+1);

   strcpy(tmpbuf,filename);
   ext = strrchr( tmpbuf, '.' );
   if( ext == NULL )
      ext = tmpbuf + strlen(tmpbuf); /* now points to end of filename */

   strcpy(ext, newext );
   if (DEBUG > 5 ) d_log("%s: %s\n",newext,tmpbuf);
   result = ( access( tmpbuf, F_OK | R_OK ) == 0 ) ? 1 : 0;
   free(tmpbuf);
   return result;
}


int get_dirlist( const char *dir, char *TEMP_PATH, unsigned long cnt, int DEBUG,  char **PAYLOAD) {
char *targetfileName, *ext;
DIR *dirp,*test_dirp;
struct dirent *direntp;
int attrib;
struct stat st;
FILE *f;

    if ((dirp = opendir(TEMP_PATH))== NULL) {
      return cnt;
    }

    targetfileName = (char*)malloc( sizeof(char) * PATH_MAX );

    while ((direntp = readdir(dirp)) != NULL) {
        *targetfileName = 0;
        sprintf(targetfileName,"%s/%s",TEMP_PATH,direntp->d_name);
        stat(targetfileName,&st);
        /* let's get the extension */
		ext = strrchr( direntp->d_name, '.' );

		attrib = 0;
           /*either we found a dir or a file*/
		   /* test if we have enough permissions, else skip this */
		if (S_ISDIR(st.st_mode) && strcmp(direntp->d_name,".") && strcmp(direntp->d_name,"..") ) {
			if ((test_dirp = opendir(targetfileName))== NULL) 
				continue;
			else
				closedir(test_dirp);
			attrib=16;
		} else if (S_ISREG(st.st_mode) && ext != NULL && strcasecmp(".tbn",ext)!=0) {
                       /* we send the server the full path from now on */
			if (!(f=fopen(targetfileName,"r"))) 
				continue; 
			else 
				fclose(f);
			attrib=128;
		}
		
		if( attrib ) {
            if (tbnexists(targetfileName,".tbn",DEBUG)) attrib|= 4;
			if (DEBUG >= 3)
					d_log("-> attr: %d %s/%s\n",attrib, dir, direntp->d_name);
			*PAYLOAD=(char*)realloc( *PAYLOAD,
			         strlen("/<ITEM><ATTRIB></ATTRIB><PATH></PATH></ITEM>\n") +
			((dir==NULL) ? 0 : strlen( dir )) + strlen( direntp->d_name)
						+ 10 + strlen( *PAYLOAD ) + 1 );
			sprintf( *PAYLOAD,"%s<ITEM><ATTRIB>%d</ATTRIB><PATH>%s/%s</PATH></ITEM>\n",
					*PAYLOAD, attrib, (dir==NULL) ? "" : dir, direntp->d_name );
			cnt++;
		}
    }
    closedir(dirp);
    free(targetfileName);
    return cnt;
}

char *createpath(char *first, char *last) 
{
	char *result=NULL;
	if (first!=NULL && last!=NULL) 
	{
		if (first[strlen(first)-1]=='/' && last[0]=='/') 
		{
			result=(char *)malloc(strlen(first)+strlen(last));
			first[strlen(first)-1]=0;
			sprintf(result,"%s%s",first,last);
		} else
			if ((first[strlen(first)-1]!='/' && last[0]=='/') || (first[strlen(first)-1]=='/' && last[0]!='/'))
			{
				result=(char *)malloc(strlen(first)+strlen(last)+1);
				sprintf(result,"%s%s",first,last);
			} else
			{
				result=(char *)malloc(strlen(first)+strlen(last)+2);
				sprintf(result,"%s/%s",first,last);
			}
		
	}
	return result;
}


int check_command(char *line,FILE **file,char **CMD, char **PAYLOAD, int *payloadSize, struct main_config_s *main_config)
{
  char *cmd;
  char *ptr;
  char *fileName;
  char *targetfileName;
  struct stat st;
  int size1,size2;
  static char *tempbuf;
  unsigned long cnt;
  int ret=0;
  char *TEMP_PATH;
  char *TARGET_PATH = main_config->root_dir;

  *CMD = (char *)malloc(MAX_MSG_LENGTH);
  memset(*CMD,0x0,MAX_MSG_LENGTH);
  cmd = malloc(10);
/* strncpy ! */
  if (!substr(line,&cmd,0,4,1))
    {
      printf("substr error\n");
      free(cmd);
      return 1;
    }	
  
  if (main_config->debug_lvl >= 2)
    d_log("LINE:%s CMD: %s\n", line, cmd);
  
  /***************** XBOX IS CONNECTING *****************/
  if (strcmp(line,"HELLO XSTREAM 6.0") == 0)
    {
      if (main_config->debug_lvl >= 1) d_log("Greetings, xbox !\n");
      strcpy(*CMD,"HELLO XBOX!");
      ret = 1;
    }

  /****************** OPEN A FILE ***********************/
  if (strcmp(cmd,"OPEN") == 0)
    {
      /* open the file */
      if (main_config->debug_lvl >= 1) d_log("OPEN\n");
      
      /* Get the argument after comma */
      ptr = strtok(line,",");
      ptr = strtok(NULL,",");
      
      fileName = (char *)malloc(strlen(ptr)+1);
  //    targetfileName = (char *)malloc(strlen(TARGET_PATH)+strlen(ptr)+2);
      
      /* Check if the file is already opened, since the XBPlayer code does not send the CLSE command yet */
      if (*file != NULL) fclose(*file);
      
      strcpy(fileName, ptr);

      /* Create a new filename containing the default media path */

      /* Attach the requested filename to the end. */
//	  sprintf(targetfileName,"%s%s",TARGET_PATH,fileName);
	  targetfileName=createpath(TARGET_PATH,fileName);

      /* If /../ exists in our filename, we will report it and refuse */
      /* to open the requested file.  Otherwise processing should continue */
      if (!strstr(fileName,"/../")) 
	{
	  if (!(*file = fopen(targetfileName,"r")))
	    {
	      if (main_config->debug_lvl >= 1)	{
			  d_log("Cannot open file: %s\n",targetfileName);
		  }	     
		  sprintf(*CMD,"%-32d",-1);
	      free(fileName);
	      free(targetfileName);
	      ret = 1;
	    }
	  else
	    {
	      fstat(fileno(*file),&st);
		  /* set it to 32 char, so player understands it.*/
		  sprintf(*CMD,"%-32d",(int)st.st_size);
		  
	      free(fileName);
	      free(targetfileName);
	      ret = 1;
	    }
	}
      else d_log("Illegal string in filename: %s\n", fileName); 
    }

  /***************************** READ A PART OF A FILE ****************/
  if (strcmp(cmd,"READ") == 0)
    {
      /* read part of the file */
      if (!(strchr(line,',')))
	printf("No comma in command line, ignoring\n");
      else
	{
	  /* let's get the second token, it's the length of the packet wanted */
	  ptr = strtok(line,",");
	  ptr = strtok(NULL,",");
	  size1 = atoi(ptr);
	  ptr = strtok(NULL,",");
	  size2 = atoi(ptr);
	  
	  if (size2 > 0)
	    {
	      if ((file == NULL))
		{
		  d_log("File not opened !\n");
		  /* Send a 0 OK message so XBPlayer knows it's the "end" of the stream... */
		  sprintf(*CMD,"%-32d",0);
		  ret = 1;
		}
	      else
		{
		  *PAYLOAD = (char *)calloc(size2,1);
		  fseek(*file,size1,SEEK_SET);
		  if ((*payloadSize = fread((*PAYLOAD),1,size2,*file)))
		    {
				/* set it to 32 char, so player understands it.*/
				sprintf(*CMD,"%-32d",*payloadSize);
				/* let's pad the result to the right of a 32 bytes string */
				ret = 2;
		    }
		  else
		    {
		      if (feof(*file))
			{
			  /* We have reached the end of the file, send a 0 length header so the xbox knows
			     about it ! */
			  if (main_config->debug_lvl >= 1)
			    d_log("Past file end !\n");
			  sprintf(*CMD,"%-32s","-1");
			  ret = 1;
			}
		      else
			d_log("can't read from file !\n");
		    }
		  
		  /* Do some cleanup ! */
		}
	    } /* buffer size is ok */
	  else
	    d_log("0 or invalid paramater for READ, ignoring\n");
	} /* comma detected */
    }

  /******************************* TELL ************************/
  /* really not sure if we still need this command             */
  /* soneone has any advice                                    */
  /* anyway changed it to 32 chars too                         */

  if (strcmp(cmd,"TELL") == 0)
    {
      /* sends the current position in the file stream */
      size1 = ftell(*file);
      if (size1 >= 0)
	{
	  sprintf(*CMD,"%-32d",size1);
	  ret = 1;
	}
      else
	{
	  sprintf(*CMD,"%-32d",-1);
	  ret = 1;
	}
    }

  /***************************** CLOSE ************************/
  if (strcmp(cmd,"CLSE") == 0)
    {
      /* close the file */
      if (*file != NULL)
	{
	  fclose(*file);
	  if (main_config->debug_lvl>0)  d_log("Closing file\n");
	}
      else	if (main_config->debug_lvl>0)  d_log("File not opened ?\n");
      *file = NULL;
      
      /* send the answer */
      /* set it to 32 char, so player understands it.*/
	  sprintf(*CMD,"%-32i",0);
      ret = 1;
    }
  
  /***************************** DBUG ************************/
  /* really not sure if we still need this command             */
  /* soneone has any advice                                    */
  /* anyway changed it to 32 chars too                         */
  if (strcmp(cmd,"DBUG") == 0)
    {
	  sprintf(*CMD,"%-32i",0);
      ret = 1;
    }
  
  /***************************** INFO ************************/
  /* anyway changed it to 32 chars too                         */
  if (strcmp(cmd,"INFO") == 0)
    {
      /* Not implemented yet ! */
	  sprintf(*CMD,"%-32i",-1);
      ret = 1;
    }

  /***************************** *CAT ************************/
  /* ok, we need *CURRENT_PATH here if we got the BACK cmd   */
  /* in all other cases we use full paths from now on, so    */
  /* mp3 playlists work from now on                          */
  if (strcmp(cmd,"*CAT") == 0) {
	  if (main_config->debug_lvl >= 5) 
			d_log("CurDirectory: %s\n",main_config->current_path);

      /* Get the character after the comma, if any */
      ptr = strtok(line,",");
      if ((ptr = strtok(NULL,",")))	{
		  /* Check the command */
		if (!strcmp(ptr,"BACK")) {
			  /* We can't go back if the current path is empty ! */
			if (main_config->current_path!=NULL && main_config->current_path[0] != 0) {
				  fileName = (char *)malloc(strlen(main_config->current_path)+1);
				  tempbuf=strdup(rindex(main_config->current_path,'/'));
				  substr(main_config->current_path,&fileName,0,strlen(main_config->current_path)-strlen(tempbuf),1);
				  free(main_config->current_path);
				  main_config->current_path=strdup(fileName);
				  free(tempbuf);
				  free(fileName);
			  }
		  } else {
			  if (ptr[strlen(ptr)-1]=='/') ptr[strlen(ptr)-1]=0;
			  if (main_config->current_path!=NULL) free(main_config->current_path);
			  main_config->current_path=strdup(ptr);
		  }
	  }

	  if (main_config->debug_lvl >= 5) 
		  d_log("targetpath: %s - curpath: %s - ptr: %s\n",TARGET_PATH,main_config->current_path,ptr);

	  if (main_config->current_path!=NULL) {
//		  TEMP_PATH=(char *)malloc(strlen(TARGET_PATH)+strlen(main_config->current_path)+1);
//    	  sprintf(TEMP_PATH,"%s%s",TARGET_PATH,main_config->current_path);
    	  TEMP_PATH=createpath(TARGET_PATH,main_config->current_path);
	  } else {
		  TEMP_PATH=(char *)malloc(strlen(TARGET_PATH)+1);
		  strcpy(TEMP_PATH,TARGET_PATH);
	  }

      if (main_config->debug_lvl >= 1) d_log("Directory: %s\n",TEMP_PATH);

      *PAYLOAD = (char *)malloc(strlen("<SHARES>\n")+1);
      sprintf(*PAYLOAD,"<SHARES>\n");
	  cnt=get_dirlist(main_config->current_path,TEMP_PATH,0,main_config->debug_lvl,PAYLOAD);
	  free(TEMP_PATH);

      if (cnt > 0)	{
		  /* We found at least one media file, we can finish the buffer*/
		  *PAYLOAD = (char *)realloc(*PAYLOAD,strlen(*PAYLOAD)+strlen("</SHARES>")+1);
		  strcat(*PAYLOAD,"</SHARES>");
		  *payloadSize = strlen(*PAYLOAD);
		  sprintf(*CMD,"%-32d",*payloadSize);

		  ret = 2;
	  } else { /* No file found ! */
		  free(*PAYLOAD);
		  *PAYLOAD=NULL;
		  *payloadSize = 0;
		  sprintf(*CMD,"%-32d", -1);
		  TEMP_PATH=strrchr(main_config->current_path,'/');
		  if (TEMP_PATH!=NULL)  TEMP_PATH[0]=0;
		  if (main_config->debug_lvl>=3)  d_log("curpath after: %s\n",main_config->current_path);
		  ret = 1;
	  }
	}

  free(cmd);
  return ret;
}

int switch_user(char *user,char *groupname) {
  	struct passwd *u_pwd;
    struct group *g_group;
	
    if (groupname!=NULL) {
        g_group=getgrnam(groupname);
        if (g_group==NULL) {
            d_log("ERROR! Could not get gid for group %s\n",groupname);
            return -1;
        }
        if (setgid(g_group->gr_gid)!=0) {
            d_log("WARNING! Could not change to group %s(%i): %s\n",groupname,g_group->gr_gid,strerror(errno));
        } else
            d_log("Changed to group: %s(%i)\n",groupname,g_group->gr_gid);
    }
    u_pwd=getpwnam(user);
    if (u_pwd==NULL) {
        d_log("ERROR! Could not get uid for user %s\n",user);
        return -1;
    }
    if (setuid(u_pwd->pw_uid)!=0) {
        d_log("WARNING! Could not change to user %s(%i): %s\n",user,u_pwd->pw_uid,strerror(errno));
    } else
        d_log("Changed to user: %s(%i)\n",user,u_pwd->pw_uid);
	
	return 0;
}


/*  
	Ok, mutex is being implemented. I assume that current_path is the only thing needing protection thus i ignore
	all access to i.e debug.
*/
void* client_thread(void *arg)
{
	char *line = (char*)malloc(MAX_MSG_LENGTH);
	char *msg = (char*)malloc(MAX_MSG_LENGTH);
	char *cmd = (char*)malloc(MAX_MSG_LENGTH);
	char *linetmp = (char*)malloc(MAX_MSG_LENGTH);
	char *CMD;
	char *PAYLOAD;
	int payloadSize;
	int ret;
	FILE *file = NULL;
	int newSd = *((int *)arg);
   int linelength;
	
	pthread_detach(pthread_self());
	
	/* init line */
	memset(line,0x0,MAX_MSG_LENGTH);
	/* receive commands */
	while(read(newSd,line,MAX_MSG_LENGTH) > 0) {
		/* strip all unwanted characters */
		linelength = strlen( line ) - 1;
		while( line[linelength] == '\n' || line[linelength] == '\r' )	 
			--linelength;
		line[linelength+1] = 0;
      
		pthread_mutex_lock(&data_mutex); // Lock struct so no other thread can access it.
		if (config.debug_lvl >= 13)	 
			d_log("got line: %s\n", line);
		if (config.debug_lvl >= 12)	 
			d_log("calling check_command with path: %s\n", config.current_path);

		/* Check what do do with the line */
		ret = check_command(line,&file,&CMD,&PAYLOAD,&payloadSize, &config);
		pthread_mutex_unlock(&data_mutex);

		/* There is a CMD */
		if (ret >= 1) {
			pad_right( CMD, msg, ' ', strlen(CMD) );
			write(newSd,msg,strlen(msg));
			pthread_mutex_lock(&data_mutex);
			if (config.debug_lvl >= 1)	
				d_log("%i -> %s\n",strlen(msg),msg);
			pthread_mutex_unlock(&data_mutex);
			free(CMD);
		}

		/* There is a payload too */
		if (ret == 2) {
			write(newSd,PAYLOAD,payloadSize);
			free(PAYLOAD);
		}
		memset(line,0x0,MAX_MSG_LENGTH);
	} /* while(read) */
		  
	close(newSd);
	file = NULL;
	free(linetmp);
	free(line);
	free(msg);
	free(cmd);
	pthread_mutex_lock(&data_mutex);
	if (config.debug_lvl >= 1)
		    d_log("Connection closed\n");	
	pthread_mutex_unlock(&data_mutex);

	return (NULL); 
}



int main (int argc, char *argv[]) {
  int newSd;
  int cliLen;
  int flag;
  int opt;
  int nbyt;
  int running = 1;
  pthread_t thread_id;
  char parameter[16]="";
  struct sockaddr_in cliAddr, servAddr;
  FILE *pid_file;

  /* Need to set this up early. */
  pthread_mutex_init(&data_mutex, NULL); // initialize the mutex with default attributes
  
  config.root_dir = strdup( MEDIA_FILES_PATH );
  config.current_path = strdup( MEDIA_FILES_PATH );
  
  config.config_file = strdup( CONFIG );
  
  config.pid_file = strdup( PIDFILE );
  
  config.local_ip.s_addr = INADDR_ANY;
  config.user=NULL;
  config.group=NULL;
  config.log_file=NULL;
/* lets init this with NULL to get some clean code ;) */
  //  config.current_path = strdup("");
  config.current_path = NULL;
  
  config.first_user=getuid();
  config.debug_lvl=0;
  config.port=C_SERVER_PORT;
  config.remote_ip = 0;
  config.sd=0;
  config.running=1;
  config.foreground=0;
	  
  while ((opt = getopt(argc, argv, "i:d:u:g:p:D:c:l:P:r:hf")) != EOF)
      switch(opt) {
        case 'c':
			free(config.config_file);
			config.config_file=(char *)malloc(strlen(optarg)+1);
			strcpy(config.config_file,optarg);
			break;
        case 'i':
			strcat(parameter,"i");
            config.local_ip.s_addr=inet_addr(optarg);
            break;
        case 'd':
			strcat(parameter,"d");
			free(config.root_dir);
			config.root_dir = strdup( optarg );
            break;
        case 'u':
			strcat(parameter,"u");
			config.user = strdup( optarg );
            break;
        case 'g':
			strcat(parameter,"g");
			config.group = strdup( optarg );
            break;
        case 'p':
			strcat(parameter,"p");
            config.port=atoi(optarg);
            break;
        case 'P':
            strcat(parameter,"P");
			free(config.pid_file);
			config.pid_file = strdup( optarg );
            break;
        case 'D':
			strcat(parameter,"D");
            config.debug_lvl=atoi(optarg);
            break;
        case 'h':
            usage(argv[0], &config);
            break;
        case 'l':
			config.log_file = strdup( optarg );
            break;
		case 'r':
			config.remote_ip = inet_addr(optarg);
			strcat(parameter,"r");
			break;
		case 'f':
			config.foreground=1;
			break;
        case '?':
            printf("Unknown option: %s\n", optarg);
            usage(argv[0], &config);
            break;
        default:
            usage(argv[0], &config);
            break;
	  }

  read_config(config.config_file,parameter, &config);
  
  nbyt = 0;
  /* lets fork into background */
  if (config.foreground<1) {
	if ((nbyt = fork()) == -1) {
	      d_log("could not fork into background");
		  free_mainconfig(&config);
		  return -1;
	  }
	  if (nbyt != 0) {
		  printf("forking into background (%i).\n",nbyt);
		  free_mainconfig(&config);
	      return 0;
	  }
	  setsid();
  }

  /* save PID into file */
  if (!(pid_file = fopen(config.pid_file, "w"))) {
     fprintf(stderr, "cannot create pidfile %s: %s\n", config.pid_file, strerror(errno));
  } else {
    fprintf(pid_file,"%i",getpid());
    fclose(pid_file);
  }

  /* set up some signal handlers */
  signal(SIGINT,sig_int);
  signal(SIGTERM,sig_int);
  signal(SIGHUP,sig_hup);
  
  pthread_mutex_lock(&data_mutex);
 /* running as root or got option by cmd line, try to change user */
  if (config.user!=NULL && (getuid() == 0 || strchr(parameter,'u')!=NULL)) 
		if (switch_user(config.user,config.group)<0) return -1;
  

  d_log("Address: %s\n",inet_ntoa((struct in_addr)config.local_ip));
  d_log("Port: %u\n", config.port);
  d_log("Directory: %s\n", config.root_dir);

  /* create socket */
  config.sd = socket(AF_INET, SOCK_STREAM, 0);
  if (config.sd < 0) {
      perror("cannot open socket ");
	  free_mainconfig(&config);
      return -1;
  }
  
  /* Does not seem to work on linux 2.4.18+ */
  /* at least SO_REUSEADDR does work fine with 2.4.19+ */
  flag = 1;
  setsockopt (config.sd, SOL_SOCKET, SO_REUSEADDR, (char *) &flag, sizeof(int));
  setsockopt (config.sd, IPPROTO_TCP, TCP_NODELAY, (char *) &flag, sizeof(int));

  /* bind server port */
  servAddr.sin_family = AF_INET;
  servAddr.sin_addr.s_addr = config.local_ip.s_addr;
  servAddr.sin_port = htons(config.port);

  if (bind(config.sd, (struct sockaddr *) &servAddr, sizeof(servAddr)) < 0) {
      perror("cannot bind port ");
	  free_mainconfig(&config);
      return -1;
  }
  listen(config.sd,5);
  
  if (config.debug_lvl >= 1) d_log("%s: waiting for commands on port TCP %u\n",argv[0],config.port);
  /* ok startup done, lets wait for our work */ 

  cliLen = sizeof(cliAddr);

  pthread_mutex_unlock(&data_mutex);
  
  while (running) {
	  int sd; 
      /* Create the client socket */
	  
	  pthread_mutex_lock(&data_mutex);
	  sd = config.sd;
	  pthread_mutex_unlock(&data_mutex);

      newSd = accept(sd, (struct sockaddr *) &cliAddr, &cliLen);

	  /* check remote ip */
	  pthread_mutex_lock(&data_mutex);
	  
	  running = config.running;

	  if(bcmp(&config.remote_ip, &cliAddr.sin_addr, sizeof(in_addr_t)) && config.remote_ip) {
		  if(config.debug_lvl >= 1)
			  d_log("Denied remote host: %s\n", inet_ntoa(cliAddr.sin_addr));
		  if(newSd > 0)
			  close(newSd);
		  newSd=-1;
	  }
	  pthread_mutex_unlock(&data_mutex);
		  
      if (newSd <= 0) {
		  pthread_mutex_lock(&data_mutex);
		  if (config.debug_lvl>=1) d_log("no connection found, prolly shutting down....\n");
		  pthread_mutex_unlock(&data_mutex);
		  continue;
	  }

      flag = 1;
      setsockopt (newSd, IPPROTO_TCP, TCP_NODELAY, (char *) &flag, sizeof(int));
      flag = 2048;
      setsockopt (newSd, IPPROTO_TCP, TCP_MAXSEG, (char *) &flag, sizeof(int));

	  if (newSd > 0) {
	      if(pthread_create(&thread_id, NULL, client_thread, &newSd)){
		      d_log("Error creating thread, this is fatal\n");
		  }
	  }
  } /* while (RUNNING) */	  

  d_log("closing xbms\n");
  /*switching back to 1st user*/

  pthread_mutex_lock(&data_mutex);
  if (getuid()!=config.first_user) setuid(config.first_user);
  unlink(config.pid_file);
  free_mainconfig(&config);
  
  pthread_mutex_unlock(&data_mutex);
  
  return 0;
}
