#pragma once
#pragma warning(disable:4786)

#include <string>
#include <stdarg.h>
#include <stdlib.h>
#include <set>

/////////////////////////////////////////////////////////////////////////////////////////
//
template<typename CT>
class _CStringEx : public std::basic_string<CT>
{
	public:
		_CStringEx()
			{}
		_CStringEx( const std::basic_string<CT>& str ) : std::basic_string<CT>( str )
			{}
		_CStringEx( const _CStringEx<CT>& str ) :	std::basic_string<CT>( str )
			{}

		virtual ~_CStringEx()
			{}

		_CStringEx<CT>& operator=( const _CStringEx& str )
			{ assign( (std::basic_string<CT>&)str ); return *this; }

		_CStringEx<CT>& operator=( const std::string& str )
			{ assign( str ); return *this; }

		CT* GetBuffer( int nMinLen=-1 )
		{
			if ( static_cast<int>( size() ) < nMinLen )
				resize( static_cast< std::basic_string<CT>::size_type >( nMinLen ) );

			return empty() ? const_cast< CT* >( data() ) : &( at( 0 ) );
		}

		CT* SetBuffer( int nLen )
		{
			nLen = ( nLen > 0 ? nLen : 0 );
			if ( capacity() < 1 && nLen == 0 )
				resize(1);

			resize( static_cast< std::basic_string<CT>::size_type >( nLen ) );
			return const_cast< CT* >( data() );
		}

		void ReleaseBuffer( int nNewLen=-1 )
		{
			resize( static_cast< std::basic_string<CT>::size_type >( nNewLen > -1 ? nNewLen : sslen( c_str() ) ) );
		}

		_CStringEx<CT>& toUpper( void )
		{
			if ( !empty() )
				_strupr( &at( 0 ) );

			return *this;
		}

		_CStringEx<CT>& toLower( void )
		{
			if ( !empty() )
				_strlwr( &at( 0 ) );

			return *this;
		}

		int CompareNoCase( const char* pszThat ) const
		{
			return ssicmp( c_str(), pszThat );
		}

		int CompareNoCase( const _CStringEx<CT>& sThat ) const
		{
			return ssicmp( c_str(), sThat.c_str() );
		}

		bool EqualsNoCase( const char* pszThat ) const
		{
			return ( ssicmp( c_str(), pszThat ) == 0 );
		}

		bool EqualsNoCase( const _CStringEx<CT>& sThat ) const
		{
			return ( ssicmp( c_str(), sThat.c_str() ) == 0 );
		}

		bool Contains( const char* pszThat ) const
		{
			return ( find( pszThat ) != npos );
		}

		bool Contains( const _CStringEx<CT>& sThat ) const
		{
			return ( find( sThat ) != npos );
		}

		/////////////////////////////////////////////////////////////////////////////////////
		//
		//	Implementation of a case insensitive search for a substring.  Returns the index
		//	of the first occurence of the substring, or npos if not found.
		//
		int FindNoCase( const char* pszThat ) const
		{
			std::locale loc;
			const std::ctype<CT>& ct = std::_USE(loc, std::ctype<CT>);
			char* cp = (char*)c_str();
			char* s1;
			char* s2;

			if ( !*pszThat )
				return 0;

			while ( *cp )
			{
				s1 = cp;
				s2 = (char*)pszThat;

				while ( *s1 && *s2 && !( ct.tolower( *s1 ) - ct.tolower( *s2 ) ) )
					s1++, s2++;

				if ( !*s2 )
					return ( cp - (char*)c_str() );

				cp++;
			}

			return npos;
		}

		int FindNoCase( const _CStringEx<CT>& sThat ) const
		{
			return FindNoCase( sThat.c_str() );
		}

		bool ContainsNoCase( const char* pszThat ) const
		{
			return ( FindNoCase( pszThat ) != npos );
		}

		bool ContainsNoCase( const _CStringEx<CT>& sThat ) const
		{
			return ( FindNoCase( sThat.c_str() ) != npos );
		}

		bool StartsWith( const char* pszThat ) const
		{
			return ( find( pszThat ) == 0 );
		}

		bool StartsWith( const _CStringEx<CT>& sThat ) const
		{
			return ( find( sThat ) == 0 );
		}

		bool StartsWithNoCase( const char* pszThat ) const
		{
			return ( FindNoCase( pszThat ) == 0 );
		}

		bool StartsWithNoCase( const _CStringEx<CT>& sThat ) const
		{
			return ( FindNoCase( sThat.c_str() ) == 0 );
		}

		int atoi( void ) const 
		{
			if ( !empty() )
				return ::atoi( &at( 0 ) );

			return 0;
		}

		int atof( void ) const
		{
			if ( !empty() )
				return ::atof( &at( 0 ) );

			return 0;
		}

		_CStringEx<CT>& itoa( int iValue, int iRadix )
		{
			const int iMaxLength = 100;
			::itoa( iValue, GetBuffer( iMaxLength ), iRadix );
			ReleaseBuffer();
			return *this;
		}

		void Format( const char* pszFormat, ... )
		{
			va_list argList;
			va_start( argList, pszFormat );
			Format( pszFormat, argList );
			va_end( argList );
		}

		void Format( const char* pszFormat, va_list argList )
		{
			const int STD_BUF_SIZE = 1024;
			int nLen = sslen( pszFormat ) + STD_BUF_SIZE;
			_vsnprintf( GetBuffer( nLen ), nLen, pszFormat, argList );
			ReleaseBuffer();
		}

		_CStringEx<CT>& Trim( char cTrim = ' ' )
		{
			return TrimLeft( cTrim ).TrimRight( cTrim );
		}

		_CStringEx<CT>& TrimLeft( char cTrim = ' ' )
		{
			erase( 0, find_first_not_of( cTrim ) );
			return *this;
		}

		_CStringEx& TrimRight( char cTrim = ' ' )
		{
			std::basic_string<CT>::size_type nIdx = find_last_not_of( cTrim );
			erase( npos == nIdx ? 0 : ++nIdx );
			return *this;
		}

		bool GetToken
		( 
			std::basic_string<CT>::size_type&  pos,
			const char  delim,
			_CStringEx<CT>&  sToken
		)
			const
		{
			std::basic_string<CT>::size_type paramLength, startPos(pos);
			if( pos == std::basic_string<CT>::npos || size() == 0 )
				return false;
			pos = find(delim, pos);
			if (pos == std::basic_string<CT>::npos)
				paramLength = std::basic_string<CT>::npos;
			else
			{
				paramLength = pos - startPos;
				pos++;
			}
			sToken = substr(startPos, paramLength);
			return true;
		}

		
		// Checks for Integer or Decimal Number.
		bool isNumber( bool a_AllowDecimal = true, bool a_AllowNegative = true ) const
		{
			bool bAllowDecimal( a_AllowDecimal );

			if( empty() )
				return false;

			for( int i=0 ; i < size() ; i++ )
			{
				if( isdigit( at(i) ) == 0 )
				{
					if( at(i) == '.' && bAllowDecimal )
						bAllowDecimal = false;
					else if( ( i != 0 ) || !a_AllowNegative || ( at(i) != '-' ) ) 
						return false;
				}
			}
			return true;
		}

		/////////////////////////////////////////////////////////////////////////////////////
		//
		//		Split the string into two parts around a delimiter.  Nesting of parameters
		//		is allowed using escape characters.
		//
		bool SplitList
		(
			std::set< std::string >& tokenSet,
			const char cDelim,
			const char cNestStart = '{',
			const char cNestEnd = '}'
		)
			const
		{
			// "key={key2=val2,key3=val3},key4=val4,key5=val5" becomes
			//		key={key2=val2,key3=val3}
			//		key4=val4
			//		key5=val5

			_CStringEx<CT> sToken;
			std::basic_string<CT>::size_type pos = 0;

			while ( SplitList( cDelim, cNestStart, cNestEnd, pos, sToken ) )
				tokenSet.insert( sToken );

			return ( tokenSet.size() != 0 );
		}

		/////////////////////////////////////////////////////////////////////////////////////
		//
		//		Split the string into two parts around a delimiter.  Nesting of parameters
		//		is allowed using escape characters.
		//
		bool SplitList
		(
			const char cDelim,
			const char cNestStart,
			const char cNestEnd,
			std::basic_string<CT>::size_type& pos,
			_CStringEx<CT>& sToken
		)
			const
		{
			// "key={key2=val2,key3=val3},key4=val4,key5=val5" becomes
			//		key={key2=val2,key3=val3}
			//		key4=val4
			//		key5=val5

			int iNest = 0;
			char c;
			std::basic_string<CT>::size_type posStart( pos ); 

			while ( pos < size() )
			{
				c = at( pos );

				if ( c == cNestStart )
					iNest++;
				else if ( c == cNestEnd )
					iNest--;
				else if ( iNest == 0 && c == cDelim )
				{
					sToken = substr( posStart, pos - posStart );

					// Skip over the delim
					pos++;
					return true;
				}

				pos++;
			}

			sToken = substr( posStart, pos - posStart );
			return !sToken.empty();
		}

		/////////////////////////////////////////////////////////////////////////////////////
		//
		bool SplitPair
		(
			const char cDelim,
			const char cNestStart,
			const char cNestEnd,
			_CStringEx<CT>& sKey,
			_CStringEx<CT>& sValue
		)
			const
		{
			// "key={key2=val2,key3=val3}" becomes
			//		key, key2=val2,key3=val3

			if ( size() == 0 )
				return false;

			std::basic_string<CT>::size_type pos( 0 ), posStart( 0 ), posEnd( size()-1 );
			int iNest = 0;
			char c;

			while ( pos <= posEnd )
			{
				c = at( pos );

				if ( c == cNestStart )
					iNest++;
				else if ( c == cNestEnd )
					iNest--;

				if ( iNest == 0 )
				{
					if ( c == cDelim )
					{
						sKey = substr( posStart, pos - posStart );
						sValue = substr( pos+1 );
						sValue.TrimLeft( cNestStart );
						sValue.TrimRight( cNestEnd );
						break;
					}
				}

				pos++;
			}

			return true;
		}

		int Replace( const char* pszOld, const char* pszNew )
		{
			int nReplaced		= 0;
			int nIdx			= 0;
			static const CT _C	= CT(0);
			int nOldLen			= sslen( pszOld );
			int nNewLen			= sslen( pszNew );
			const char* pszRealNew	= pszNew == NULL ? &_C : pszNew;
			const char* pszRealOld	= pszOld == NULL ? &_C : pszOld;

			while ( ( nIdx = find( pszRealOld, nIdx ) ) != npos )
			{
				replace( begin() + nIdx, begin() + nIdx + nOldLen, pszRealNew );
				nReplaced++;
				nIdx += nNewLen;
			}

			return nReplaced;
		}
};

template<typename CT> inline int sslen( const CT* pT )
{
	return NULL == pT ? 0 : std::char_traits<CT>::length( pT );
}

template<typename CT> inline int ssicmp( const CT* pA1, const CT* pA2 )
{
	std::locale loc;
	const std::ctype<CT>& ct = std::_USE(loc, std::ctype<CT>);
	CT f;
	CT l;

        do 
		{
			f = ct.tolower(*(pA1++));
			l = ct.tolower(*(pA2++));
        } while ( (f) && (f == l) );

    return (int)(f - l);
}

typedef _CStringEx< char > CStringEx;
