#include "Stdafx.h"
#include "CResourcePool.h"
#include "CTrace.h"

/////////////////////////////////////////////////////////////////////////////////////////////
//
//
//

CResourcePool::CResourcePool
(
	CResourceFactory* pResourceFactory, 
	int iResourcePoolSize
)
	:
	m_pResourceFactory( pResourceFactory ),
	m_iResourcePoolSize( iResourcePoolSize ),
	m_iResourcesAllocated( 0 ),
	m_AvailableEvent( true, true )
{
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//
//

CResourcePool::~CResourcePool()
{
	TRACE( "CResourcePool: Destructing\n" );

	// Wait for any outstanding resources to return.
	while ( m_IdleResourceMap.size() != m_iResourcesAllocated )
	{
		TRACE
		( 
			"CResourcePool: Waiting for %u of %u resources to return\n", 
			m_iResourcesAllocated - m_IdleResourceMap.size(),
			m_iResourcesAllocated			
		);

		// Clear the availability event and wait for the threads to return
		m_AvailableEvent.Clear();
		Wait();
	}

	// Now free the resources
	CPoolable* pResource;
	CLock myLock( m_CriticalSection );
	RESOURCEMAP::iterator it = m_IdleResourceMap.begin();

	while ( it != m_IdleResourceMap.end() )
	{
		pResource = (*it).second;

		TRACE
		(
			"CResourcePool: Detaching resource %u\n", 
			pResource->GetId()
		);

		pResource->DetachFromPool();
		it++;
	}

	m_IdleResourceMap.clear();
	TRACE( "CResourcePool: Destructed\n" );
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//
//

bool CResourcePool::Acquire
( 
	CPoolable*& pResource 
)
{
	CLock myLock( m_CriticalSection );
	bool bResult = false;

	if ( IsAvailable() )
	{
		if ( m_IdleResourceMap.empty() )
			bResult = CreateResource( pResource );
		else
		{
			RESOURCEMAP::iterator it = m_IdleResourceMap.begin();
			pResource = (*it).second;
			m_IdleResourceMap.erase( it );
			bResult = true;
		}
	}

	// Update the availability state
	UpdateAvailability();

	if ( bResult )
	{
		TRACE
		(
			"CResourcePool: Acquired resource %u, %u available, %u allocated, %u max\n", 
			pResource->GetId(),
			m_IdleResourceMap.size(),
			m_iResourcesAllocated,
			m_iResourcePoolSize
		);
	}
	else
	{
		TRACE
		(
			"CResourcePool: Failed to acquired a resource, %u available, %u allocated, %u max\n", 
			m_IdleResourceMap.size(),
			m_iResourcesAllocated,
			m_iResourcePoolSize
		);
	}

	return bResult;
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//		Put a resource back in the idle thread map.
//

bool CResourcePool::Relinquish
( 
	CPoolable* pResource 
)
{
	CLock myLock( m_CriticalSection );

	m_IdleResourceMap[ pResource->GetId() ] = pResource;

	UpdateAvailability();

	TRACE
	(
		"CResourcePool: Relinquished resource %u, %u available, %u allocated, %u max\n", 
		pResource->GetId(),
		m_IdleResourceMap.size(),
		m_iResourcesAllocated,
		m_iResourcePoolSize
	);

	return true;
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//
//

bool CResourcePool::CreateResource
( 
	CPoolable*& pResource 
)
{
	if ( m_pResourceFactory->CreateInstance( pResource ) )
	{
		pResource->AttachToPool( this );
		m_iResourcesAllocated++;

		TRACE
		(
			"CResourcePool: Created resource %u, %u allocated, %u max\n", 
			pResource->GetId(),
			m_iResourcesAllocated,
			m_iResourcePoolSize
		);

		return true;
	}

	return false;
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//		Signal\Clear the event the indicates that there is at least on resource available.
//

void CResourcePool::UpdateAvailability( void )
{
	if ( IsAvailable() )
		m_AvailableEvent.Signal();
	else
		m_AvailableEvent.Clear();
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
bool CResourcePool::IsAvailable( void )
{
	bool bAvailable = !m_IdleResourceMap.empty() || 
					  ( m_iResourcesAllocated < m_iResourcePoolSize );

	return  bAvailable;
}

