#include "Stdafx.h"
#include "CWaitable.h"

/////////////////////////////////////////////////////////////////////////////////////////////
//
//		Construct an instance of an CWaitable object.
//

CWaitable::CWaitable()
{
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//		Destruct an instance of an CWaitable object.
//

CWaitable::~CWaitable()
{
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//		The Wait function checks the current state of the specified object.  If the object's 
//		state is nonsignaled, the calling thread enters an efficient wait state. The thread 
//		consumes very little processor time while waiting for the object state to become 
//		signaled or the time-out interval to elapse.
//
//		If the function succeeds, the return value indicates the event that caused the 
//		function to return. This value can be one of the following:
//
//		Value				Meaning 
//		WAIT_ABANDONED		The specified object is a mutex object that was not released by 
//							the thread that owned the mutex object before the owning thread 
//							terminated. Ownership of the mutex object is granted to the 
//							calling thread, and the mutex is set to nonsignaled. 
//		WAIT_OBJECT_0		The state of the specified object is signaled. 
//		WAIT_TIMEOUT		The time-out interval elapsed, and the object's state is 
//							nonsignaled. 
//
//		If the function fails, the return value is WAIT_FAILED. To get extended error 
//		information, call GetLastError. 

DWORD CWaitable::Wait
( 
	DWORD dwMilliseconds		// The time to wait (in milliseconds) before abandoning the
								// Wait.
)
{
	return ::WaitForSingleObject( GetWaitHandle(), dwMilliseconds ); 
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//		Construct an instance of an CWaitableCollection object.
//

CWaitableCollection::CWaitableCollection
( 
	int iMaxSize			// The size of the collection.  This size cannot be changed, and
							// each entry must be initialized with a valid CWaitable 
							// instance using the Add member.
)
	:
	m_iMaxSize( iMaxSize ),
	m_iSize( 0 )
{
	m_pHandles = new HANDLE[ m_iMaxSize ]; 
	m_pObjects = new CWaitable*[ m_iMaxSize ];
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//		Destruct an instance of an CWaitableCollection object.  Does not destruct the 
//		contained CWaitable instances.
//

CWaitableCollection::~CWaitableCollection()
{ 
	delete[] m_pHandles; 
	delete[] m_pObjects; 
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//		The Wait function returns when one of the following occurs: 
//
//			1. Either any one or all of the specified objects are in the signaled state. 
//			2. The time-out interval elapses. 
//
//		If the function succeeds, the return value indicates the event that caused the 
//		function to return. This value can be one of the following. 
//
//		Value							Meaning 
//		WAIT_OBJECT_0 to 
//		(WAIT_OBJECT_0 + iSize  1)		If bWaitAll is TRUE, the return value indicates that 
//										the state of all specified objects is signaled. If 
//										bWaitAll is FALSE, the return value minus WAIT_OBJECT_0 
//										indicates the index of the object that satisfied the 
//										wait. If more than one object became signalled during 
//										the call, this is the array index of the signalled 
//										object with the smallest index value of all the 
//										signalled objects.
//		WAIT_ABANDONED_0 to 
//		(WAIT_ABANDONED_0 + iSize  1)	If bWaitAll is TRUE, the return value indicates that 
//										the state of all specified objects is signaled and at 
//										least one of the objects is an abandoned mutex object. 
//										If bWaitAll is FALSE, the return value minus 
//										WAIT_ABANDONED_0 indicates the index of an abandoned 
//										mutex object that satisfied the wait.
//
//		WAIT_TIMEOUT					The time-out interval elapsed and the conditions 
//										specified by the bWaitAll parameter are not satisfied. 
//
//		If the function fails, the return value is WAIT_FAILED. To get extended error 
//		information, call GetLastError. 
//

CWaitable* CWaitableCollection::Wait
( 
	DWORD dwMilliseconds, 		// The time to wait (in milliseconds) before abandoning the
								// Wait.
	bool bWaitAll				// Specifies the wait type. If true, the function returns 
								// when the state all objects in the collection are signaled. 
								// If false, the function returns when the state of any one 
								// of the objects set to is signaled. In the latter case, 
								// the return value indicates the object whose state caused 
								// the function to return. 
)
{ 
	DWORD dwResult = ::WaitForMultipleObjects( m_iSize, m_pHandles, bWaitAll, dwMilliseconds ); 

	if ( dwResult == WAIT_FAILED )
		return false;

	if ( bWaitAll )
		return m_pObjects[0];

	return m_pObjects[ dwResult ];
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//		Add a CWaitable instance to this collection.
//
bool CWaitableCollection::Add
( 
	CWaitable* pWaitable		// Pointer to the instance to add.
)
{ 
	if ( ( m_iSize ) < m_iMaxSize )
	{
		m_pObjects[ m_iSize ] = pWaitable; 
		m_pHandles[ m_iSize ] = pWaitable->GetWaitHandle();

		m_iSize++;

		return true;
	}

	return false;
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//		Causes each of the CWaitables in the collection to CancelWait, effectively 
//		terminating	any current waits.
//

void CWaitableCollection::Cancel( void )
{
	for ( int i=0; i < m_iSize; i++ )
		m_pObjects[ i ]->CancelWait();
}
