#include "stdafx.h"
#include "CRunnable.h"
#include "CThread.h"

/////////////////////////////////////////////////////////////////////////////////////////////
//
//		Construct an CRunnable object instance.
//

CRunnable::CRunnable
( 
	bool bAutoDelete 
)
	:
	m_bRun( false ),
	m_bPaused( false ),
	m_pThread( NULL ),
	m_bAutoDelete( bAutoDelete ),
	m_StoppedEvent( true, true ),
	m_RunningEvent( true, false )
{
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//		Destruct an CRunnable object instance.
//

CRunnable::~CRunnable()
{
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//	Run this object in the context of the current stack\thread.
//
void CRunnable::Invoke( void )
{
	m_bRun = true;
	Run();
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//		Run this instance.  If a thread is supplied it will be used.  If no thread is 
//		supplied a new thread will be constructed.  Returns true successfull, false 
//		otherwise.
//

bool CRunnable::Start
( 
	CThread* pThread		// Thread in which to run, or NULL to create a new thread.
)
{
	bool bResult = false;
	{
		CLock myLock( m_ShutDownSection );

		if ( m_pThread && pThread )
			return false;
		else if ( !m_pThread && pThread )
			m_pThread = pThread;
		else if ( !m_pThread && !pThread )
			m_pThread = new CThread();

		m_bRun = true;

		bResult = m_pThread->Run( this );

		if ( m_bAutoDelete )
			return bResult;
	}

	// Now wait until we are really running
	if ( bResult )
		m_RunningEvent.Wait();

	return bResult;		
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//		Pause the thread in which this object is currently running.  Returns true if 
//		successfull, false otherwise.
//

bool CRunnable::Pause( void )
{
	{
		CLock myLock( m_ShutDownSection );
	
		if ( m_pThread && !m_bPaused )
		{
			m_pThread->Suspend();
			m_bPaused = true;

			return true;
		}
	}

	return false;
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//		Resume the thread in which this object is currently running.  Returns true if
//		successfull, false otherwise.
//

bool CRunnable::Resume( void )
{
	{
		CLock myLock( m_ShutDownSection );

		if ( m_pThread && m_bPaused )
		{
			m_pThread->Resume();
			m_bPaused = false;

			return true;
		}
	}

	return false;
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//		Stop the execution of the thread in which this instance is running.  Optionally
//		wait untill the thread exits and the thread object is destroyed.  If the thread is
//		currently paused it will be resumed to allow it to exit normally.
//

bool CRunnable::Stop( void )
{
	if ( m_pThread )
	{
		{
			CLock myLock( m_ShutDownSection );

			// Set the run flag to false.
			m_bRun = false;

			// If we are paused allow the thread to start running again
			if ( m_bPaused )
				Resume();

			// Call derived classes and let them know it's time to stop.  This lets them stop
			// any waits that are active on the thread.
			HandleStop();

			// If this runnable autodeletes when it stops we need to get out now!
			if ( m_bAutoDelete )
				return true;
		}
		
		m_StoppedEvent.Wait();
		return true;
	}

	return false;
}

/////////////////////////////////////////////////////////////////////////////////////////////
//
//		Run this instance, calling the following virtual functions in order:
//
//			HandlePreRun()
//			HandleRun()
//			HandlePostRun()
//

void CRunnable::Run( void )
{
	bool bAutoDelete = m_bAutoDelete;

	if ( HandlePreRun() )
	{
		{
			// Signal that we are running
			CLock myLock( m_ShutDownSection );
			m_RunningEvent.Signal();
			m_StoppedEvent.Clear();
		}

		HandleRun();
		HandlePostRun();

		{
			// Wait till Stop() exits.  Note that we do this inside a scope to insure that
			// The critical section is exited before this runnable is destroyed.  If we
			// don't use this scope the critical section is deleted in the destructor, and
			// then exited as we exit the function scope.  This is a very bad thing and will
			// completely wack the current thread when used in OS synchronization objects.
			CLock myLock( m_ShutDownSection );

			// We are stopped!
			m_StoppedEvent.Signal();
			m_RunningEvent.Clear();
			m_pThread = NULL;
		}
	}

	// Should we automatically delete ourselves?
	if ( bAutoDelete )
		delete this;
}

 