// FuelGuage.cpp: implementation of the CFuelGuage class.
//
//////////////////////////////////////////////////////////////////////
//#include <d3dx9.h>
//#include "DXUtil.h"
#include <xtl.h>
#include "port.h"

#include "DxObjects.h"

#include ".\FuelGuage.h"
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
LPDIRECT3DTEXTURE8			CFuelGuage::m_pTexture = NULL;
//-----------------------------------------------------------------------------

CFuelGuage::CFuelGuage()
{
	// Const Pixel offsets.
	m_nLeftPixelsWords		= 36;
	m_nRightPixelsWords		= m_nLeftPixelsWords + 51;
	m_nHeightPixelsWords	= 26;

	m_nVertGapPixels		= 26;

	m_nLeftPixelsFuel		= 47;
	m_nRightPixelsFuel		= m_nLeftPixelsFuel + 18;


	// Variable Pixel offsets. These will be re-assigned by RestoreDeviceObjects()
	m_nBottomPixelsWords	= 0;
	m_nTopPixelsWords		= 0;
	m_nBottomPixelsFuel		= 0;
	m_nTopPixelsFuel		= 0;


	// Constant texture offsets
	m_fTopTexCoordWords		= 0.0f;
	m_fBottomTexCoordWords	= 0.5f;
	m_fLeftTexCoordWords	= 0.0f;
	m_fRightTexCoordWords	= 1.0f;

	m_fTopTexCoordFuel		= 0.55f;
	m_fBottomTexCoordFuel	= 0.95f;
	m_fLeftTexCoordFuel		= 0.0f;
	m_fRightTexCoordFuel	= 1.0f;


	m_pVB		= NULL;
	m_dwFVF		= D3DFVF_XYZRHW|D3DFVF_TEX1;

	m_bRender	= false;
}

//-----------------------------------------------------------------------------

CFuelGuage::~CFuelGuage()
{
}

//-----------------------------------------------------------------------------

HRESULT CFuelGuage::InitDeviceObjects()
{
	if(!m_pTexture)
	{
		HRESULT hr;
		LPDIRECT3DDEVICE8 pDevice = CDxObjects::GetDevice();

		// Load the texture. Define the alpha channel
		if((hr = D3DXCreateTextureFromFileEx(pDevice, "D:\\fuelguage2.bmp", D3DX_DEFAULT,
						D3DX_DEFAULT, D3DX_DEFAULT, 0, D3DFMT_A8R8G8B8, D3DPOOL_MANAGED,
						D3DX_DEFAULT, D3DX_DEFAULT, 0, NULL, NULL, &m_pTexture)) != S_OK)
		{
//			MessageBox(NULL, "Could not load fuel guage texture map", "FlyPhil.exe", MB_OK);
			return(hr);
		}
		
		D3DSURFACE_DESC pDDS;
		if((hr=m_pTexture->GetLevelDesc(0, &pDDS)) != S_OK)
		{
//			MessageBox(NULL, "Can't GetLevelDesc() fuelguage texture", "FlyPhil.exe", MB_OK);
			return hr;
		}
		int nNumPixels = pDDS.Width * pDDS.Height;

		D3DLOCKED_RECT lockedRect;
		if((hr=m_pTexture->LockRect(0, &lockedRect, NULL, 0)) != S_OK)
		{
//			MessageBox(NULL, "Can't LockRect() fuelguage texture", "FlyPhil.exe", MB_OK);
			return hr;
		}

		BYTE *ptr = (BYTE *)(lockedRect.pBits);
		BYTE r;
		for(int i=0; i < nNumPixels; ++i)
		{
			r = *(ptr + (i * 4) + 2);
			if(r == 255)
				*(ptr + (i * 4) + 3) = 0;
			else
				*(ptr + (i * 4) + 3) = 100;
		}

		m_pTexture->UnlockRect(0);
	}

	return S_OK;
}

//-----------------------------------------------------------------------------

void CFuelGuage::DeleteDeviceObjects()
{
	SAFE_RELEASE(m_pTexture);

	m_bRender = false;
}

//-----------------------------------------------------------------------------

HRESULT	CFuelGuage::RestoreDeviceObjects()
{
	HRESULT hr;
	// Create the vertex buffer.
	if((hr = CreateSimpleVertexBuffer()) != S_OK)
		return(hr);


	D3DVIEWPORT8 viewPort;
	CDxObjects::GetDevice()->GetViewport(&viewPort);

	m_nBottomPixelsWords	= viewPort.Height - m_nVertGapPixels + viewPort.Y;
	m_nTopPixelsWords		= m_nBottomPixelsWords - m_nHeightPixelsWords;
	m_nBottomPixelsFuel		= m_nBottomPixelsWords - m_nHeightPixelsWords - 4;
	m_nTopPixelsFuel		= viewPort.Y + m_nVertGapPixels;

	AssignWindowGeom();


	m_bRender = true;

	return S_OK;
}

//-----------------------------------------------------------------------------

void CFuelGuage::InvalidateDeviceObjects()
{
	SAFE_RELEASE(m_pVB);
	m_bRender = false;
}

//-----------------------------------------------------------------------------

void CFuelGuage::Render() const
{
	if(m_bRender)
	{
		LPDIRECT3DDEVICE8 pDevice = CDxObjects::GetDevice();

		pDevice->SetStreamSource(0, m_pVB, sizeof(CUSTOMVERTEX));
		pDevice->SetVertexShader(m_dwFVF);
		pDevice->SetTexture(0, m_pTexture);
		pDevice->DrawPrimitive(D3DPT_TRIANGLELIST, 0, NUM_PRIMS);
	}
}

//-----------------------------------------------------------------------------

HRESULT CFuelGuage::CreateSimpleVertexBuffer()
{
	HRESULT hr;

	LPDIRECT3DDEVICE8 pDevice = CDxObjects::GetDevice();

	if((hr = pDevice->CreateVertexBuffer(	NUM_VERTS * sizeof(CUSTOMVERTEX), D3DUSAGE_DYNAMIC,
											m_dwFVF, D3DPOOL_DEFAULT, &m_pVB)) != S_OK)
		return(hr);

	CUSTOMVERTEX* pVertices;
	if((hr = m_pVB->Lock( 0, 0, (BYTE**)&pVertices, 0 )) != S_OK)
		return(hr);

	for(int i=0; i < NUM_VERTS; ++i)
	{
		pVertices[i].position.z = 0.1f;
		pVertices[i].position.w = 1.0f;
	}

	// Top and Bottom determined by window size and fuel level
	pVertices[ 0].position.x = (float)m_nLeftPixelsWords;
	pVertices[ 1].position.x = (float)m_nLeftPixelsWords;
	pVertices[ 2].position.x = (float)m_nRightPixelsWords;
	pVertices[ 3].position.x = (float)m_nRightPixelsWords;
	pVertices[ 4].position.x = (float)m_nLeftPixelsWords;
	pVertices[ 5].position.x = (float)m_nRightPixelsWords;

	pVertices[ 6].position.x = (float)m_nLeftPixelsFuel;
	pVertices[ 7].position.x = (float)m_nLeftPixelsFuel;
	pVertices[ 8].position.x = (float)m_nRightPixelsFuel;
	pVertices[ 9].position.x = (float)m_nRightPixelsFuel;
	pVertices[10].position.x = (float)m_nLeftPixelsFuel;
	pVertices[11].position.x = (float)m_nRightPixelsFuel;


	// Words...
	pVertices[ 0].tu = m_fLeftTexCoordWords;
	pVertices[ 0].tv = m_fTopTexCoordWords;
	pVertices[ 1].tu = m_fLeftTexCoordWords;
	pVertices[ 1].tv = m_fBottomTexCoordWords;
	pVertices[ 2].tu = m_fRightTexCoordWords;
	pVertices[ 2].tv = m_fTopTexCoordWords;
	pVertices[ 3].tu = m_fRightTexCoordWords;
	pVertices[ 3].tv = m_fTopTexCoordWords;
	pVertices[ 4].tu = m_fLeftTexCoordWords;
	pVertices[ 4].tv = m_fBottomTexCoordWords;
	pVertices[ 5].tu = m_fRightTexCoordWords;
	pVertices[ 5].tv = m_fBottomTexCoordWords;

	// Fuel guage...
	pVertices[ 6].tu = m_fLeftTexCoordFuel;
	pVertices[ 6].tv = m_fTopTexCoordFuel;
	pVertices[ 7].tu = m_fLeftTexCoordFuel;
	pVertices[ 7].tv = m_fBottomTexCoordFuel;
	pVertices[ 8].tu = m_fRightTexCoordFuel;
	pVertices[ 8].tv = m_fTopTexCoordFuel;
	pVertices[ 9].tu = m_fRightTexCoordFuel;
	pVertices[ 9].tv = m_fTopTexCoordFuel;
	pVertices[10].tu = m_fLeftTexCoordFuel;
	pVertices[10].tv = m_fBottomTexCoordFuel;
	pVertices[11].tu = m_fRightTexCoordFuel;
	pVertices[11].tv = m_fBottomTexCoordFuel;

	m_pVB->Unlock();

	return S_OK;
}

//-----------------------------------------------------------------------------

void CFuelGuage::AssignWindowGeom()
{
	CUSTOMVERTEX* pVertices;
	m_pVB->Lock( 0, 0, (BYTE**)&pVertices, 0);

	pVertices[ 0].position.y = (float)m_nTopPixelsWords;
	pVertices[ 1].position.y = (float)m_nBottomPixelsWords;
	pVertices[ 2].position.y = (float)m_nTopPixelsWords;
	pVertices[ 3].position.y = (float)m_nTopPixelsWords;
	pVertices[ 4].position.y = (float)m_nBottomPixelsWords;
	pVertices[ 5].position.y = (float)m_nBottomPixelsWords;

	pVertices[ 7].position.y = (float)m_nBottomPixelsFuel;
	pVertices[10].position.y = (float)m_nBottomPixelsFuel;
	pVertices[11].position.y = (float)m_nBottomPixelsFuel;

	m_pVB->Unlock();
}

//-----------------------------------------------------------------------------

void CFuelGuage::FrameMove(float fFuel)
{
	if(m_bRender)
	{
		CUSTOMVERTEX* pVertices;
		m_pVB->Lock( 0, 0, (BYTE**)&pVertices, 0 );

		float fPosTop = ((m_nBottomPixelsFuel - m_nTopPixelsFuel) * (1.0f - fFuel)) + m_nTopPixelsFuel;
		pVertices[6].position.y = fPosTop;
		pVertices[8].position.y = fPosTop;
		pVertices[9].position.y = fPosTop;

		m_pVB->Unlock();
	}
}
