// Altimeter.cpp: implementation of the CAltimeter class.
//
//////////////////////////////////////////////////////////////////////
//#include <d3dx9.h>
//#include "DXUtil.h"
#include <xtl.h>
#include "port.h"

#include "DxObjects.h"

#include <assert.h>

#include ".\Altimeter.h"
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------

CAltimeter::CAltimeter()
{
	// Pixel offsets. These will be re-assigned by RestoreDeviceObjects()
	m_nRightPixels		= 0;
	m_nLeftPixels		= 0;
	m_nTopPixels		= 0;
	m_nBottomPixels		= 0;

	m_pVB		= NULL;
	m_dwFVF		= D3DFVF_XYZRHW | D3DFVF_DIFFUSE;

	m_bRender = false;
}

//-----------------------------------------------------------------------------

CAltimeter::~CAltimeter()
{
}

//-----------------------------------------------------------------------------

//
// Recalculate pixel coordinates. Repopulate vertex buffer with non-
// altitude dependent data.
//
HRESULT	CAltimeter::RestoreDeviceObjects()
{
	HRESULT hr;
	if((hr = CreateSimpleVertexBuffer()) != S_OK)
		return(hr);


	D3DVIEWPORT8 viewPort;
	CDxObjects::GetDevice()->GetViewport(&viewPort);

	m_nRightPixels		= viewPort.Width - 47;
	m_nLeftPixels		= m_nRightPixels - 18;
	m_nTopPixels		= viewPort.Y + 36;
	m_nBottomPixels		= viewPort.Y + viewPort.Height - 36;

	AssignWindowGeom();


	m_bRender = true;

	return(S_OK);
}

//-----------------------------------------------------------------------------

void CAltimeter::InvalidateDeviceObjects()
{
	SAFE_RELEASE(m_pVB);
	m_bRender = false;
}

//-----------------------------------------------------------------------------

void CAltimeter::Render() const
{
	if(m_bRender)
	{
		LPDIRECT3DDEVICE8 pDevice = CDxObjects::GetDevice();

		pDevice->SetTexture(0, NULL);
		pDevice->SetStreamSource(0, m_pVB, sizeof(CUSTOMVERTEX));
		pDevice->SetVertexShader(m_dwFVF);
		pDevice->DrawPrimitive(D3DPT_TRIANGLELIST, 0, NUM_PRIMS);
	}
}

//-----------------------------------------------------------------------------

//
// Create the vertex buffer. Populate constant vertex data only.
//
HRESULT CAltimeter::CreateSimpleVertexBuffer()
{
	HRESULT hr;

	if((hr = CDxObjects::GetDevice()->CreateVertexBuffer
								(	NUM_VERTS * sizeof(CUSTOMVERTEX), D3DUSAGE_DYNAMIC,
									m_dwFVF, D3DPOOL_DEFAULT, &m_pVB)) != S_OK)
		return(hr);

	CUSTOMVERTEX* pVertices;
	if((hr = m_pVB->Lock( 0, 0, (BYTE**)&pVertices, 0 )) != S_OK)
		return(hr);

	int i = 0;
	for(; i < 6; ++i)			// Sky above section
	{
		pVertices[i].position.z = 0.1f;
		pVertices[i].position.w = 1.0f;
		pVertices[i].diffuse = D3DCOLOR_RGBA(50, 100, 250, 80);
	}
	for(; i < 12; ++i)			// Sky below section
	{
		pVertices[i].position.z = 0.1f;
		pVertices[i].position.w = 1.0f;
		pVertices[i].diffuse = D3DCOLOR_RGBA(0, 0, 200, 128);
	}
	for(; i < NUM_VERTS; ++i)	// Land section
	{
		pVertices[i].position.z = 0.1f;
		pVertices[i].position.w = 1.0f;
		pVertices[i].diffuse = D3DCOLOR_RGBA(50, 150, 0, 128);
	}

	m_pVB->Unlock();

	return S_OK;
}

//-----------------------------------------------------------------------------

//
// Repopulate vertex buffer with non-altitude dependent data.
//
void CAltimeter::AssignWindowGeom()
{
	CUSTOMVERTEX* pVertices;
	m_pVB->Lock( 0, 0, (BYTE**)&pVertices, 0 );

	pVertices[ 0].position.x = (float)m_nLeftPixels;
	pVertices[ 0].position.x = (float)m_nLeftPixels;
	pVertices[ 0].position.y = (float)m_nTopPixels;
	pVertices[ 1].position.x = (float)m_nLeftPixels;
	pVertices[ 2].position.x = (float)m_nRightPixels;
	pVertices[ 2].position.y = (float)m_nTopPixels;
	pVertices[ 3].position.x = (float)m_nRightPixels;
	pVertices[ 3].position.y = (float)m_nTopPixels;
	pVertices[ 4].position.x = (float)m_nLeftPixels;
	pVertices[ 5].position.x = (float)m_nRightPixels;

	pVertices[ 6].position.x = (float)m_nLeftPixels;
	pVertices[ 7].position.x = (float)m_nLeftPixels;
	pVertices[ 8].position.x = (float)m_nRightPixels;
	pVertices[ 9].position.x = (float)m_nRightPixels;
	pVertices[10].position.x = (float)m_nLeftPixels;
	pVertices[11].position.x = (float)m_nRightPixels;

	pVertices[12].position.x = (float)m_nLeftPixels;
	pVertices[13].position.x = (float)m_nLeftPixels;
	pVertices[13].position.y = (float)m_nBottomPixels;
	pVertices[14].position.x = (float)m_nRightPixels;
	pVertices[15].position.x = (float)m_nRightPixels;
	pVertices[16].position.x = (float)m_nLeftPixels;
	pVertices[16].position.y = (float)m_nBottomPixels;
	pVertices[17].position.x = (float)m_nRightPixels;
	pVertices[17].position.y = (float)m_nBottomPixels;

	m_pVB->Unlock();
}

//-----------------------------------------------------------------------------

//
// Called every frame to repopulate vertex buffer with altitude dependent data.
//
void CAltimeter::FrameMove(float fShipAltitude, float fLandAltitude)
{
	if(m_bRender)
	{
		float fShipAlt	= m_nBottomPixels - (m_nBottomPixels - m_nTopPixels) * (fShipAltitude + 0.25f);
		float fLandAlt	= m_nBottomPixels - (m_nBottomPixels - m_nTopPixels) * (fLandAltitude + 0.25f);

		CUSTOMVERTEX* pVertices;
		m_pVB->Lock( 0, 0, (BYTE**)&pVertices, 0 );

		pVertices[ 1].position.y = fShipAlt;
		pVertices[ 4].position.y = fShipAlt;
		pVertices[ 5].position.y = fShipAlt;

		pVertices[ 6].position.y = fShipAlt;
		pVertices[ 7].position.y = fLandAlt;
		pVertices[ 8].position.y = fShipAlt;
		pVertices[ 9].position.y = fShipAlt;
		pVertices[10].position.y = fLandAlt;
		pVertices[11].position.y = fLandAlt;

		pVertices[12].position.y = fLandAlt;
		pVertices[14].position.y = fLandAlt;
		pVertices[15].position.y = fLandAlt;

		m_pVB->Unlock();
	}
}
