#include "gfx.h"
#include "error.h"
#include "file.h"
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include "main.h"
#include <math.h>

Gfx *gfxInit(int width, int height, int fullscreen, int num_rects){
  Gfx *temp_gfx;
  SDL_Surface *font_image;
  SDL_PixelFormat *format;
  SDL_Rect rect;
  
  int i;

  temp_gfx = (Gfx *) malloc( sizeof(Gfx) );
  
  temp_gfx->screen = SDL_SetVideoMode( width, height, 16,
                       SDL_SWSURFACE | ( fullscreen ? SDL_FULLSCREEN : 0 ) );

  if( !temp_gfx->screen ){
    free( temp_gfx );
    errorSet(("GFX"), SDL_GetError());
    return NULL;
  }
  
  SDL_ShowCursor( 0 );

  temp_gfx->background = NULL;
  temp_gfx->frame_delta = 0.0001;
  temp_gfx->frame_count = 0;
  temp_gfx->last_time = SDL_GetTicks();
  temp_gfx->time = 0;
  temp_gfx->sprites = NULL;
  temp_gfx->strings = NULL;
  
  temp_gfx->_dirty_rects = NULL;
  temp_gfx->_clean_rects = (Rect *) malloc( sizeof(Rect) * num_rects );
  temp_gfx->_rect_mem = temp_gfx->_clean_rects;
  
  for( i = 0; i < num_rects; i++ )
    temp_gfx->_clean_rects[i].next = &temp_gfx->_clean_rects[i+1];

  temp_gfx->_clean_rects[num_rects-1].next = NULL;
  
  temp_gfx->_sprite_types = NULL;
 
  format = temp_gfx->screen->format;
  font_image = gfxLoadImage( fileCreateName( GFX, "font2.bmp" ) );
  if( !font_image ) return NULL;

  temp_gfx->scroller = stringNew( temp_gfx, width / FONT_WIDTH );
  
  temp_gfx->scroller_rect.x = 0;
  temp_gfx->scroller_rect.y = temp_gfx->screen->h - FONT_HEIGHT - 1;
  temp_gfx->scroller_rect.w = temp_gfx->screen->w;
  temp_gfx->scroller_rect.h = FONT_HEIGHT;
  temp_gfx->scroller_status = SCROLLER_IDLE;
  gfxScrollerArgs( temp_gfx, 2000, 1000, 1500 );
  
  for( i = 0; i < FONT_GLYPHS; i++ ){
    temp_gfx->glyphs[i] = SDL_CreateRGBSurface( SDL_SWSURFACE,
                                                FONT_WIDTH,
                                                FONT_HEIGHT,
                                                format->BitsPerPixel,
                                                format->Rmask,
                                                format->Gmask,
                                                format->Bmask,
                                                format->Amask );
    rect.x = FONT_WIDTH * i;
    rect.y = 0;
    rect.w = FONT_WIDTH;
    rect.h = FONT_HEIGHT;
    SDL_BlitSurface( font_image, &rect, temp_gfx->glyphs[i], NULL );
    SDL_SetColorKey( temp_gfx->glyphs[i], SDL_SRCCOLORKEY,
                      SDL_MapRGB( format, 0, 255, 0 ) );
  }
  SDL_FreeSurface( font_image );
  
  for( i = 0; i < 24; i++ )
    mathCreateRothstein( 24, i, &temp_gfx->rothstein24[i * 3] );
  
  return temp_gfx;
}

void gfxQuit(Gfx *gfx){
  int i;
  Sprite *sprite;
  SpriteType *type;
  
  if( !gfx ){
    errorSet("GFX", "Invalide Gfx context");
    return;
  }
  
  free( gfx->_rect_mem );
  
  for( i = 0; i < FONT_GLYPHS; i++ )
    SDL_FreeSurface( gfx->glyphs[i] );
  
  while( gfx->sprites ){
    sprite = gfx->sprites;
    gfx->sprites = sprite->next;

    spriteDelete( sprite );
  }

  while( gfx->strings ){
    stringDelete( gfx->strings );
  }
   
  while( gfx->_sprite_types ){
    type = gfx->_sprite_types;
    gfx->_sprite_types = type->next;

    sprite = type->sprite;
    for( i = 0; i < sprite->num_frames; i++ )
      SDL_FreeSurface( sprite->frames[i] );
    free( sprite->frames );
    free( sprite );
    free( type->name );
    free( type );
  }
  
  free( gfx );
}

void gfxDirtyRectangle(Gfx *gfx, SDL_Rect *rect){
  Rect *temp_rect;

  if( !gfx ){
    errorSet("GFX", "Invalid Gfx context");
    return;
  }

  if( !(temp_rect = gfx->_clean_rects) ){
    errorSet("GFX", "No clean rectangles available");
    return;
  }

  gfx->_clean_rects = temp_rect->next;
  if( rect )
    memcpy( &temp_rect->rect, rect, sizeof( SDL_Rect ) );
  else{
    temp_rect->rect.x = 0;
    temp_rect->rect.y = 0;
    temp_rect->rect.w = gfx->screen->w;
    temp_rect->rect.h = gfx->screen->h;
  }
  temp_rect->next = gfx->_dirty_rects;
  gfx->_dirty_rects = temp_rect;
}

void gfxUpdate(Gfx *gfx, int take_shot){
  Rect *temp_rect;
  Sprite *sprite;
  String *string;
  SDL_Rect rect;
  SDL_Rect update_rect[64];
  int current_rect = 0;
  
  if( !gfx ){
    errorSet("GFX", "Invalid Gfx context");
    return;
  }

  gfxScrollerUpdate( gfx );
  while( gfx->_dirty_rects ){
    temp_rect = gfx->_dirty_rects;
    SDL_BlitSurface( gfx->background, &temp_rect->rect,
                     gfx->screen, &temp_rect->rect );
    memcpy( &update_rect[current_rect], &temp_rect->rect, sizeof( SDL_Rect ) );
    current_rect++;
    gfx->_dirty_rects = temp_rect->next;
    temp_rect->next = gfx->_clean_rects;
    gfx->_clean_rects = temp_rect;
  }

  
  sprite = gfx->sprites;
  while( sprite ){
    if( !sprite->hidden ){
      rect.x = sprite->x;
      rect.y = sprite->y;
      rect.w = sprite->width;
      rect.h = sprite->height;
      SDL_BlitSurface( sprite->frames[sprite->current_frame], NULL,
                       gfx->screen, &rect);
      memcpy( &update_rect[current_rect], &rect, sizeof( SDL_Rect ) );
      current_rect++;
      gfxDirtyRectangle( gfx, &rect );
    }
    sprite = sprite->next;
  }
  
  string = gfx->strings;
  while( string ){
    rect.x = string->x;
    rect.y = string->y;
    rect.w = string->width;
    rect.h = string->height;
    SDL_BlitSurface( string->image, NULL,
                     gfx->screen, &rect );
    memcpy( &update_rect[current_rect], &rect, sizeof( SDL_Rect ) );
    current_rect++;
    gfxDirtyRectangle( gfx, &rect );
    
    string = string->next;
  }

  SDL_UpdateRects( gfx->screen, current_rect, update_rect );
  
  if( take_shot )
    SDL_SaveBMP( gfx->screen, "shot.bmp" );
  
  gfx->frame_count++;
  if( !( gfx->frame_count%5 ) ){
    Uint32 current_time=SDL_GetTicks();
    gfx->frame_delta = ( current_time - gfx->last_time ) / 5000.0;
    gfx->last_time = current_time;
  }
  gfx->time += gfx->frame_delta;
}


SDL_Surface *gfxLoadImage(char *filename){
  SDL_Surface *temp1, *temp2;
  char error[256];

  
  /*temp1 = IMG_Load( filename );*/
  temp1 = SDL_LoadBMP( filename );
  strcpy( error, "Could not load image " );
  strcat( error, filename );
  if( !temp1 ){
    errorSet( "GFX", error );
    return NULL;
  }

  temp2 = SDL_DisplayFormat( temp1 );
  SDL_FreeSurface( temp1 );

  return temp2;
}

SDL_Surface *gfxLoadImageTrans(char *filename, int rle){
  SDL_Surface *temp1, *temp2;
  char error[256];

  
  /*temp1 = IMG_Load( filename );*/
  temp1 = SDL_LoadBMP( filename );
  strcpy( error, "Could not load image " );
  strcat( error, filename );
  if( !temp1 ){
    errorSet( "GFX", error );
    return NULL;
  }
  SDL_LockSurface( temp1 );
  SDL_SetColorKey( temp1,
                   SDL_SRCCOLORKEY | ( rle ? SDL_RLEACCEL : 0 ),
                   *(Uint32 *) temp1->pixels);
  temp2 = SDL_DisplayFormat( temp1 );
  SDL_FreeSurface( temp1 );

  return temp2;
}
  
static Sprite *_spriteCopy(Sprite *in){
  Sprite *out = ( Sprite* ) malloc( sizeof( Sprite ) );

  out->x = 0;
  out->y = 0;
  out->width = 0;
  out->height = 0;
  out->num_frames = in->num_frames;
  out->current_frame = 0;
  out->priority = 0;
  out->hidden = 1;
  out->frames = ( SDL_Surface** ) malloc( sizeof( SDL_Surface* ) * 
      out->num_frames );
  memcpy( out->frames, in->frames, sizeof( SDL_Surface * ) * out->num_frames );
  out->next = 0;

  return out;
}


Sprite *gfxAddSprite(Gfx *gfx, char *name){
  Sprite *prev, *next, *current;
  Sprite *sprite;
  SpriteType *type = gfx->_sprite_types;
  
  while( type && strcmp( name, type->name ) )
    type = type->next;

  if( !type ) return NULL;
  
  sprite = _spriteCopy( type->sprite );
  
  if( !gfx->sprites ){
    gfx->sprites = sprite;
    sprite->next = NULL;
    return sprite;
  }

  if( gfx->sprites->priority >= sprite->priority ){
    next = gfx->sprites;
    gfx->sprites = sprite;
    sprite->next = next;
    return sprite;
  }

  prev = gfx->sprites;
  current = gfx->sprites->next;

  while( current ){
    if( current->priority >= sprite->priority ){
      sprite->next = current;
      prev->next = sprite;
      return sprite;
    }
    prev = current;
    current = current->next;
  }

  prev->next = sprite;
  sprite->next = NULL;

  return sprite;
}

void gfxRemoveSprite(Gfx *gfx, Sprite *sprite){
  Sprite *prev, *current;

  if( gfx->sprites == sprite ){
    gfx->sprites = gfx->sprites->next;
    spriteDelete( sprite );
    return;
  }

  prev = gfx->sprites;
  current = prev->next;

  while( current ){
    if( current == sprite ){
      prev->next = current->next;
      spriteDelete( current );
      return;
    }
    prev = current;
    current = current->next;
  }
}

void spriteNewType(Gfx *gfx, char *name, int width, int height, int num_frames,
    SDL_Surface *image, Uint8 ck_red, Uint8 ck_green, Uint8 ck_blue){
  
  SpriteType *type;
  Sprite *sprite;
  SDL_PixelFormat *fmt;
  SDL_Rect rect;
  int i;
  
  type = (SpriteType *) malloc( sizeof( SpriteType ) );
  type->name = _strdup( name );

  sprite = (Sprite *) malloc( sizeof( Sprite ) );
  type->sprite = sprite;
  
  sprite->x = 0;
  sprite->y = 0;
  sprite->width = width;
  sprite->height = height;
  sprite->num_frames = num_frames;
  sprite->current_frame = 0;
  sprite->priority = 0;
  sprite->hidden = 1;
  sprite->next = NULL;
  
  fmt = gfx->screen->format;
  
  sprite->frames = ( SDL_Surface** ) malloc( sizeof( SDL_Surface* ) *
                                              num_frames );

  for( i = 0; i < sprite->num_frames; i++ ){
    sprite->frames[i] = SDL_CreateRGBSurface( 0,
                                              width,
                                              height,
                                              fmt->BitsPerPixel,
                                              fmt->Rmask,
                                              fmt->Gmask,
                                              fmt->Bmask,
                                              fmt->Amask );
    rect.x = i * width;
    rect.y = 0;
    rect.w = width;
    rect.h = height;
    SDL_BlitSurface( image, &rect, sprite->frames[i], NULL );
    SDL_SetColorKey( sprite->frames[i], SDL_SRCCOLORKEY | SDL_RLEACCEL,
        SDL_MapRGB( sprite->frames[i]->format, ck_red, ck_green, ck_blue ) );
  }
  
  type->next = gfx->_sprite_types;
  gfx->_sprite_types = type;
}

void spriteDelete(Sprite *sprite){
  free( sprite->frames );
  free( sprite );
}

String *stringNew(Gfx *gfx, int length){
  String *str;
  
  str = ( String* ) malloc( sizeof( String ) );

  str->gfx = gfx;
  str->length = length;
  str->string = ( char* ) malloc( length + 1 );
  str->string[0] = '\0';
  str->x = 0;
  str->y = 0;
  str->width = FONT_WIDTH * length;
  str->height = FONT_HEIGHT;
  str->need_redraw = 0;
  str->next = NULL;
  str->image = SDL_CreateRGBSurface( SDL_SWSURFACE,
                                     FONT_WIDTH * length,
                                     FONT_HEIGHT,
                                     gfx->screen->format->BitsPerPixel,
                                     gfx->screen->format->Rmask,
                                     gfx->screen->format->Gmask,
                                     gfx->screen->format->Bmask,
                                     gfx->screen->format->Amask );
  SDL_SetColorKey( str->image, SDL_SRCCOLORKEY,
                    SDL_MapRGB( str->image->format, 0, 255, 255 ) );
  SDL_FillRect( str->image, NULL,
                SDL_MapRGB( str->image->format, 0, 255, 255 ) );
  
  str->next = gfx->strings;
  gfx->strings = str;
  
  return str;
}

void stringDelete(String *str){
  String *prev, *current;

  if( str->gfx->strings == str )
    str->gfx->strings = str->next;
  else{
    prev = str->gfx->strings;
    current = str->gfx->strings->next;
    while( current ){
      if( current == str ){
        prev->next = current->next;
        current = NULL;
      }
      else{
        prev = current;
        current = current->next;
      }
    }
  }
      
  SDL_FreeSurface( str->image );
  
  free( str->string );

  free( str );
}

void stringSet(String *str, char *string, int centre){
  char *ptr;
  int i;
  SDL_Rect rect; 
  ptr = str->string;
 
  if( !string ) return;
  if( strlen( string ) < (unsigned)str->length )
    strcpy( str->string, string );
  else{
    strncpy( str->string, string, str->length );
    str->string[str->length] = '\0';
  }

  if( centre ){
      int string_w = str->length * FONT_WIDTH;
      int text_w = strlen( string ) * FONT_WIDTH;
      
      i = ( string_w - text_w ) / 2;
  }
  else
      i = 0;

  SDL_FillRect( str->image, NULL,
                SDL_MapRGB( str->image->format, 0, 255, 255 ) );
  while( *ptr != '\0' ){
    rect.x = i;
    rect.y = 0;
    rect.w = FONT_WIDTH;
    rect.h = FONT_HEIGHT;
    SDL_BlitSurface( str->gfx->glyphs[*ptr - 1], NULL, str->image, &rect );
    ptr++;
    i += FONT_WIDTH;
  }
  str->need_redraw = 1;
}

void stringMove(String *str, int x, int y){
  str->x = x;
  str->y = y;
  str->need_redraw = 1;
}

void mathCreateRothstein(int p, int q, Uint8 *code){
  double delta = q / (double) p;
  double last_int = 0;
  double current = 0;
  int column;
  Uint8 bit;
  
  *code = 0;
  bit = 0;
  for( column = 0; column < p; column++ ){
    if( bit & 0x8 ){
      code++;
      *code = 0;
      bit = 0;
    }
    current += delta;
    if( floor( current ) > last_int ){
      last_int = floor( current );
      *code |= ( 1 << bit );
    }
    bit++;
  }
}
  
/*FIXME Not endian safe */
void gfxShrinkBlit16(SDL_Surface *src,
                     SDL_Surface *dest,
                     SDL_Rect *dest_rect,
                     Uint8 *rothstein){
  int x, y;
  Uint16 *pixel_src;
  Uint16 *pixel_dest, *line_dest;
  int src_pitch, dest_pitch;
  Uint8 rx_bit = 0, ry_bit = 0;
  Uint8 *rx_ptr = rothstein, *ry_ptr = rothstein;
  
  SDL_LockSurface( src );
  SDL_LockSurface( dest );

  src_pitch = src->pitch >> 1;
  dest_pitch = dest->pitch >> 1;
  pixel_src = (Uint16 *) src->pixels;
  line_dest = pixel_dest = (Uint16 *) dest->pixels +
                              ( dest_rect->y * dest_pitch ) + dest_rect->x;

  for( y = 0; y < src->h; y++ ){
    if( *ry_ptr & ( 1 << ry_bit ) ){
      rx_ptr = rothstein;
      rx_bit = 0;
      for( x = 0; x < src->w; x++ ){
        if( *rx_ptr & ( 1 << rx_bit ) )
          *(pixel_dest++) = *(pixel_src++);
        else
          pixel_src++;
        rx_bit++;
        if( rx_bit & 0x8 ){
          rx_bit = 0;
          rx_ptr++;
        }
      }
      line_dest += dest_pitch;
      pixel_dest = line_dest;
    }
    else
      pixel_src += src_pitch;

    ry_bit++;
    if( ry_bit & 0x8 ){
      ry_bit = 0;
      ry_ptr++;
    }
  }
  SDL_UnlockSurface( dest );
  SDL_UnlockSurface( src );
}

void gfxORBlit16(SDL_Surface *src,
                 SDL_Surface *dest,
                 SDL_Rect *dest_rect){
  int x, y;
  Uint16 *pixel_src;
  Uint16 *pixel_dest, *line_dest;
  int dest_pitch;
  
  SDL_LockSurface( src );
  SDL_LockSurface( dest );

  dest_pitch = dest->pitch >> 1;
  line_dest = pixel_dest = (Uint16 *) dest->pixels +
                              ( dest_rect->y * dest_pitch ) + dest_rect->x;
  pixel_src = src->pixels;

  for( y = 0; y < src->h; y++ ){
    for( x = 0; x < src->w; x++ ){
        *pixel_dest |= *(pixel_src++);
        pixel_dest++;
    }
    line_dest += dest_pitch;
    pixel_dest = line_dest;
  }
  SDL_UnlockSurface( dest );
  SDL_UnlockSurface( src );
}          

int gfxScrollerSet(Gfx *gfx, char *text, int force){
    int status = gfxScrollerStatus( gfx );

    if( !force && status != SCROLLER_IDLE )
        return 0;

    stringSet( gfx->scroller, text, 1 );
    gfxScrollerReset( gfx );
    gfxScrollerStart( gfx );
    
    return 1;
}

int gfxScrollerStatus(Gfx *gfx){
    return gfx->scroller_status;
}

void gfxScrollerArgs(Gfx *gfx, int scrollin_time, int scrollout_time,
        int rest_time){
    gfx->scroller_inpps = gfx->screen->w / ( scrollin_time / 1000.0 );
    gfx->scroller_outpps = gfx->screen->w / ( scrollout_time / 1000.0 );
    gfx->scroller_resttime = rest_time / 1000.0;
}

void gfxScrollerReset(Gfx *gfx){
    gfx->scroller_pos = gfx->screen->w;
    gfx->scroller_status = SCROLLER_IDLE;
    SDL_FillRect( gfx->screen, &gfx->scroller_rect, 1 );
}

void gfxScrollerStart(Gfx *gfx){
    gfx->scroller_status = SCROLLER_IN;
}

void gfxScrollerUpdate(Gfx *gfx){
    static float rest_time = 0;

    switch( gfx->scroller_status ){
        case SCROLLER_IDLE:
            break;

        case SCROLLER_IN:
            gfx->scroller_pos -= gfx->scroller_inpps * gfx->frame_delta;
            stringMove( gfx->scroller, gfx->scroller_pos, gfx->scroller_rect.y);
            if( gfx->scroller_pos <= 0 ){
                gfx->scroller_status = SCROLLER_REST;
                rest_time = 0;
            }
            break;
        case SCROLLER_REST:
            rest_time += gfx->frame_delta;
            if( rest_time > gfx->scroller_resttime )
                gfx->scroller_status = SCROLLER_OUT;
            break;
        case SCROLLER_OUT:
            gfx->scroller_pos -= gfx->scroller_outpps * gfx->frame_delta;
            stringMove( gfx->scroller, gfx->scroller_pos, gfx->scroller_rect.y);
            if( gfx->scroller_pos <= -gfx->screen->w )
                gfx->scroller_status = SCROLLER_IDLE;
            break;
        default:
            break;
    }
}

