/* -- Include the precompiled libraries -- */
#pragma comment(lib, "libSDLx.lib")
#pragma comment(lib, "SDL_Image.lib")
#pragma comment(lib, "SDL_ttf.lib")

#include <stdio.h>
#include <stdlib.h>
#include "abombniball.h"
#include "gfx.h"
#include "game.h"
#include "error.h"
#include "file.h"
#include "misc.h"
#include <string.h>
#include "editor.h"
#include "input.h"

#include "tile_types.h"
#include "level_header.h"

static Gfx *gfx = NULL;
static Grid *grid = NULL;
static Sprite *selector = NULL;
static Sprite *marker = NULL;
static Sprite *start = NULL;
static Bomb2 *offgrid_bombs = NULL;
static Bomb2 *ongrid_bombs = NULL;
static struct DraggingObject{
    int type;
    union d{
        Bomb2 *bomb;
        Sprite *start;
    }d;
} dragging_object;

#define BOMB_TYPE 1
#define START_TYPE 2

static Bomb2 *_bomb_memory;

static int selector_state = SELECTOR_VOID;
static int selected_tile = 0;
static char *filename = "unknown";

int editorLoadGfx();
void editorUnloadGfx();

#define TILE_SELECTOR_X ( 640 - TILE_WIDTH )
#define BOMB_SELECTOR_X ( 640 - TILE_WIDTH * 3 )

#define NUM_BUTTONS 6
#define BUTTON_X 480
#define BUTTON_Y 300
#define BUTTON_W FONT_WIDTH * 8
#define BUTTON_H FONT_HEIGHT
static String *buttons[NUM_BUTTONS] = { NULL, NULL, NULL, NULL, NULL, NULL };
static char *button_text[NUM_BUTTONS] = { NULL, "NAME", "MUSIC", "LOAD",
                                            "SAVE", "QUIT" };
int main(int argc, char *argv[]){
	/*
    abombniball_dir = getenv( "ABOMBNIBALL_DIR" );

    if( !abombniball_dir ){
        abombniball_dir = DEFAULT_DATA_DIR;
    }
	*/
	abombniball_dir = "d:";

    if( SDL_Init( SDL_INIT_VIDEO | SDL_INIT_AUDIO | SDL_INIT_NOPARACHUTE)){
        errorSet( "MAIN", SDL_GetError() );
        exit( -1 );
    }
    gfx = gfxInit( 640, 480, 0, 256 );
    if( !gfx ){
        errorSet( "MAIN", "Could not create graphcis context" );
        fprintf( stderr, "  Ensure game data is present in %s\n",
                abombniball_dir );
        fprintf( stderr,
                "  or set the ABOMBNIBALL_DIR environment variable\n" );
        fprintf( stderr, "  to the correct directory.\n" );
        exit( -1 );
    }

    if( !editorLoadGfx() ){
        errorSet( "MAIN", "Could not load graphics" );
        fprintf( stderr, "  Ensure game data is present in %s\n",
                abombniball_dir );
        fprintf( stderr,
                "  or set the ABOMBNIBALL_DIR environment variable\n" );
        fprintf( stderr, "  to the correct directory.\n" );
        exit( -1 );
    }
    gfxScrollerArgs( gfx, 1000, 1000, 1500 ); 
    SDL_ShowCursor( 1 );
    gfx->background = SDL_CreateRGBSurface( SDL_SWSURFACE, gfx->screen->w,
            gfx->screen->h, 16, gfx->screen->format->Rmask,
            gfx->screen->format->Gmask, gfx->screen->format->Bmask,
            gfx->screen->format->Amask );

    if( argc == 2 ){
        if( editorLoadLevel( fileCreateName( LEVELS, argv[1] ) ) < 0 )
            editorGridNew( 18, 18 );
        filename = argv[1];
    }
    else
        editorGridNew( 18, 18 );

    editorGridRedraw();
    editorRedrawTileSelector();
    editorRedrawButtons();
    editorReshuffleBombs();
    editorRedrawJumps();
    
    gfxScrollerSet( gfx, "Abombniball Level Editor", 1 );
    while( !editorHandleEvents() ){
        gfxUpdate( gfx, 0 );
    }
    gfxUpdate( gfx, 1 );
    editorUnloadGfx();
    gfxQuit( gfx );
    SDL_Quit();
    exit( 0 );
}

int editorLoadGfx(){
    SDL_Surface *image;
    Tile *tile;
    Bomb2 *bomb;
    int i;
    char filename[255];
    
    image = gfxLoadImage( fileCreateName( GFX, "bomb.bmp" ) );
    if( !image ) return 0;
    spriteNewType( gfx, "bomb", 24, 24, 20, image, 0, 255, 0 );
    SDL_FreeSurface( image );
    
    _bomb_memory = (Bomb2 *) malloc( sizeof( Bomb2 ) * 9 );
    offgrid_bombs = _bomb_memory;
    
    for( i = 0; i < 9; i++ ){
        offgrid_bombs[i].next = &offgrid_bombs[i + 1];
        offgrid_bombs[i].sprite = gfxAddSprite( gfx, "bomb" );
        offgrid_bombs[i].sprite->current_frame = 19;
        offgrid_bombs[i].sprite->x = BOMB_SELECTOR_X;
        offgrid_bombs[i].sprite->y = 0;
    }
    offgrid_bombs[8].next = NULL;
            
    for( i = 0; i < NUM_TILES; i++ ){
        tile = &_tile_types[i];
        strcpy( filename, tile->name );
        strcat( filename, ".bmp" );
        tile->image = gfxLoadImage( fileCreateName( GFX, filename ) );
        if( !tile->image ) return 0;
    }
    image = gfxLoadImage( fileCreateName( GFX, "selector.bmp" ) );
    if( !image ) return 0;
    spriteNewType( gfx, "selector", 24, 24, 1, image, 0, 255, 0 );
    SDL_FreeSurface( image );
    image = gfxLoadImage( fileCreateName( GFX, "ball.bmp" ) );
    if( !image ) return 0;
    spriteNewType( gfx, "start", 24, 24, 8, image, 0, 255, 0 );
    SDL_FreeSurface( image );
   
    selector = gfxAddSprite( gfx, "selector" );
    selector->current_frame = 0;
    selector->hidden = 0;
    marker = gfxAddSprite( gfx, "selector" );
    marker->current_frame = 0;
    marker->hidden = 0;
    marker->x = TILE_SELECTOR_X;
    marker->y = 0;
    start = gfxAddSprite( gfx, "start" );
    start->hidden = 0;
    start->current_frame = 5;
    return 1;
}

void editorUnloadGfx(){
    int i;

    for( i = 0; i < NUM_TILES; i++ )
        SDL_FreeSurface( _tile_types[i].image );
    free( _bomb_memory );
}


void editorGridNew(int width, int height){
    int i;

    grid = (Grid *) malloc( sizeof( Grid ) );
    grid->width = width;
    grid->height = height;
    grid->num_tiles = width * height;
    
    grid->tilemap = (Tile *) malloc( sizeof( Tile ) * grid->num_tiles );

    grid->jumps_allowed = 0;

    editorMoveStart( 9, 9 );
    grid->name = _strdup( "UNNAMED" );
    grid->tune = _strdup( "title.xm" );
    for( i = 0; i < grid->num_tiles; i++ )
        memcpy( &grid->tilemap[i], &_tile_types[0], sizeof( Tile ) );
}

void editorGridDelete(){
    free( grid->tilemap );
    free( grid->name );
    free( grid->tune );
    free( grid );
}

void editorGridRedraw(){
    int x, y, i = 0;
    SDL_Rect rect;

    for( y = 0; y < grid->height; y++ ){
        for( x = 0; x < grid->width; x++ ){
            rect.x = x * TILE_WIDTH;
            rect.y = y * TILE_HEIGHT;

            if( grid->tilemap[i].image )
                SDL_BlitSurface( grid->tilemap[i].image, NULL,
                        gfx->background, &rect );
            i++;
        }
    }
    rect.x = rect.y = 0;
    rect.w = grid->width * TILE_WIDTH;
    rect.h = grid->height * TILE_HEIGHT;

    gfxDirtyRectangle( gfx, &rect );
}

void editorMoveStart( int grid_x, int grid_y ){
    grid->start_x = grid_x;
    grid->start_y = grid_y;

    start->x = grid_x * 24;
    start->y = grid_y * 24;
} 
    
void editorRedrawTileSelector(){
    SDL_Rect rect;
    int i;

    for( i = 0; i < NUM_TILES; i++ ){
        rect.x = TILE_SELECTOR_X;
        rect.y = i * TILE_HEIGHT;
        SDL_BlitSurface( _tile_types[i].image, NULL,
                gfx->background, &rect );
    }

    rect.x = TILE_SELECTOR_X;
    rect.y = 0;
    rect.w = TILE_WIDTH;
    rect.h = TILE_HEIGHT * NUM_TILES;
    gfxDirtyRectangle( gfx, &rect );
}

void editorRedrawButtons(){
    int i;
    for( i = 0; i < NUM_BUTTONS; i++ ){
        if( buttons[i] == NULL )
            buttons[i] = stringNew( gfx, BUTTON_W / FONT_WIDTH );
        stringSet( buttons[i], button_text[i], 1 );
        stringMove( buttons[i], BUTTON_X, BUTTON_Y
                + BUTTON_H * i );
    }
}

void editorRedrawJumps(){
    if( button_text[0] )
        free( button_text[0] );

    button_text[0] = ( char * )malloc( 9 );
    sprintf( button_text[0], "%d Jumps", grid->jumps_allowed );
    editorRedrawButtons();
}

void editorReshuffleBombs(){
    Bomb2 *bomb;
    int num = 0;
    
    bomb = ongrid_bombs;
    while( bomb ){
        bomb->sprite->x = bomb->grid_x * 24;
        bomb->sprite->y = bomb->grid_y * 24;
        bomb->sprite->current_frame = 1 + num * 2;
        bomb->sprite->hidden = 0;
        num++;
        bomb = bomb->next;
    }

    bomb = offgrid_bombs;
    while( bomb ){
        bomb->sprite->x = BOMB_SELECTOR_X;
        bomb->sprite->y = 0;
        bomb->sprite->current_frame = 1 + num * 2;
        bomb->sprite->hidden = 1;
        num++;
        bomb = bomb->next;
    }
    if( offgrid_bombs ) offgrid_bombs->sprite->hidden = 0;
}
    
        
int editorHandleEvents(){
    SDL_Event event;
    int quit = 0;
    
    while( SDL_PollEvent( &event ) ){
        switch( event.type ){
            case SDL_MOUSEMOTION:
                editorMoveSelector( &event.motion );
                break;

            case SDL_MOUSEBUTTONDOWN:
                editorMousePress( &event.button );
                break;
            case SDL_MOUSEBUTTONUP:
                editorMouseRelease( event.button.button );
                break;
            case SDL_QUIT:
                quit = 1;
                break;

            default:
                break;
        }
    }

    return quit;
}

void editorMoveSelector(SDL_MouseMotionEvent *motion){
    int x = motion->x;
    int y = motion->y;
    
    if( x < grid->width * TILE_WIDTH && y < grid->height * TILE_HEIGHT ){
        selector_state = SELECTOR_GRID;
        if( dragging_object.type ){
            marker->hidden = 1;
            selector->hidden = 1;
            if( dragging_object.type == BOMB_TYPE ){
                Bomb2 *bomb = dragging_object.d.bomb;
                bomb->grid_x = ( x / 24 );
                bomb->grid_y = ( y / 24 );
                bomb->sprite->x = bomb->grid_x * 24;
                bomb->sprite->y = bomb->grid_y * 24;
            }
            else if( dragging_object.type == START_TYPE ){
                editorMoveStart( x / 24, y / 24 );
            }
                
        }
        else{
            static int been_here_seen_that = 0;
            marker->hidden = 0;
            selector->hidden = 0;
            selector->x = ( x / 24 ) * 24;
            selector->y = ( y / 24 ) * 24;
            if( motion->state & SDL_BUTTON_LMASK ){
                editorPlaceTile( selector->x / 24, selector->y / 24,
                        selected_tile);
            }
            else if( motion->state & SDL_BUTTON_RMASK ){
                editorPlaceTile( selector->x / 24, selector->y / 24, 0 );
            }
            if( !been_here_seen_that )
                been_here_seen_that = gfxScrollerSet( gfx,
                    "Left Button: Set   Right Button: Clear", 0 );
        }
    }
    else if( x > BOMB_SELECTOR_X && x < BOMB_SELECTOR_X + TILE_WIDTH
            && y < TILE_HEIGHT ){
        static int been_here_seen_that = 0;
        selector_state = SELECTOR_BOMB;
        marker->hidden = 0;
        selector->hidden = 1;
        if( !been_here_seen_that )
            been_here_seen_that = gfxScrollerSet( gfx, 
                    "Left Button: Select and drag bomb", 0 );
    }        
    else if( x > TILE_SELECTOR_X && y < TILE_HEIGHT * NUM_TILES ){
        selector_state = SELECTOR_TILES;
        if( dragging_object.type ){
            if( dragging_object.type == BOMB_TYPE ){
                Bomb2 *bomb = dragging_object.d.bomb;
                marker->hidden = 1;
                selector->hidden = 1;
                bomb->sprite->x = x;
                bomb->sprite->y = y;
            }
            else if( dragging_object.type == START_TYPE ){
                dragging_object.type = 0;
            }
        }
        else{
            static int been_here_seen_that = 0;
            selector->x = TILE_SELECTOR_X;
            selector->y = ( y / 24 ) * 24;
            marker->hidden = 1;
            selector->hidden = 0;
            if( !been_here_seen_that )
                been_here_seen_that = gfxScrollerSet( gfx, 
                        "Left Button: Select Tile Type", 0 );
        }
    }
    else if( x > BUTTON_X && x < BUTTON_X + BUTTON_W && y > BUTTON_Y ){
        selector_state = SELECTOR_BUTTONS;
    }
    else{
        selector_state = SELECTOR_VOID;
        if( dragging_object.type == BOMB_TYPE ){
            Bomb2 *bomb = dragging_object.d.bomb;
            marker->hidden = 1;
            selector->hidden = 1;
            bomb->sprite->x = x;
            bomb->sprite->y = y;
        }
        else{
            dragging_object.type = 0;
            selector->hidden = 1;
            marker->hidden = 0;
            marker->x = TILE_SELECTOR_X;
            marker->y = selected_tile * TILE_HEIGHT;
        }
    }
}

void editorMousePress(SDL_MouseButtonEvent *mbutton){
    int grid_x = selector->x / 24;
    int grid_y = selector->y / 24;
    int button_press;
    int button = mbutton->button;
    SDL_Event event;
    
    switch( selector_state ){
        case SELECTOR_GRID:
            if( button == SDL_BUTTON_LEFT ){
                Bomb2 *bomb = ongrid_bombs;
                while( bomb ){
                    if( bomb->grid_x == grid_x &&
                            bomb->grid_y == grid_y ){
                        break;
                    }
                    bomb = bomb->next;
                }
                if( bomb ){
                    dragging_object.type = BOMB_TYPE;
                    dragging_object.d.bomb = bomb;
                }
                else if( grid_x == grid->start_x
                    && grid_y == grid->start_y ){
                    dragging_object.type = START_TYPE;
                    dragging_object.d.bomb = start;
                }
                else
                    editorPlaceTile( selector->x / 24, selector->y / 24,
                            selected_tile);
            }
            else if( button == SDL_BUTTON_RIGHT ){
                editorPlaceTile( selector->x / 24, selector->y / 24, 0 );
            }
            break;

        case SELECTOR_TILES:
            selected_tile = selector->y / 24;
            break;

        case SELECTOR_BOMB:
            if( button == SDL_BUTTON_LEFT ){
                dragging_object.type = BOMB_TYPE;
                dragging_object.d.bomb = offgrid_bombs;
            }
            break;

        case SELECTOR_BUTTONS:
            button_press = ( mbutton->y - BUTTON_Y ) / BUTTON_H;
            switch( button_press ){
                case 0: /* jumps */
                    if( button == SDL_BUTTON_LEFT )
                        grid->jumps_allowed++;
                    if( button == SDL_BUTTON_RIGHT )
                        grid->jumps_allowed--;
                    if( grid->jumps_allowed < 0 )
                        grid->jumps_allowed = 9;
                    if( grid->jumps_allowed > 9 )
                        grid->jumps_allowed = 0;
                    editorRedrawJumps();
                    break;

                case 1: /* name */
                    break;

                case 2: /* music */
                    break;

                case 3: /* load */
                    break;

                case 4: /* save */
                    editorSaveLevel( fileCreateName( LEVELS, filename ) );
                    gfxScrollerSet( gfx, "Saved", 1 ); 
                    break;

                case 5: /* quit */
                    event.type = SDL_QUIT;
                    SDL_PushEvent( &event );
                    break;

                default:
                    break;
            }
            break;

        default:
            break;
    }
}

void editorMouseRelease(int button){
    if( dragging_object.type && button == SDL_BUTTON_LEFT ){
        if( selector_state == SELECTOR_GRID
                && dragging_object.type == BOMB_TYPE
                && dragging_object.d.bomb == offgrid_bombs ){
            Bomb2 *bomb = ongrid_bombs;
            offgrid_bombs = dragging_object.d.bomb->next;
            if( !bomb ){
                ongrid_bombs = dragging_object.d.bomb;
                ongrid_bombs->next = NULL;
            }
            else{
                while( bomb->next )
                    bomb = bomb->next;
                bomb->next = dragging_object.d.bomb;
                dragging_object.d.bomb->next = NULL;
            }
        }
        else if ( selector_state != SELECTOR_GRID 
                && dragging_object.type == BOMB_TYPE
                && dragging_object.d.bomb != offgrid_bombs ){
            Bomb2 *prev = ongrid_bombs;
            if( prev == dragging_object.d.bomb ){
                ongrid_bombs = ongrid_bombs->next;
                prev->next = offgrid_bombs;
                offgrid_bombs = prev;
            }
            else{
                while( prev->next != dragging_object.d.bomb )
                    prev = prev->next;
                prev->next = dragging_object.d.bomb->next;
                dragging_object.d.bomb->next = offgrid_bombs;
                offgrid_bombs = dragging_object.d.bomb;
            }
        }
        dragging_object.type = 0;
        editorReshuffleBombs();
    }
}

void editorPlaceTile(int x, int y, int type){
    SDL_Rect rect;
    int tile = y * grid->width + x;
    memcpy( &grid->tilemap[tile], &_tile_types[type], sizeof( Tile ) );
    rect.x = x * TILE_WIDTH;
    rect.y = y * TILE_HEIGHT;
    SDL_BlitSurface( grid->tilemap[tile].image, NULL, gfx->background, &rect );
    gfxDirtyRectangle( gfx, &rect );
}


int editorLoadLevel(char *levelname){
  Bomb2 *bomb;
  FILE *fp;
  int reading_grid=0;
  char name[17];
  char line[256];
  char tune[256];
  Tile *tile_types[256];
  char *tile_name, tile_char;
  int jumps = 3;
  int i;
  int tiles_read = 0;
  char read_char;
  
  int bomb_coords[256];
  int bomb_count=0;
  int start_x, start_y;
  int grid_width, grid_height;
  
  tune[0] = '\0';
  
  memset( tile_types, 0, sizeof( Tile* ) * 256 );
  if( grid ) editorGridDelete();
  bomb_coords[0] = -1;

  fp = fopen( levelname, "r" );
  if( !fp ){
    errorSet( "LEVEL", "Could not load level" );
    return -1;
  }
  while( !reading_grid && fgets( line, 255, fp ) ){
    switch( line[0] ){
      case '\"':
        strip_crud_space( &line[2] );
        strncpy( name, &line[2], 16 );
        name[16] = '\0';
        break;

      case '!':
        tile_char = line[2];
        tile_name = &line[4];
        strip_crud( tile_name ); 
        for( i = 0; i < NUM_TILES; i++ ){
          if( !strcmp( _tile_types[i].name, tile_name ) )
            tile_types[ (int) tile_char ] = &_tile_types[i];
        }
        break;

      case 'd':
        get_coord( &line[2], &grid_width, &grid_height );
        break;
       
      case 's':
        get_coord( &line[2], &start_x, &start_y );
        break;

      case 'b':
        get_coord( &line[2], &bomb_coords[bomb_count],
                             &bomb_coords[bomb_count+1] );
        bomb_coords[bomb_count+2] = -1;
        bomb_count += 2;
        break;
      case 'j':
        strip_crud( &line[2] );
        jumps = atoi( &line[2] );
        break;
      case 'm':
        strip_crud( &line[2] );
        strcpy( tune, &line[2] );
        break;
      case '$':
        reading_grid = 1;
        break;

      default:
        break;
    }
  }

  editorGridNew( grid_width, grid_height );
  
  while( tiles_read < grid->num_tiles && ( read_char = getc( fp ) ) != EOF ){
    if( tile_types[(int) read_char] ){
      memcpy( &grid->tilemap[tiles_read],
              tile_types[(int) read_char],
              sizeof( Tile ) );
      tiles_read++;
    }
  }
  grid->jumps_allowed = jumps;
  grid->name = _strdup( name );
  grid->tune = _strdup( tune );
  editorMoveStart( start_x, start_y );
  
  bomb_count = 0;
  bomb = NULL;
  while( bomb_coords[bomb_count] != -1 ){
      offgrid_bombs->grid_x = bomb_coords[bomb_count];
      offgrid_bombs->grid_y = bomb_coords[bomb_count+1];
      if( !ongrid_bombs )
          ongrid_bombs = offgrid_bombs;
      else
          bomb->next = offgrid_bombs;
      bomb = offgrid_bombs;
      offgrid_bombs = bomb->next;
      bomb->next = NULL;
      bomb_count += 2;
  }
  editorReshuffleBombs();   
      
  
  fclose( fp );
  return 0;
}

int editorSaveLevel(char *levelname){
    FILE *fp;
    char *comment="# Generated by abomb-edit\n";
    int i;
    Bomb2 *bomb;
    
    fp = fopen( levelname, "w" );
    if( !fp ){
        errorSet( "LEVEL", "Could open level for saving" );
        return -1;
    }
    fprintf( fp, "\" %s\n", grid->name );
    fprintf( fp, "m %s\n", grid->tune );
    
    for( i = 0; i < NUM_TILES; i++ ){
        fprintf( fp, "%s\n", level_header[i] );
    }

    fprintf( fp, "d %d %d\n", grid->width, grid->height );
    fprintf( fp, "s %d %d\n", grid->start_x, grid->start_y );
    fprintf( fp, "j %d\n", grid->jumps_allowed );
    
    bomb = ongrid_bombs;
    while( bomb ){
        fprintf( fp, "b %d %d\n", bomb->grid_x, bomb->grid_y );
        bomb = bomb->next;
    }

    fprintf( fp, "$\n" );
    for( i = 0; i < grid->num_tiles; i++ ){
        fprintf( fp, "%c", level_header[grid->tilemap[i].type][2] );
    }
    fprintf( fp, "\n" );

    fclose( fp );
    return 0;
}

        

