/*
 * Abombniball
 * Copyright (C) 2001 Martin Donlon, akawaka@csn.ul.ie
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */
/* -- Include the precompiled libraries -- */
#pragma comment(lib, "libSDLx.lib")
#pragma comment(lib, "SDL_Image.lib")
#pragma comment(lib, "SDL_ttf.lib")


#include <stdio.h>
#include <stdlib.h>
#include "abombniball.h"
#include "gfx.h"
#include "game.h"
#include "input.h"
#include "error.h"
#include "main.h"
#include "file.h"
#include <string.h>

enum{ INIT = 1,
      IDLE,
      GAME,
      REDRAW,
      QUIT,
      MENU_MAIN
} Actions;

#define MAX_MENU 10

typedef struct MenuItem{
  char name[32];
  int action;
} MenuItem;

static MenuItem main_menu[] = { { "Start", GAME },
                                { "Quit", QUIT },
                                { "", -1 }
};


static MenuItem *current_menu = main_menu;
static String *menu_text[MAX_MENU] = { NULL, NULL, NULL, NULL, NULL,
                                       NULL, NULL, NULL, NULL, NULL
};
static int menu_length = 0;
static int pointer_pos = 0;

#define NUM_SCROLLER_ITEMS 11

static char *scroller_text[NUM_SCROLLER_ITEMS] = { "Abombniball",
                                                   "By Martin Donlon",
                                                   "akawaka at skynet.ie",
												   "XBOX Port By freakdave",
												   "Music By",
                                                   "Various",
                                                   "Levels By",
                                                   "Martin Donlon,",
                                                   "Peter Knowles",
                                                   "and",
                                                   "Martin Berglund"
};


int play_music = 1;
int start_level = 1;
int time_limit = 1;
int fullscreen = 0;

int mainProcessArgs(int argc, char *argv[]);
void mainPrintUsage();

int mainTitle(Gfx *gfx);
void mainSetMenu(Gfx *gfx, MenuItem *items);
int mainLoadGfx(Gfx *gfx);
void mainUnloadGfx();
int mainLoadSfx();
void mainUnloadSfx();

int main(int argc, char *argv[]){
  Gfx *gfx;
  
  printf( "+=============================================+\n" );
  printf( "| Abombniball                  (C) 2001|\n");
  printf( "| by Martin Donlon <akawaka@skynet.ie>        |\n" );
  printf( "+=============================================+\n\n" );

  if( mainProcessArgs( argc, argv ) == -1 )
    exit( 0 );
  /*
  abombniball_dir = getenv( "ABOMBNIBALL_DIR" );

  if( !abombniball_dir ){
    abombniball_dir = DEFAULT_DATA_DIR;
  }
  printf( "Using %s as data directory\n", abombniball_dir );
  */
  
  abombniball_dir = "d:";

  if( SDL_Init( SDL_INIT_VIDEO | SDL_INIT_AUDIO | SDL_INIT_JOYSTICK)){
    errorSet( "MAIN", SDL_GetError() );
    exit( -1 );
  }
  
  gfx = gfxInit( 640, 480, fullscreen, 256 );
  if( !gfx ){
    errorSet( "MAIN", "Could not create graphcis context" );
    fprintf( stderr, "  Ensure game data is present in %s\n", abombniball_dir );
    fprintf( stderr, "  or set the ABOMBNIBALL_DIR environment variable\n" );
    fprintf( stderr, "  to the correct directory.\n" );
    exit( -1 );
  }
  
  if( !mainLoadGfx( gfx ) ){
    errorSet( "MAIN", "Could not load graphics" );
    fprintf( stderr, "  Ensure game data is present in %s\n", abombniball_dir );
    fprintf( stderr, "  or set the ABOMBNIBALL_DIR environment variable\n" );
    fprintf( stderr, "  to the correct directory.\n" );
    exit( -1 );
  }
  
  if( Mix_OpenAudio( 22050, AUDIO_S16, 2, 1024 ) ){
    errorSet( "AUDIO", SDL_GetError() );
    exit( -1 );
  }
 /*
  if( !mainLoadSfx() ){
    errorSet( "MAIN", "Could not load sound fx" );
    fprintf( stderr, "  Ensure game data is present in %s\n", abombniball_dir );
    fprintf( stderr, "  or set the ABOMBNIBALL_DIR environment variable\n" );
    fprintf( stderr, "  to the correct directory.\n" );
    exit( -1 );
  }*/
  
  inputInit ();
  
  if( !mainTitle( gfx ) ){
    errorSet( "MAIN", "Undefined error" );
    fprintf( stderr, "  Ensure game data is present in %s\n", abombniball_dir );
    fprintf( stderr, "  or set the ABOMBNIBALL_DIR environment variable\n" );
    fprintf( stderr, "  to the correct directory.\n" );
  }

  mainUnloadGfx();
  /* mainUnloadSfx(); */

  gfxQuit( gfx );
  
  Mix_HaltMusic();
  Mix_CloseAudio();
  SDL_Quit();

  exit( 0 ); 
}

int mainProcessArgs(int argc, char *argv[]){
  int meaning;
  
  while(argc > 1) {
    argc--;
    switch(argv[argc][0]) {
    case '-':
    case '+':
      meaning = ( argv[argc][0] == '-' ? 0 : 1 );
      switch(argv[argc][1]) {
      case 's': 
        play_music = meaning; 
        break;
      case 'f':
        fullscreen = meaning;
        break;
      case 'h':
        mainPrintUsage();
        return -1;
        break; 
#ifdef CHEATING_SCUMBAG
      case 'l':
        start_level = atoi(argv[argc] + 2);
        break;
      case 't':
        time_limit = meaning;
        break;
#endif
      default: 
        errorSet("MAIN", "unsupported switch");
        break;
      }
      break;
    default: 
      errorSet("MAIN", "unsupported argument");
      break;
    }
  }
  return 0;
}

void mainPrintUsage(){
  printf( "abombniball [options]\n" );
  printf( "Standard Options\n");
  printf( "\t+s/-s   Enable/Disable sound\n" );
  printf( "\t+f/-f   Enable/Disable fullscreen\n" );
  printf( "\t-h      Print this help text\n" );
#ifdef CHEATING_SCUMBAG
  printf( "Cheat Options\n" );
  printf( "\t+t/-t   Enable/Disable bomb timers\n" );
  printf( "\t-ln     Start on level n\n" );
  printf( "\n" );
#endif
}

  
int mainLoadGfx(Gfx *gfx){
  SDL_Surface *image;
  Tile *tile;
  int i;
  char filename[255];

  image = gfxLoadImage( fileCreateName( GFX, "bomb.bmp" ) );
  if( !image ) return 0;
  spriteNewType( gfx, "bomb", 24, 24, 20, image, 0, 255, 0 );
  SDL_FreeSurface( image );

  image = gfxLoadImage( fileCreateName( GFX, "ball.bmp" ) );
  if( !image ) return 0;
  spriteNewType( gfx, "ball", 24, 24, 8, image, 0, 255, 0 );
  SDL_FreeSurface( image );

  for( i = 0; i < NUM_TILES; i++ ){
    tile = &_tile_types[i];
    if( tile->visible ){
      strcpy( filename, tile->name );
      strcat( filename, ".bmp" );
      tile->image = gfxLoadImage( fileCreateName( GFX, filename ) );
      if( !tile->image ) return 0;
    }
  } 
  
  _game_background = gfxLoadImage( fileCreateName( GFX, "background.bmp" ));
  return 1;

}

void mainUnloadGfx(){
  int i;
  
  for( i = 0; i < NUM_TILES; i++ )
    SDL_FreeSurface( _tile_types[i].image );
    
  SDL_FreeSurface( _game_background );
}

int mainLoadSfx(){
    bounce_fx = Mix_LoadWAV( fileCreateName( AUDIO, "bounce.wav" ) );
    if( !bounce_fx ) return 0;

    return 1;
}

void mainUnloadSfx(){
    if( bounce_fx ){
        Mix_FreeChunk( bounce_fx );
        bounce_fx = NULL;
    }
}

int mainTitle(Gfx *gfx){
  Game *game;
  Mix_Music *title_tune = NULL;
  SDL_Surface *title = NULL;
  Sprite *pointer = NULL;
  int state, quit = 0;
  int scroller_item = 0;
  Input input = { 0, 0, 0, 0, 0, "" };
  state = INIT;
  
  while( !quit ){
    switch( state ){
      case INIT:
        title = gfxLoadImage( fileCreateName( GFX, "title.bmp" ) );
        if( !title ) return 0;
        pointer = gfxAddSprite( gfx, "ball" );
      case REDRAW:
        gfx->background = title;
        gfxDirtyRectangle( gfx, NULL );
        state = IDLE;
        mainSetMenu( gfx, main_menu );
        pointer_pos = 0;
        pointer->hidden = 0;
        if( title_tune )
          Mix_FreeMusic( title_tune );
        title_tune = Mix_LoadMUS( fileCreateName( AUDIO, "title.xm" ) );
        if( !title_tune ){
          errorSet( "AUDIO", "Could not load title.xm" );
          return 0;
        }
        if(play_music) 
	        Mix_PlayMusic(title_tune, -1);
        
        scroller_item = 0;
        break;

      case GAME:
        mainSetMenu( gfx, NULL );
        Mix_FadeOutMusic( 1000 );
        while( Mix_FadingMusic() == MIX_FADING_OUT )
          SDL_Delay( 100 );
        Mix_FreeMusic( title_tune );
        title_tune = NULL;
        pointer->hidden = 1;
        game = gameNew( gfx );
        if( !game ){
          errorSet( "MAIN", "Could not create game" );
          return 0;
        }
        if( gameLoop( game ) ){
          errorSet( "MAIN", "Error during game" );
          return 0;
        }
        gameDelete( game );
        state = REDRAW;
        break;

      case IDLE:
        input.fire = 0;
        input.direction = 0;
        inputHandle( &input );
        if( input.direction == INPUT_UP && pointer_pos > 0 ){
          pointer_pos--;
        }
        if( input.direction == INPUT_DOWN && pointer_pos < menu_length - 1 ){
          pointer_pos++;
        }
        
        if( input.fire ){
          state = current_menu[pointer_pos].action;
          break;
        }
        
        pointer->x = 220;
        pointer->y = 205 + ( pointer_pos * 40 );
        
        if( !gfxScrollerStatus( gfx ) ){
            gfxScrollerSet( gfx, scroller_text[ scroller_item++ ], 1 );
            if( scroller_item == NUM_SCROLLER_ITEMS )
                scroller_item = 0;
        }
        gfxUpdate( gfx , 0 );
        break;

      case QUIT:
        quit = 1;
        SDL_FreeSurface( title );
        gfxRemoveSprite( gfx, pointer );
        Mix_FadeOutMusic( 1000 );
        while( Mix_FadingMusic() == MIX_FADING_OUT )
          SDL_Delay( 100 );
        Mix_HaltMusic();
        Mix_FreeMusic( title_tune );
        break;

      default:
        break;
    }
  }
  return 1;
}

void mainSetMenu(Gfx *gfx, MenuItem *items){
  int i;
  MenuItem *item;
  
  for( i = 0; i < MAX_MENU; i++ ){
    if( menu_text[i] ){
      stringDelete( menu_text[i] );
      menu_text[i] = NULL;
    }
  }
  
  current_menu = items;
  item = items;
  menu_length = 0;
  if( items == NULL ) return;

  i = 0;
  
  while( item->action != -1 ){
    menu_text[i] = stringNew( gfx, strlen( item->name ) );
    stringSet( menu_text[i], item->name, 1 );
    stringMove( menu_text[i], 250, 200 + ( i * 40 ) );
    item++;
    i++;
  }
  menu_length = i;
}

